/*
 * Copyright 2010 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package com.vaadin.pro.licensechecker.json.impl;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import com.vaadin.pro.licensechecker.json.LicenseCheckerJson;
import com.vaadin.pro.licensechecker.json.LicenseCheckerJsonArray;
import com.vaadin.pro.licensechecker.json.LicenseCheckerJsonBoolean;
import com.vaadin.pro.licensechecker.json.LicenseCheckerJsonFactory;
import com.vaadin.pro.licensechecker.json.LicenseCheckerJsonNumber;
import com.vaadin.pro.licensechecker.json.LicenseCheckerJsonObject;
import com.vaadin.pro.licensechecker.json.LicenseCheckerJsonString;
import com.vaadin.pro.licensechecker.json.LicenseCheckerJsonType;
import com.vaadin.pro.licensechecker.json.LicenseCheckerJsonValue;

/**
 * Server-side implementation of LicenseCheckerJsonObject.
 */
public class LicenseCheckerJreJsonObject extends LicenseCheckerJreJsonValue
        implements LicenseCheckerJsonObject {

    private static final long serialVersionUID = 1L;

    private static List<String> stringifyOrder(String[] keys) {
        List<String> toReturn = new ArrayList<String>();
        List<String> nonNumeric = new ArrayList<String>();
        for (String key : keys) {
            if (key.matches("\\d+")) {
                toReturn.add(key);
            } else {
                nonNumeric.add(key);
            }
        }
        Collections.sort(toReturn);
        toReturn.addAll(nonNumeric);
        return toReturn;
    }

    private transient LicenseCheckerJsonFactory factory;
    private transient Map<String, LicenseCheckerJsonValue> map = new LinkedHashMap<String, LicenseCheckerJsonValue>();

    public LicenseCheckerJreJsonObject(LicenseCheckerJsonFactory factory) {
        this.factory = factory;
    }

    @Override
    public boolean asBoolean() {
        return true;
    }

    @Override
    public double asNumber() {
        return Double.NaN;
    }

    @Override
    public String asString() {
        return "[object Object]";
    }

    public LicenseCheckerJsonValue get(String key) {
        return map.get(key);
    }

    public LicenseCheckerJsonArray getArray(String key) {
        return (LicenseCheckerJsonArray) get(key);
    }

    public boolean getBoolean(String key) {
        return ((LicenseCheckerJsonBoolean) get(key)).getBoolean();
    }

    public double getNumber(String key) {
        return ((LicenseCheckerJsonNumber) get(key)).getNumber();
    }

    public LicenseCheckerJsonObject getObject(String key) {
        return (LicenseCheckerJsonObject) get(key);
    }

    public Object getObject() {
        Map<String, Object> obj = new HashMap<String, Object>();
        for (Map.Entry<String, LicenseCheckerJsonValue> e : map.entrySet()) {
            obj.put(e.getKey(),
                    ((LicenseCheckerJreJsonValue) e.getValue()).getObject());
        }
        return obj;
    }

    public String getString(String key) {
        return ((LicenseCheckerJsonString) get(key)).getString();
    }

    public LicenseCheckerJsonType getType() {
        return LicenseCheckerJsonType.OBJECT;
    }

    @Override
    public boolean hasKey(String key) {
        return map.containsKey(key);
    }

    @Override
    public boolean jsEquals(LicenseCheckerJsonValue value) {
        return getObject()
                .equals(((LicenseCheckerJreJsonValue) value).getObject());
    }

    public String[] keys() {
        return map.keySet().toArray(new String[map.size()]);
    }

    public void put(String key, LicenseCheckerJsonValue value) {
        if (value == null) {
            value = factory.createNull();
        }
        map.put(key, value);
    }

    public void put(String key, String value) {
        put(key, factory.create(value));
    }

    public void put(String key, double value) {
        put(key, factory.create(value));
    }

    public void put(String key, boolean bool) {
        put(key, factory.create(bool));
    }

    @Override
    public void remove(String key) {
        map.remove(key);
    }

    public void set(String key, LicenseCheckerJsonValue value) {
        put(key, value);
    }

    public String toJson() {
        return LicenseCheckerJsonUtil.stringify(this);
    }

    public String toString() {
        return toJson();
    }

    @Override
    public void traverse(LicenseCheckerJsonVisitor visitor,
            LicenseCheckerJsonContext ctx) {
        if (visitor.visit(this, ctx)) {
            LicenseCheckerJsonObjectContext objCtx = new LicenseCheckerJsonObjectContext(
                    this);
            for (String key : stringifyOrder(keys())) {
                objCtx.setCurrentKey(key);
                if (visitor.visitKey(objCtx.getCurrentKey(), objCtx)) {
                    visitor.accept(get(key), objCtx);
                    objCtx.setFirst(false);
                }
            }
        }
        visitor.endVisit(this, ctx);
    }

    private void readObject(ObjectInputStream stream)
            throws IOException, ClassNotFoundException {
        LicenseCheckerJreJsonObject instance = parseJson(stream);
        this.factory = LicenseCheckerJson.instance();
        this.map = instance.map;
    }

    private void writeObject(ObjectOutputStream stream) throws IOException {
        stream.writeObject(toJson());
    }
}
