/*
 * Copyright 2000-2025 Vaadin Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package com.vaadin.pro.licensechecker;

import java.nio.file.Path;
import java.nio.file.attribute.FileTime;

/**
 * Provides filesystem operations for Machine ID generation. This abstraction
 * allows for testing with mock implementations.
 */
interface MachineIdFileAccess {

    /**
     * Gets the user's home directory path.
     *
     * @return the user's home directory path
     */
    Path getUserHome();

    /**
     * Gets file attributes for the given path.
     *
     * @param path
     *            the path to read attributes from
     * @return the file attributes
     * @throws Exception
     *             if an error occurs reading the attributes
     */
    FileAttributes getFileAttributes(Path path) throws Exception;

    /**
     * Container for file attributes needed by Machine ID generation.
     */
    class FileAttributes {
        private final FileTime creationTime;
        private final FileTime lastModifiedTime;

        /**
         * Creates a new file attributes container.
         *
         * @param creationTime
         *            the file creation time
         * @param lastModifiedTime
         *            the file last modified time
         */
        public FileAttributes(FileTime creationTime,
                FileTime lastModifiedTime) {
            this.creationTime = creationTime;
            this.lastModifiedTime = lastModifiedTime;
        }

        /**
         * Gets the file creation time.
         *
         * @return the creation time
         */
        public FileTime getCreationTime() {
            return creationTime;
        }

        /**
         * Gets the file last modified time.
         *
         * @return the last modified time
         */
        public FileTime getLastModifiedTime() {
            return lastModifiedTime;
        }
    }
}
