/*
 * Copyright 2000-2024 Vaadin Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package com.vaadin.pro.licensechecker;

import java.time.Duration;
import java.time.LocalDateTime;
import java.time.Period;
import java.time.ZoneOffset;
import java.util.Optional;
import java.util.function.BooleanSupplier;
import java.util.function.Consumer;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.vaadin.open.Open;
import com.vaadin.pro.licensechecker.OnlineKeyValidator.Result;

public class LicenseChecker {

    static final int DEFAULT_KEY_URL_HANDLER_TIMEOUT_SECONDS = 60;

    private static boolean strictOffline = false;

    static String loggedLicenseOwner = null;

    public interface Callback {

        public void ok();

        public void failed(Exception e);
    }

    /**
     * Options for downloading a license key.
     */
    public static class DownloadOptions {
        private final Product source;
        private final int timeoutSeconds;
        private final Consumer<String> urlHandler;

        /**
         * Creates download options with the specified source product.
         *
         * @param source
         *            the product that triggered the download (who is requesting
         *            the license)
         * @throws IllegalArgumentException
         *             if source is null
         */
        public DownloadOptions(Product source) {
            this(source, DEFAULT_KEY_URL_HANDLER_TIMEOUT_SECONDS,
                    systemBrowserUrlHandler);
        }

        /**
         * Creates download options with a custom timeout.
         *
         * @param source
         *            the product that triggered the download (who is requesting
         *            the license)
         * @param timeoutSeconds
         *            timeout in seconds to wait for user to complete
         *            authentication
         * @throws IllegalArgumentException
         *             if source is null or timeoutSeconds is less than or equal
         *             to 0
         */
        public DownloadOptions(Product source, int timeoutSeconds) {
            this(source, timeoutSeconds, systemBrowserUrlHandler);
        }

        /**
         * Creates download options with a custom URL handler.
         *
         * @param source
         *            the product that triggered the download (who is requesting
         *            the license)
         * @param timeoutSeconds
         *            timeout in seconds to wait for user to complete
         *            authentication
         * @param urlHandler
         *            custom handler for opening the authentication URL (for
         *            testing or custom browser behavior), or null to use the
         *            default system browser handler
         * @throws IllegalArgumentException
         *             if source is null or timeoutSeconds is less than or equal
         *             to 0
         */
        public DownloadOptions(Product source, int timeoutSeconds,
                Consumer<String> urlHandler) {
            if (source == null) {
                throw new IllegalArgumentException("source cannot be null");
            }
            if (timeoutSeconds <= 0) {
                throw new IllegalArgumentException(
                        "timeoutSeconds must be greater than 0");
            }
            this.source = source;
            this.timeoutSeconds = timeoutSeconds;
            this.urlHandler = urlHandler != null ? urlHandler
                    : systemBrowserUrlHandler;
        }

        /**
         * Gets the product that triggered the download.
         *
         * @return the source product
         */
        public Product getSource() {
            return source;
        }

        /**
         * Gets the timeout in seconds.
         *
         * @return the timeout in seconds
         */
        public int getTimeoutSeconds() {
            return timeoutSeconds;
        }

        /**
         * Gets the URL handler.
         *
         * @return the URL handler
         */
        public Consumer<String> getUrlHandler() {
            return urlHandler;
        }
    }

    static Consumer<String> systemBrowserUrlHandler = url -> {
        try {
            getLogger().info(
                    "Opening system browser to validate license. If the browser is not opened, please open "
                            + url + " manually");
            Open.open(url);
            getLogger().info(
                    "For CI/CD build servers, you need to download a server license key, which can only be "
                            + "used for production builds. You can download a server license key from "
                            + "https://vaadin.com/myaccount/licenses.\n"
                            + "If you are working offline in development mode, please visit "
                            + OfflineKeyValidator.getOfflineUrl(new MachineId())
                            + " for an offline development mode license.");
        } catch (Exception e) {
            getLogger().error(
                    "Error opening system browser to validate license. Please open "
                            + url + " manually",
                    e);
        }

    };

    /**
     * @deprecated use
     *             {@link #checkLicenseFromStaticBlock(String, String, BuildType)}
     */
    @Deprecated
    // Note for maintainers: to be not removed as older released
    // Vaadin (e.g. 7/8) versions may use this method, and we don't want to
    // break the apps that use them
    public static void checkLicenseFromStaticBlock(String productName,
            String productVersion) {
        checkLicenseFromStaticBlock(productName, productVersion,
                BuildType.DEVELOPMENT);
    }

    /**
     * Checks the license for the given product version from a {@code static}
     * block.
     *
     * @param productName
     *            the name of the product to check
     * @param productVersion
     *            the version of the product to check
     * @param buildType
     *            the type of build: production or development
     *
     * @throws ExceptionInInitializerError
     *             if the license check fails
     */
    public static void checkLicenseFromStaticBlock(String productName,
            String productVersion, BuildType buildType) {
        checkLicenseFromStaticBlock(productName, productVersion, buildType,
                Capabilities.none());
    }

    /**
     * Checks the license for the given product version from a {@code static}
     * block with the given capabilities (extra license checks).
     *
     * @param productName
     *            the name of the product to check
     * @param productVersion
     *            the version of the product to check
     * @param buildType
     *            the type of build: production or development
     * @param capabilities
     *            the license checker capabilities to run.
     *
     * @throws ExceptionInInitializerError
     *             if the license check fails
     */
    public static void checkLicenseFromStaticBlock(String productName,
            String productVersion, BuildType buildType,
            Capabilities capabilities) {
        try {
            checkLicense(productName, productVersion, capabilities, buildType);
        } catch (Exception e) {
            throw new ExceptionInInitializerError(e);
        }
    }

    /**
     * @deprecated use {@link #checkLicense(String, String, BuildType)}
     */
    @Deprecated
    // Note for maintainers: to be not removed as older released
    // Vaadin (e.g. 7/8) versions may use this method, and we don't want to
    // break the apps that use them
    public static void checkLicense(String productName, String productVersion) {
        checkLicense(productName, productVersion, BuildType.DEVELOPMENT,
                systemBrowserUrlHandler);
    }

    /**
     * Checks the license for the given product version.
     *
     * @param productName
     *            the name of the product to check
     * @param productVersion
     *            the version of the product to check
     * @param buildType
     *            the type of build: development or production
     *
     * @throws LicenseException
     *             if the license check fails
     */
    public static void checkLicense(String productName, String productVersion,
            BuildType buildType) {
        checkLicense(productName, productVersion, Capabilities.none(),
                buildType);
    }

    /**
     * Checks the license for the given product version with the given
     * capabilities (extra license checks).
     *
     * @param productName
     *            the name of the product to check
     * @param productVersion
     *            the version of the product to check
     * @param capabilities
     *            the license checker capabilities to run
     * @param buildType
     *            the type of build: development or production
     *
     * @throws LicenseException
     *             if the license check fails
     */
    public static void checkLicense(String productName, String productVersion,
            Capabilities capabilities, BuildType buildType) {
        checkLicense(productName, productVersion, buildType,
                systemBrowserUrlHandler, capabilities);
    }

    /**
     * Checks the license for the given pro key and product version.
     *
     * @param productName
     *            the name of the product to check
     * @param productVersion
     *            the version of the product to check
     * @param buildType
     *            the type of build: development or production
     * @param noKeyUrlHandler
     *            a handler that is invoked to open the vaadin.com URL to
     *            download the key file. Used when no key file is available.
     * @param machineId
     *            the identifier of machine which owns pro key
     * @param proKey
     *            the pro key to be validated
     * @param offlineKey
     *            the offline key to be validated
     *
     * @throws LicenseException
     *             if the license check fails
     */
    public static void checkLicense(String productName, String productVersion,
            BuildType buildType, Consumer<String> noKeyUrlHandler,
            String machineId, ProKey proKey, SubscriptionKey subscriptionKey,
            OfflineKey offlineKey) {
        checkLicense(productName, productVersion, buildType, noKeyUrlHandler,
                machineId, proKey, subscriptionKey, offlineKey,
                Capabilities.none());
    }

    /**
     * Checks the license for the given pro key and product version with the
     * given capabilities (extra license checks).
     *
     * @param productName
     *            the name of the product to check
     * @param productVersion
     *            the version of the product to check
     * @param buildType
     *            the type of build: development or production
     * @param noKeyUrlHandler
     *            a handler that is invoked to open the vaadin.com URL to
     *            download the key file. Used when no key file is available.
     * @param machineId
     *            the identifier of machine which owns pro key
     * @param proKey
     *            the pro key to be validated
     * @param offlineKey
     *            the offline key to be validated
     * @param capabilities
     *            the license checker capabilities to run.
     *
     * @throws LicenseException
     *             if the license check fails
     */
    public static void checkLicense(String productName, String productVersion,
            BuildType buildType, Consumer<String> noKeyUrlHandler,
            String machineId, ProKey proKey, SubscriptionKey subscriptionKey,
            OfflineKey offlineKey, Capabilities capabilities) {
        checkLicense(new Product(productName, productVersion), buildType,
                noKeyUrlHandler, new MachineId(machineId), proKey,
                subscriptionKey, offlineKey, new OnlineKeyValidator(),
                new OfflineKeyValidator(),
                capabilities.getPreTrialValidator().orElse(null),
                new VaadinComIntegration(),
                DEFAULT_KEY_URL_HANDLER_TIMEOUT_SECONDS, capabilities);
    }

    /**
     * @deprecated use
     *             {@link #checkLicense(String, String, BuildType, Consumer)}
     */
    @Deprecated
    // Note for maintainers: to be not removed as older released
    // Vaadin (e.g. 7/8) versions may use this method, and we don't want to
    // break the apps that use them
    public static void checkLicense(String productName, String productVersion,
            Consumer<String> noKeyUrlHandler) {
        checkLicense(productName, productVersion, BuildType.DEVELOPMENT,
                noKeyUrlHandler);
    }

    /**
     * Checks the license for the given product version.
     *
     * @param productName
     *            the name of the product to check
     * @param productVersion
     *            the version of the product to check
     * @param buildType
     *            the type of build: development or production
     * @param noKeyUrlHandler
     *            a handler that is invoked to open the vaadin.com URL to
     *            download the key file. Used when no key file is available.
     *
     * @throws LicenseException
     *             if the license check fails
     */
    public static void checkLicense(String productName, String productVersion,
            BuildType buildType, Consumer<String> noKeyUrlHandler) {
        checkLicense(productName, productVersion, buildType, noKeyUrlHandler,
                Capabilities.none());
    }

    /**
     * Checks the license for the given product version with the given
     * capabilities (extra license checks).
     *
     * @param productName
     *            the name of the product to check
     * @param productVersion
     *            the version of the product to check
     * @param buildType
     *            the type of build: development or production
     * @param noKeyUrlHandler
     *            a handler that is invoked to open the vaadin.com URL to
     *            download the key file. Used when no key file is available.
     * @param capabilities
     *            the license checker capabilities to run.
     *
     * @throws LicenseException
     *             if the license check fails
     */
    public static void checkLicense(String productName, String productVersion,
            BuildType buildType, Consumer<String> noKeyUrlHandler,
            Capabilities capabilities) {
        checkLicense(productName, productVersion, buildType, noKeyUrlHandler,
                DEFAULT_KEY_URL_HANDLER_TIMEOUT_SECONDS, capabilities);
    }

    /**
     * Checks the license for the given product version.
     *
     * @param productName
     *            the name of the product to check
     * @param productVersion
     *            the version of the product to check
     * @param buildType
     *            the type of build: development or production
     * @param noKeyUrlHandler
     *            a handler that is invoked to open the vaadin.com URL to
     *            download the key file. Used when no key file is available.
     * @param timeoutKeyUrlHandler
     *            timeout for the key url handler
     * @throws LicenseException
     *             if the license check fails
     */
    public static void checkLicense(String productName, String productVersion,
            BuildType buildType, Consumer<String> noKeyUrlHandler,
            int timeoutKeyUrlHandler) {
        checkLicense(productName, productVersion, buildType, noKeyUrlHandler,
                timeoutKeyUrlHandler, Capabilities.none());
    }

    /**
     * Checks the license for the given product version with the given
     * capabilities (extra license checks).
     *
     * @param productName
     *            the name of the product to check
     * @param productVersion
     *            the version of the product to check
     * @param buildType
     *            the type of build: development or production
     * @param noKeyUrlHandler
     *            a handler that is invoked to open the vaadin.com URL to
     *            download the key file. Used when no key file is available.
     * @param timeoutKeyUrlHandler
     *            timeout for the key url handler
     * @param capabilities
     *            the license checker capabilities to run.
     * @throws LicenseException
     *             if the license check fails
     */
    public static void checkLicense(String productName, String productVersion,
            BuildType buildType, Consumer<String> noKeyUrlHandler,
            int timeoutKeyUrlHandler, Capabilities capabilities) {
        checkLicense(new Product(productName, productVersion), buildType,
                noKeyUrlHandler, timeoutKeyUrlHandler, capabilities);
    }

    /**
     * Checks the license for the given product version. Returns {@code true} if
     * the license is valid, {@code false} otherwise.
     *
     * @param productName
     *            the name of the product to check
     * @param productVersion
     *            the version of the product to check
     * @param buildType
     *            the type of build: development or production
     * @return {@code true} if the license is valid, {@code false} otherwise
     */
    public static boolean isValidLicense(String productName,
            String productVersion, BuildType buildType) {
        try {
            checkLicense(productName, productVersion, buildType,
                    (Consumer<String>) null);
            return true;
        } catch (LicenseException e) {
            return false;
        }
    }

    /**
     * @deprecated use
     *             {@link #checkLicenseAsync(String, String, BuildType, Callback)}
     */
    @Deprecated
    // Note for maintainers: to be not removed as older released
    // Vaadin (e.g. 7/8) versions may use this method, and we don't want to
    // break the apps that use them
    public static void checkLicenseAsync(String productName,
            String productVersion, Callback callback) {
        checkLicenseAsync(productName, productVersion, BuildType.DEVELOPMENT,
                callback);
    }

    /**
     * Checks the license for the given product version in the background and
     * invokes the callback when done.
     *
     * @param productName
     *            the name of the product to check
     * @param productVersion
     *            the version of the product to check
     * @param buildType
     *            the type of build: development or production
     * @param callback
     *            the callback to invoke with the result
     */
    public static void checkLicenseAsync(String productName,
            String productVersion, BuildType buildType, Callback callback) {
        checkLicenseAsync(productName, productVersion, buildType, callback,
                Capabilities.none());
    }

    /**
     * Checks the license for the given product version in the background and
     * invokes the callback when done with the given capabilities (extra license
     * checks).
     *
     * @param productName
     *            the name of the product to check
     * @param productVersion
     *            the version of the product to check
     * @param buildType
     *            the type of build: development or production
     * @param callback
     *            the callback to invoke with the result
     * @param capabilities
     *            the license checker capabilities to run.
     */
    public static void checkLicenseAsync(String productName,
            String productVersion, BuildType buildType, Callback callback,
            Capabilities capabilities) {
        checkLicenseAsync(productName, productVersion, buildType, callback,
                systemBrowserUrlHandler, capabilities);
    }

    /**
     * Checks the license for the given pro key and product version in the
     * background and invokes the callback when done.
     *
     * @param productName
     *            the name of the product to check
     * @param productVersion
     *            the version of the product to check
     * @param buildType
     *            the type of build: development or production
     * @param callback
     *            the callback to invoke with the result
     * @param noKeyUrlHandler
     *            a handler that is invoked to open the vaadin.com URL to
     *            download the key file. Used when no key file is available.
     * @param machineId
     *            the identifier of machine which owns pro key
     * @param proKey
     *            the pro key to be validated
     * @param offlineKey
     *            the offline key to be validated
     *
     * @throws LicenseException
     *             if the license check fails
     */
    public static void checkLicenseAsync(String productName,
            String productVersion, BuildType buildType, Callback callback,
            Consumer<String> noKeyUrlHandler, String machineId, ProKey proKey,
            SubscriptionKey subscriptionKey, OfflineKey offlineKey) {
        checkLicenseAsync(new Product(productName, productVersion), buildType,
                callback, noKeyUrlHandler, new MachineId(machineId), proKey,
                subscriptionKey, offlineKey, new OnlineKeyValidator(),
                new OfflineKeyValidator(), null, new VaadinComIntegration(),
                DEFAULT_KEY_URL_HANDLER_TIMEOUT_SECONDS);
    }

    /**
     * Checks the license for the given product version in the background and
     * invokes the callback when done.
     *
     * @param productName
     *            the name of the product to check
     * @param productVersion
     *            the version of the product to check
     * @param buildType
     *            the type of build: development or production
     * @param callback
     *            the callback to invoke with the result
     * @param noKeyUrlHandler
     *            a handler that is invoked to open the vaadin.com URL to
     *            download the key file. Used when no key file is available.
     */
    public static void checkLicenseAsync(String productName,
            String productVersion, BuildType buildType, Callback callback,
            Consumer<String> noKeyUrlHandler) {
        checkLicenseAsync(productName, productVersion, buildType, callback,
                noKeyUrlHandler, Capabilities.none());
    }

    /**
     * Checks the license for the given product version in the background and
     * invokes the callback when done with the given capabilities (extra license
     * checks).
     *
     * @param productName
     *            the name of the product to check
     * @param productVersion
     *            the version of the product to check
     * @param buildType
     *            the type of build: development or production
     * @param callback
     *            the callback to invoke with the result
     * @param noKeyUrlHandler
     *            a handler that is invoked to open the vaadin.com URL to
     *            download the key file. Used when no key file is available.
     * @param capabilities
     *            the license checker capabilities to run.
     */
    public static void checkLicenseAsync(String productName,
            String productVersion, BuildType buildType, Callback callback,
            Consumer<String> noKeyUrlHandler, Capabilities capabilities) {
        checkLicenseAsync(productName, productVersion, buildType, callback,
                noKeyUrlHandler, DEFAULT_KEY_URL_HANDLER_TIMEOUT_SECONDS,
                capabilities);
    }

    /**
     * Checks the license for the given product version in the background and
     * invokes the callback when done with the given capabilities (extra license
     * checks).
     *
     * @param productName
     *            the name of the product to check
     * @param productVersion
     *            the version of the product to check
     * @param buildType
     *            the type of build: development or production
     * @param callback
     *            the callback to invoke with the result
     * @param noKeyUrlHandler
     *            a handler that is invoked to open the vaadin.com URL to
     *            download the key file. Used when no key file is available.
     * @param timeoutKeyUrlHandler
     *            timeout for the key url handler
     * @param capabilities
     *            the license checker capabilities to run.
     */
    public static void checkLicenseAsync(String productName,
            String productVersion, BuildType buildType, Callback callback,
            Consumer<String> noKeyUrlHandler, int timeoutKeyUrlHandler,
            Capabilities capabilities) {
        checkLicenseAsync(new Product(productName, productVersion), buildType,
                callback, noKeyUrlHandler, new MachineId(), LocalProKey.get(),
                LocalSubscriptionKey.get(), LocalOfflineKey.get(),
                new OnlineKeyValidator(), new OfflineKeyValidator(),
                capabilities.getPreTrialValidator().orElse(null),
                new VaadinComIntegration(), timeoutKeyUrlHandler);
    }

    /**
     * Checks the license for the given product version in the background and
     * invokes the callback when done.
     *
     * @param productName
     *            the name of the product to check
     * @param productVersion
     *            the version of the product to check
     * @param buildType
     *            the type of build: development or production
     * @param callback
     *            the callback to invoke with the result
     * @param noKeyUrlHandler
     *            a handler that is invoked to open the vaadin.com URL to
     *            download the key file. Used when no key file is avialable.
     * @param timeoutKeyUrlHandler
     *            timeout for the key url handler
     */
    public static void checkLicenseAsync(String productName,
            String productVersion, BuildType buildType, Callback callback,
            Consumer<String> noKeyUrlHandler, int timeoutKeyUrlHandler) {
        checkLicenseAsync(productName, productVersion, buildType, callback,
                noKeyUrlHandler, timeoutKeyUrlHandler, Capabilities.none());
    }

    // For testing
    static void checkLicenseAsync(Product product, BuildType buildType,
            Callback callback, Consumer<String> noKeyUrlHandler,
            MachineId machineId, ProKey proKey, SubscriptionKey subscriptionKey,
            OfflineKey offlineKey, OnlineKeyValidator proKeyValidator,
            OfflineKeyValidator offlineKeyValidator,
            PreTrialValidator preTrialValidator,
            VaadinComIntegration vaadinComIntegration,
            int timeoutKeyUrlHandler) {
        new Thread(() -> {
            try {
                checkLicense(product, buildType, noKeyUrlHandler, machineId,
                        proKey, subscriptionKey, offlineKey, proKeyValidator,
                        offlineKeyValidator, preTrialValidator,
                        vaadinComIntegration, timeoutKeyUrlHandler,
                        Capabilities.none());
                callback.ok();
            } catch (Exception e) {
                callback.failed(e);
            }
        }).start();
    }

    /**
     * The internal method all license check methods end up calling.
     * <p>
     * Checking works like:
     * <ol>
     * <li>If there is a local pro key, check with the license server that the
     * proKey has access to the given product
     * <li>If there is no local pro key but there is an offline key, check that
     * this offline key is valid and allows access to the given product
     * <li>If there was neither a local proKey nor an offline key, open the
     * vaadin.com URL for fetching a pro key for the user. Wait for the user to
     * log in, store the pro key and then validate it for the select product
     * like in step 1.
     * </ol>
     *
     * @param product
     *            the name and version of the product to check
     * @param buildType
     *            the type of build: development or production
     * @param noKeyUrlHandler
     *            a handler that is invoked to open the vaadin.com URL to
     *            download the key file. Used when no key file is avialable.
     * @param timeoutKeyUrlHandler
     *            timeout for the key url handler
     */
    static void checkLicense(Product product, BuildType buildType,
            Consumer<String> noKeyUrlHandler, int timeoutKeyUrlHandler,
            Capabilities capabilities) {
        getLogger().debug("Checking license for " + product);
        checkLicense(product, buildType, noKeyUrlHandler, new MachineId(),
                LocalProKey.get(), LocalSubscriptionKey.get(),
                LocalOfflineKey.get(), new OnlineKeyValidator(),
                new OfflineKeyValidator(),
                capabilities.getPreTrialValidator().orElse(null),
                new VaadinComIntegration(), timeoutKeyUrlHandler, capabilities);
    }

    /**
     * Starts a pre-trial period by using the default machine ID and a
     * predefined user key.
     * <p>
     * If a pre-trial is already started it returns the state with updated
     * remaining days. Otherwise, if the pre-trial has expired but cannot yet be
     * renewed a {@link PreTrialCreationException.Expired} is thrown.
     *
     * @return detail about the pre-trial.
     * @throws PreTrialCreationException.Expired
     *             if the pre-trial is expired and cannot yet be renewed
     * @throws PreTrialCreationException
     *             if the License Server prevents the start of a pre-trial
     */
    public static PreTrial startPreTrial() throws PreTrialCreationException {
        return startPreTrial(new MachineId(), UserKey.get(),
                new PreTrialValidator());
    }

    /**
     * Starts a pre-trial period by using given machine ID and given user key.
     * <p>
     * If a pre-trial is already started it returns the state with updated
     * remaining days. Otherwise, if the pre-trial has expired but cannot yet be
     * renewed a {@link PreTrialCreationException.Expired} is thrown.
     *
     * @param machineId
     *            the identifier of machine which owns pro key
     * @param userKey
     *            the user key
     * @return detail about the pre-trial.
     * @throws PreTrialCreationException.Expired
     *             if the pre-trial is expired and cannot yet be renewed
     * @throws PreTrialCreationException
     *             if the License Server prevents the start of a pre-trial
     */
    public static PreTrial startPreTrial(String machineId, String userKey)
            throws PreTrialCreationException {
        return startPreTrial(new MachineId(machineId), new UserKey(userKey),
                new PreTrialValidator());
    }

    /**
     * Gets the status of a pre-trial period for the given product using the
     * default machine ID.
     * <p>
     * It returns the state of the pre-trial with information about remaining
     * days.
     *
     * @param product
     *            the product to validate
     * @return detail about the pre-trial.
     * @throws PreTrialStatusCheckException
     *             if pre-trial status cannot be checked against the License
     *             Server.
     */
    public static Optional<PreTrial> getPreTrial(Product product) {
        return new PreTrialValidator().statusCheck(product, new MachineId());
    }

    /**
     * Gets the status of a pre-trial period for the given product and
     * machineId.
     * <p>
     * It returns the state of the pre-trial with information about remaining
     * days.
     *
     * @param product
     *            the product to validate
     * @param machineId
     *            the identifier of machine which owns pro key
     * @return detail about the pre-trial.
     * @throws PreTrialStatusCheckException
     *             if pre-trial status cannot be checked against the License
     *             Server.
     */
    public static Optional<PreTrial> getPreTrial(Product product,
            String machineId) {
        return new PreTrialValidator().statusCheck(product,
                new MachineId(machineId));
    }

    /**
     * Downloads a Vaadin Pro license key by opening the system browser for
     * authentication. This method blocks until the user completes
     * authentication in the browser or the timeout expires. The downloaded key
     * is automatically saved to ~/.vaadin/proKey.
     *
     * @param options
     *            the download options specifying the source product, timeout,
     *            and URL handler
     * @throws LicenseException
     *             if download fails, times out, or user cancels authentication
     */
    public static void downloadLicense(DownloadOptions options)
            throws LicenseException {
        Product product = options.getSource();
        int timeoutSeconds = options.getTimeoutSeconds();
        Consumer<String> urlHandler = options.getUrlHandler();

        getLogger().debug(
                "Starting license download for {} {} with timeout of {} seconds",
                product.getName(), product.getVersion(), timeoutSeconds);

        VaadinComIntegration vaadinComIntegration = new VaadinComIntegration();
        ProKey proKey;
        try {
            proKey = vaadinComIntegration.openBrowserAndWaitForKey(product,
                    urlHandler, timeoutSeconds);
        } catch (Exception e) {
            throw new LicenseException(
                    "License download failed: " + e.getMessage(), e);
        }

        if (proKey == null) {
            throw new LicenseException(
                    "License download failed. User did not complete authentication within "
                            + timeoutSeconds + " seconds.");
        }

        if (LocalProKey.write(proKey)) {
            getLogger().debug("License key downloaded and saved to {}",
                    LocalProKey.getLocation());
        } else {
            throw new LicenseException(
                    "Failed to save downloaded license key to "
                            + LocalProKey.getLocation());
        }
    }

    // Version for testing only
    static void checkLicense(Product product, BuildType buildType,
            Consumer<String> noKeyUrlHandler, MachineId machineId,
            ProKey proKey, SubscriptionKey subscriptionKey,
            OfflineKey offlineKey, OnlineKeyValidator proKeyValidator,
            OfflineKeyValidator offlineKeyValidator,
            PreTrialValidator preTrialValidator,
            VaadinComIntegration vaadinComIntegration, int timeoutKeyUrlHandler,
            Capabilities capabilities) {

        if (subscriptionKey == null && offlineKey != null
                && offlineKey.getSubscriptionKey() != null) {
            subscriptionKey = new SubscriptionKey(
                    offlineKey.getSubscriptionKey());
        }
        Result onlineCheckResult = proKeyValidator.validate(product,
                subscriptionKey, buildType, noKeyUrlHandler == null);
        if (onlineCheckResult == Result.OK) {
            return;
        }
        // fail fast for subscription key
        if (subscriptionKey != null) {
            if (onlineCheckResult == Result.CANNOT_REACH_SERVER) {
                if (offlineKey != null
                        && offlineKey.getSubscriptionKey() != null
                        && offlineKeyValidator.validate(null, null, offlineKey,
                                machineId)) {
                    // Valid offline key, assume subscription key is also valid
                    return;
                } else if (canWorkOffline(product, buildType, machineId,
                        new ProKey(null, subscriptionKey.getKey()))) {
                    return;
                }
            }
            throw new LicenseException(
                    "The provided license subscription key does not allow usage of "
                            + product.getName() + " " + product.getVersion()
                            + ". Check that the license has not expired.");
        }

        onlineCheckResult = proKeyValidator.validate(product, proKey, machineId,
                buildType, noKeyUrlHandler == null);

        if (onlineCheckResult == Result.OK) {
            // Online validation OK
            // TODO Show name and account
            return;
        } else if (onlineCheckResult == Result.NO_ACCESS && proKey != null) {
            // Online validation was done but the key does not give access to
            // the product
            throw new LicenseException(
                    "The provided license key does not allow usage of "
                            + product.getName() + " " + product.getVersion()
                            + ". Check that the license has not expired and that the product you are trying to use does not require another type of subscription.");
        }

        if (offlineKeyValidator.validate(product, buildType, offlineKey,
                machineId)) {
            // Offline validation OK
            if (loggedLicenseOwner == null && offlineKey.getName() != null) {
                // Show this message only once to avoid spamming once per
                // product

                LocalDateTime expires = LocalDateTime.ofEpochSecond(
                        offlineKey.getExpires() / 1000L, 0, ZoneOffset.UTC);
                long expiresInDays = Duration
                        .between(LocalDateTime.now(), expires).toDays();
                loggedLicenseOwner = "Using offline license registered to "
                        + offlineKey.getName() + " / " + offlineKey.getAccount()
                        + ". Expires in " + expiresInDays + " days.";
                getLogger().info(loggedLicenseOwner);
            }
            return;
        }

        if (onlineCheckResult == Result.CANNOT_REACH_SERVER
                // without machineId we cannot check for pre-trial state
                && (proKey != null || machineId == null)
                && canWorkOffline(product, buildType, machineId, proKey)) {
            return;
        }

        // Pre-trial does not allow production builds
        if (preTrialValidator != null && proKey == null && offlineKey == null
                && machineId != null && BuildType.PRODUCTION != buildType) {

            // No license keys available locally (subscriptionKey is always null
            // at this point), check for active pre-trial
            PreTrialValidator.Result preTrialResult = preTrialValidator
                    .validate(product, machineId, noKeyUrlHandler == null);
            PreTrialValidator.Access access = preTrialResult
                    .verifyAccess(noKeyUrlHandler != null);
            if (access == PreTrialValidator.Access.OK) {
                getLogger().debug("Pre-trial active for {}", product);
                return;
            } else if (access == PreTrialValidator.Access.CANNOT_REACH_SERVER
                    && canWorkOfflineWithPreTrial(product, machineId)) {
                return;
            }
        }

        if (proKey == null) {
            // No proKey found - probably first launch
            // For a background check (key handler is null) we should not do
            // anything but fail the check
            if (noKeyUrlHandler != null) {
                proKey = vaadinComIntegration.openBrowserAndWaitForKey(product,
                        noKeyUrlHandler, timeoutKeyUrlHandler);
                if (proKey != null) {
                    LocalProKey.write(proKey);
                    if (proKeyValidator.validate(product, proKey, machineId,
                            buildType, false) == Result.OK) {
                        // Online validation OK
                        return;
                    }
                }
            } else if (BuildType.PRODUCTION == buildType) {
                // Exception is specific for production builds, to allow the
                // caller to distinguish between license validation failure and
                // missing license keys
                throw new MissingLicenseKeyException(product,
                        "If you have an active subscription, please download the license key from https://vaadin.com/myaccount/licenses. "
                                + "Otherwise go to https://vaadin.com/pricing to obtain a license.");

            }
        }

        throw new LicenseException(
                "Unable to validate the license, please check your internet connection. If you need to work offline then "
                        + OfflineKeyValidator
                                .getOfflineKeyLinkMessage(machineId));

    }

    private static boolean canWorkOffline(Product product, BuildType buildType,
            MachineId machineId, ProKey proKey) {
        // Allow if offline but has validated during the last 2 days.
        return canWorkOffline(() -> History.isRecentlyValidated(product,
                Period.ofDays(2), buildType, proKey), machineId, strictOffline);
    }

    private static boolean canWorkOfflineWithPreTrial(Product product,
            MachineId machineId) {
        // Allow if offline but has validated recently and the pre-trial period
        // is not yet expired
        // Enforces strictOffline to prevent using the product if pre-trial was
        // not
        // previously validated.
        return canWorkOffline(
                () -> History.isPreTrialRecentlyValidated(product, machineId),
                machineId, true);
    }

    private static boolean canWorkOffline(BooleanSupplier recentlyValidatedTest,
            MachineId machineId, boolean enforceStrictOffline) {
        boolean recentlyValidated = recentlyValidatedTest.getAsBoolean();
        if (!recentlyValidated && enforceStrictOffline) {
            // Need an offline license
            throw new LicenseException(
                    OfflineKeyValidator.getMissingOfflineKeyMessage(machineId));
        }
        if (!recentlyValidated) {
            // Allow usage offline
            getLogger().warn(
                    "Unable to validate the license, please check your internet connection");
        }
        return true;
    }

    // Version for testing only
    static PreTrial startPreTrial(MachineId machineId, UserKey userKey,
            PreTrialValidator preTrialValidator) {
        return preTrialValidator.create(machineId,
                userKey != null ? userKey.getKey() : null);
    }

    static Logger getLogger() {
        return LoggerFactory.getLogger(LicenseChecker.class);
    }

    public static void setStrictOffline(boolean strictOffline) {
        LicenseChecker.strictOffline = strictOffline;
    }

}
