/*
 * Copyright 2000-2024 Vaadin Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package com.vaadin.pro.licensechecker.dau;

import java.util.UUID;

import com.vaadin.pro.licensechecker.LocalSubscriptionKey;
import com.vaadin.pro.licensechecker.SubscriptionKey;
import com.vaadin.pro.licensechecker.Util;

import static java.nio.charset.StandardCharsets.UTF_8;

/**
 * API to integrate an application with Daily Active User (DAU) tracking.
 * <p>
 * </p>
 * A Daily Active User represents a human end-user using applications a customer
 * built with Vaadin during a 24h period. This component provides an API to
 * track and synchronize daily active user data with Vaadin License Server.
 * <p>
 * </p>
 * DAU requires a valid {@literal subscription key}, to be provided either as
 * system property, environment variable or file in Vaadin home folder. See
 * {@link com.vaadin.pro.licensechecker.LocalSubscriptionKey} javadoc for
 * details.
 *
 * @see com.vaadin.pro.licensechecker.LocalSubscriptionKey
 */
public final class DauIntegration {

    /**
     * Starts Daily Active User tracking for the calling application.
     * <p>
     * </p>
     * On start, License Server is queried to verify subscription key validity
     * and enforcement status. If checks are passed, a background Job is started
     * to synchronize DAU data and get updated enforcement information at
     * regular intervals (24 hours).
     * <p>
     * </p>
     * This method is usually called once at application startup. Subsequent
     * calls to the method have no effects, unless tracking has been stopped
     * calling {@link #stopTracking()} method; in this case an exception will be
     * thrown.
     *
     * @param applicationName
     *            name of the application associated to the subscription key.
     * @throws com.vaadin.pro.licensechecker.LicenseException
     *             if subscription key is not available or invalid.
     * @throws IllegalStateException
     *             if invoked after a call to {@link #stopTracking()}
     */
    public static void startTracking(String applicationName) {
        LocalSubscriptionKey.getOrFail();
        DauStorage.getInstance().start(applicationName);
    }

    /**
     * Stops Daily Active User tracking for the calling application.
     * <p>
     * </p>
     * On stop, a last synchronization with License Server is attempted, to
     * flush remaining DAU data.
     * <p>
     * </p>
     * This method is usually called once at application shutdown. Subsequent
     * calls to the method have no effects. After tracking is stopped it cannot
     * be restarted.
     */
    public static void stopTracking() {
        DauStorage.getInstance().stop();
    }

    /**
     * Creates a new hashed identifier to track a potential Daily Active User of
     * the application.
     * <p>
     * </p>
     * The returned value is hashed with the {@link SubscriptionKey} and must be
     * used for subsequent interactions with DAU tracking API. Creating a new
     * tracking hash user does not imply a tracking entry to be added to DAU
     * data collection. To add the user to the tracking data set, the
     * application must call {@link #trackUser(String)} )}, providing the
     * tracking hash.
     *
     * @return a unique identifier to be used for DAU tracking API interaction.
     * @throws com.vaadin.pro.licensechecker.LicenseException
     *             if subscription key is invalid or expired
     */
    public static String newTrackingHash() {
        return hashValue(UUID.randomUUID().toString());
    }

    /**
     * Registers the user identified by the provided hash as a Daily Active
     * User, adding it to the DAU tracking data set.
     * <p>
     * </p>
     * When a user is tracked, the enforcement rule is evaluated and a
     * {@link EnforcementException} is thrown if enforcement is required and the
     * number of new users exceeds the limit.
     * <p>
     * </p>
     * Calling this method multiple times with the same {@code trackingHash} has
     * no effect, unless the DAU data set has been flushed or enforcement is
     * already applied to the user.
     *
     * @param trackingHash
     *            the user tracking identifier
     * @throws EnforcementException
     *             if enforcement should be applied by the client.
     */
    public static void trackUser(String trackingHash)
            throws EnforcementException {
        DauStorage.getInstance().track(trackingHash);
    }

    /**
     * Registers the user identified by the provided tracking hash and user
     * identity as a Daily Active User, adding it to the DAU tracking data set,
     * or assigns the identity to an existing "anonymous" tracked user.
     * <p>
     * </p>
     * A tracking hash can potentially be associated to different user
     * identities, for example if during the same browser session a user
     * accesses the application with different credentials.
     * <p>
     * </p>
     * The {@code userIdentity} value is never stored as is, but it is hashed
     * with the {@link SubscriptionKey} in order to prevent sensible information
     * to be kept in memory or sent to the License Server.
     * <p>
     * </p>
     * It the user was previously tracked without a user identity, the linked
     * entry will replace the anonymous one. However, if the tracked user is
     * already linked to different identities, it will be counted as a new user.
     * {@literal null} or {@literal blank} are intended as "anonymous" users and
     * a new tracking entry is created only if the tracking hash is not know to
     * the system.
     *
     * @param trackingHash
     *            the user tracking identifier
     * @param userIdentity
     *            a string that allow to identify an application authenticated
     *            user
     * @throws EnforcementException
     *             if enforcement should be applied by the client.
     */
    public static void trackUser(String trackingHash, String userIdentity) {
        if (userIdentity == null || userIdentity.trim().isEmpty()) {
            userIdentity = null;
        } else {
            userIdentity = hashValue(userIdentity);
        }
        DauStorage.getInstance().track(trackingHash, userIdentity);
    }

    private static String hashValue(String value) {
        SubscriptionKey key = LocalSubscriptionKey.getOrFail();
        return Util.getHash(value, key.getKey().getBytes(UTF_8), UTF_8);
    }

    /**
     * Checks if a valid subscription key is available.
     *
     * @throws com.vaadin.pro.licensechecker.LicenseException
     *             if subscription key is not available or invalid.
     */
    public static void checkSubscriptionKey() {
        DauStorage.getInstance().checkSubscriptionKey();
    }

    /**
     * Tells whether new user, i.e. not yet tracked and not yet counted, should
     * be blocked immediately.
     *
     * @return {@literal true} if the current request/user should be blocked,
     *         {@literal false} otherwise.
     */
    public static boolean shouldEnforce() {
        return DauStorage.getInstance().shouldEnforceThisUser();
    }

}
