/*
 * Copyright 2000-2025 Vaadin Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

package com.vaadin.pro.licensechecker;

import java.io.File;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.util.UUID;

import org.slf4j.Logger;

import elemental.json.Json;
import elemental.json.JsonObject;

/**
 * An internal helper class representing a user key.
 * <p>
 * </p>
 * Shares the same logic used by the Vaadin for statistics collection.
 */
class UserKey {

    static final String PROPERTY_USER_HOME = "user.home";
    static final String VAADIN_FOLDER_NAME = ".vaadin";
    static final String USER_KEY_FILE_NAME = "userKey"; // NOSONAR

    private final String key;

    UserKey(String key) {
        this.key = key;
    }

    /**
     * Gets the user key associated with this instance.
     *
     * @return the user key as a string
     */
    String getKey() {
        return key;
    }

    /**
     * Retrieves a {@link UserKey} instance. This method first attempts to
     * locate and read the user key from a file on the user's system. If the
     * file does not exist, a new {@link UserKey} is generated and written to
     * the file before being returned.
     *
     * @return a {@link UserKey} instance, either retrieved from an existing
     *         file or newly generated
     */
    static UserKey get() {
        File userKeyFile = resolveUserKeyLocation();
        if (userKeyFile.exists()) {
            return fromFile(userKeyFile);
        }
        return toFile(userKeyFile, new UserKey("user-" + UUID.randomUUID()));
    }

    private static UserKey toFile(File userKeyFile, UserKey userKey) {
        try {
            userKeyFile.getParentFile().mkdirs();
            JsonObject userKeyJson = Json.createObject();
            userKeyJson.put("key", userKey.getKey());
            Files.write(userKeyFile.toPath(),
                    userKeyJson.toJson().getBytes(StandardCharsets.UTF_8));
            return userKey;
        } catch (IOException e) {
            getLogger().debug("Failed to write generated UserKey", e);
        }
        return null;
    }

    private static UserKey fromFile(File userKeyFile) {
        try {
            String userKeyJson = new String(
                    Files.readAllBytes(userKeyFile.toPath()),
                    StandardCharsets.UTF_8);
            JsonObject json = Json.parse(userKeyJson);
            return new UserKey(json.getString("key"));
        } catch (Exception e) {
            getLogger().debug("Unable to read UserKey", e);
        }
        return null;
    }

    static File resolveUserKeyLocation() {
        String userHome = System.getProperty(PROPERTY_USER_HOME);
        File vaadinHome = new File(userHome, VAADIN_FOLDER_NAME);
        return new File(vaadinHome, USER_KEY_FILE_NAME);
    }

    private static Logger getLogger() {
        return LicenseChecker.getLogger();
    }

}
