/*
 * Decompiled with CFR 0.152.
 */
package com.vaadin.pro.licensechecker;

import com.nimbusds.jose.JOSEException;
import com.nimbusds.jose.crypto.ECDSAVerifier;
import com.nimbusds.jwt.JWTParser;
import com.nimbusds.jwt.SignedJWT;
import com.vaadin.pro.licensechecker.BuildType;
import com.vaadin.pro.licensechecker.History;
import com.vaadin.pro.licensechecker.LicenseChecker;
import com.vaadin.pro.licensechecker.LicenseException;
import com.vaadin.pro.licensechecker.MachineId;
import com.vaadin.pro.licensechecker.OfflineKey;
import com.vaadin.pro.licensechecker.Product;
import java.security.KeyFactory;
import java.security.NoSuchAlgorithmException;
import java.security.interfaces.ECPublicKey;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.X509EncodedKeySpec;
import java.text.ParseException;
import java.time.Duration;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneOffset;
import java.util.ArrayList;
import java.util.Base64;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.slf4j.Logger;

public class OfflineKeyValidator {
    private static final Map<String, String> PUBLIC_KEYS = new HashMap<String, String>();
    static String loggedLicenseOwner;

    private static Logger getLogger() {
        return LicenseChecker.getLogger();
    }

    private static boolean isExpired(long expires) {
        return Instant.now().isAfter(Instant.ofEpochMilli(expires));
    }

    boolean validate(Product product, BuildType buildType, List<OfflineKey> offlineKeys, MachineId machineId) {
        OfflineKeyValidator.getLogger().debug("Offline validation using offlineKey for " + product);
        if (offlineKeys == null || offlineKeys.isEmpty()) {
            OfflineKeyValidator.getLogger().debug("No offline key found");
            return false;
        }
        if (product != null && History.isRecentlyValidated(product, buildType, null)) {
            OfflineKeyValidator.getLogger().debug("Skipping check as product license was recently validated.");
            return true;
        }
        ArrayList<LicenseException> exceptions = new ArrayList<LicenseException>();
        for (OfflineKey offlineKey : offlineKeys) {
            try {
                this.validateForUse(offlineKey, product, buildType, machineId);
                if (loggedLicenseOwner == null && offlineKey.getName() != null) {
                    LocalDateTime expires = LocalDateTime.ofEpochSecond(offlineKey.getExpires() / 1000L, 0, ZoneOffset.UTC);
                    long expiresInDays = Duration.between(LocalDateTime.now(), expires).toDays();
                    loggedLicenseOwner = "Using offline license registered to " + offlineKey.getName() + " / " + offlineKey.getAccount() + ". Expires in " + expiresInDays + " days.";
                    OfflineKeyValidator.getLogger().info(loggedLicenseOwner);
                }
                return true;
            }
            catch (LicenseException e) {
                OfflineKeyValidator.getLogger().debug("Offline key validation failed: " + e.getMessage());
                exceptions.add(e);
            }
        }
        if (!exceptions.isEmpty()) {
            LicenseException primaryException = (LicenseException)exceptions.get(0);
            for (int i = 1; i < exceptions.size(); ++i) {
                primaryException.addSuppressed((Throwable)exceptions.get(i));
            }
            throw primaryException;
        }
        return false;
    }

    private void validateForUse(OfflineKey offlineKey, Product product, BuildType buildType, MachineId machineId) throws LicenseException {
        String licenseMachineId;
        this.validateOfflineKeySignature(offlineKey, machineId);
        if (buildType != null) {
            if (buildType == BuildType.DEVELOPMENT && !offlineKey.isDevelopmentBuildAllowed()) {
                OfflineKeyValidator.getLogger().debug("Offline key is not for development");
                throw new LicenseException(OfflineKeyValidator.getNotDevelopmentMessage(machineId));
            }
            if (buildType == BuildType.PRODUCTION && !offlineKey.isProductionBuildAllowed()) {
                OfflineKeyValidator.getLogger().debug("Offline key is not for production builds");
                throw new LicenseException(OfflineKeyValidator.getNotProductionBuildMessage(machineId));
            }
        }
        if ((licenseMachineId = offlineKey.getMachineId()) != null && !machineId.stablePartMatches(licenseMachineId)) {
            OfflineKeyValidator.getLogger().debug("Offline key has incorrect machine id");
            throw new LicenseException(OfflineKeyValidator.getInvalidOfflineKeyMessage(machineId));
        }
        if (OfflineKeyValidator.isExpired(offlineKey.getExpires())) {
            OfflineKeyValidator.getLogger().debug("Offline key expired");
            throw new LicenseException(OfflineKeyValidator.getExpiredOfflineKeyMessage(machineId));
        }
        if (product != null) {
            if (!this.containsProduct(offlineKey, product)) {
                throw new LicenseException("The offline key does not provide access to " + product.getName() + " " + product.getVersion());
            }
            History.setLastCheckTimeNow(product, buildType, null);
            History.setLastSubscription(product, offlineKey.getSubscription(), null);
        }
        OfflineKeyValidator.getLogger().debug("Offline key OK");
    }

    private boolean containsProduct(OfflineKey offlineKey, Product product) {
        if (product.getName().startsWith("test-")) {
            return true;
        }
        boolean hasExtendedSupport = offlineKey.getAllowedFeatures().contains("extendedsupport");
        if ("vaadin-framework".equals(product.getName())) {
            if (product.getVersion().startsWith("7") && (hasExtendedSupport || offlineKey.getAllowedFeatures().contains("v7extendedsupport"))) {
                return true;
            }
            if (product.getVersion().startsWith("8") && (hasExtendedSupport || offlineKey.getAllowedFeatures().contains("v8extendedsupport"))) {
                return true;
            }
        } else if ("flow".equals(product.getName())) {
            return hasExtendedSupport;
        }
        return offlineKey.getAllowedProducts().contains(product.getName());
    }

    void validateOfflineKeySignature(OfflineKey offlineKey, MachineId machineIdForLinks) {
        try {
            String jwtData = offlineKey.getJwtData();
            SignedJWT jwt = (SignedJWT)JWTParser.parse(jwtData);
            String kid = jwt.getHeader().getKeyID();
            KeyFactory fact = KeyFactory.getInstance("EC");
            byte[] encoded = Base64.getDecoder().decode(PUBLIC_KEYS.get(kid));
            X509EncodedKeySpec keySpec = new X509EncodedKeySpec(encoded);
            ECPublicKey publicKey = (ECPublicKey)fact.generatePublic(keySpec);
            ECDSAVerifier verifier = new ECDSAVerifier(publicKey);
            if (!jwt.verify(verifier)) {
                OfflineKeyValidator.getLogger().debug("Offline key failed verification");
                throw new LicenseException(OfflineKeyValidator.getInvalidOfflineKeyMessage(machineIdForLinks));
            }
        }
        catch (NoSuchAlgorithmException | InvalidKeySpecException e) {
            OfflineKeyValidator.getLogger().debug("Offline key could not be read", e);
            throw new LicenseException(OfflineKeyValidator.getErrorValidatingOfflineKeyMessage(machineIdForLinks), e);
        }
        catch (ParseException e) {
            OfflineKeyValidator.getLogger().debug("Error parsing offline key", e);
            throw new LicenseException(OfflineKeyValidator.getInvalidOfflineKeyMessage(machineIdForLinks), e);
        }
        catch (JOSEException e) {
            OfflineKeyValidator.getLogger().debug("Error reading offline key", e);
            throw new LicenseException(OfflineKeyValidator.getInvalidOfflineKeyMessage(machineIdForLinks), e);
        }
    }

    private static String getExpiredOfflineKeyMessage(MachineId machineId) {
        return "Offline key has expired, " + OfflineKeyValidator.getOfflineKeyLinkMessage(machineId);
    }

    private static String getNotDevelopmentMessage(MachineId machineId) {
        return "The provided offline key does not allow development, " + OfflineKeyValidator.getOfflineKeyLinkMessage(machineId);
    }

    private static String getNotProductionBuildMessage(MachineId machineId) {
        return "The provided offline key does not allow production builds, " + OfflineKeyValidator.getOfflineKeyLinkMessage(machineId);
    }

    static String getOfflineKeyLinkMessage(MachineId machineId) {
        return "please go to " + OfflineKeyValidator.getOfflineUrl(machineId) + " to retrieve an offline key.\nFor CI/CD build servers, you need to download a server license key, which can work offline to create production builds. You can download a server license key from https://vaadin.com/myaccount/licenses.\nFor troubleshooting steps, see https://vaadin.com/licensing-faq-and-troubleshooting.";
    }

    @Deprecated
    public static String getOfflineUrl(String machineId) {
        return "https://vaadin.com/pro/validate-license?getOfflineKey=" + machineId;
    }

    public static String getOfflineUrl(MachineId machineId) {
        String url = "https://vaadin.com/pro/validate-license?getOfflineKey=";
        if (machineId != null) {
            url = url + machineId.getPrimaryIdFull();
        }
        return url;
    }

    private static String getErrorValidatingOfflineKeyMessage(MachineId machineId) {
        return "Unable to validate offline key, " + OfflineKeyValidator.getOfflineKeyLinkMessage(machineId);
    }

    static String getInvalidOfflineKeyMessage(MachineId machineId) {
        return "Invalid offline key, " + OfflineKeyValidator.getOfflineKeyLinkMessage(machineId);
    }

    static String getMissingOfflineKeyMessage(MachineId machineId) {
        return "The license server at https://tools.vaadin.com/ could not be reached and no offline key was found. To use the product without an internet connection " + OfflineKeyValidator.getOfflineKeyLinkMessage(machineId);
    }

    static {
        PUBLIC_KEYS.put("b98c7421853a2d11fb2be2fb73f89be54414a4e9", "MIGbMBAGByqGSM49AgEGBSuBBAAjA4GGAAQB/qMpeWPlOKTd8+93GSpi3s/CQMx1gpkw728vl8iijo2965zIBD1bePNULT9VK1iul2iNJA2ev9ImXecLAA4UoMwAlz3tQHIA8zJksNbQUHZhzS74hH/jJr9pE6ra4Q3lnNvmJKEXkFvCpUoBmdYS94Hu0MFXFi16IJfooLW6qzmtUGs=");
        PUBLIC_KEYS.put("542764e7000908e65dc3fc1dabf4e2cd28966758", "MIGbMBAGByqGSM49AgEGBSuBBAAjA4GGAAQABaxDhdMljdpoM43y31co033oQjTZoCj+Wjby9LRBPmdlvMTAJV6gXOzZHDpXQb4N1O0NJr4AeXxaE4GO/p4GGywAkg+SYIO1v8X+n2beq1czN+i8WL1cfUu8DFITUkSHtULtPyNTvW1Ew7XeTGVUQ6n/Xz2YfAy7tcoFDsldrurE1nY=");
        loggedLicenseOwner = null;
    }
}

