/*
 * Decompiled with CFR 0.152.
 */
package com.vaadin.pro.licensechecker;

import com.vaadin.pro.licensechecker.History;
import com.vaadin.pro.licensechecker.LicenseValidationException;
import com.vaadin.pro.licensechecker.PreTrial;
import com.vaadin.pro.licensechecker.PreTrialCheckException;
import com.vaadin.pro.licensechecker.PreTrialCreationException;
import com.vaadin.pro.licensechecker.PreTrialLicenseValidationException;
import com.vaadin.pro.licensechecker.PreTrialStatusCheckException;
import com.vaadin.pro.licensechecker.Product;
import com.vaadin.pro.licensechecker.Util;
import com.vaadin.pro.licensechecker.json.LicenseCheckerJson;
import com.vaadin.pro.licensechecker.json.LicenseCheckerJsonObject;
import java.io.IOException;
import java.io.OutputStream;
import java.net.HttpURLConnection;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.time.Instant;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.Callable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class PreTrialValidator {
    private static final Logger LOGGER = LoggerFactory.getLogger(PreTrialValidator.class);
    private static final String VALIDATION_ERROR_MESSAGE = "Cannot validate pre-trial for {} and machineId {}. {}";
    private static final String VALIDATION_STATUS_ERROR_MESSAGE = "Cannot validate pre-trial status for {} and machineId {}. {}";
    private static final String CREATE_ERROR_MESSAGE = "Cannot create pre-trial for machineId {}";
    private static final int CONNECTION_TIMEOUT_MS = 5000;
    private static final int READ_TIMEOUT_MS = 5000;
    private static final String CONTENT_TYPE_JSON = "application/json";
    private static final String PRETRIAL_AUTH_PREFIX = "PRETRIAL ";
    private static final String VALIDATE_ENDPOINT = "/validate/v1?productName=%s&productVersion=%s";
    private static final String CREATE_ENDPOINT = "create";
    private static final String CHECK_ENDPOINT = "check";
    private final String licenseServerBaseURL;

    PreTrialValidator() {
        this(System.getProperty("vaadin.licenseServerBaseUrl", "https://tools.vaadin.com/vaadin-license-server"));
    }

    PreTrialValidator(String licenseServerURL) {
        String formattedUrl = Util.removeTrailingSlash(licenseServerURL);
        this.licenseServerBaseURL = Util.validateURL(formattedUrl);
    }

    PreTrial create(String machineId, String userKey) {
        PreTrialValidator.validateStringParameter("machineId", machineId);
        PreTrialValidator.validateStringParameter("userKey", userKey);
        LOGGER.trace("Submitting PRE TRIAL create request for machineId {} and userKey {}", (Object)machineId, (Object)userKey);
        Callable<HttpURLConnection> urlConnectionFactory = () -> this.getHttpURLConnection(this.licenseServerBaseURL + "/pretrial/" + CREATE_ENDPOINT, "POST", machineId, new String[0]);
        try {
            LicenseCheckerJsonObject requestBody = LicenseCheckerJson.createObject();
            requestBody.put("machineId", machineId);
            requestBody.put("userKey", userKey);
            requestBody.put("timestamp", Util.TIMESTAMP_FORMAT.format(Instant.now()));
            Response response = this.submitRequest(urlConnectionFactory, requestBody);
            Result result = this.parseResponse(response, false);
            LOGGER.debug("Pre-trial create result: {}", (Object)result);
            if (result instanceof LicenseServerResult) {
                return ((LicenseServerResult)result).preTrial;
            }
            RequestFailure failure = (RequestFailure)result;
            if (response.status == 401) {
                throw new PreTrialCreationException.Expired(failure.getErrorMessage());
            }
            throw new PreTrialCreationException(failure.getErrorMessage());
        }
        catch (PreTrialCreationException e) {
            throw e;
        }
        catch (Exception e) {
            LOGGER.error(CREATE_ERROR_MESSAGE, (Object)machineId, (Object)e);
            throw new PreTrialCreationException(e.getMessage());
        }
    }

    Result validate(Product product, String machineId, boolean quiet) {
        PreTrialValidator.validateStringParameter("machineId", machineId);
        PreTrialValidator.validateProduct(product);
        if (History.isPreTrialRecentlyValidated(product, machineId)) {
            LOGGER.debug("Skipping check as pre-trial license was recently validated and not yet expired.");
            return new RecentlyValidated();
        }
        try {
            LOGGER.trace("Submitting PRE TRIAL validate request for machineId {} and {}", (Object)machineId, (Object)product);
            Callable<HttpURLConnection> urlConnectionFactory = () -> this.getHttpURLConnection(this.licenseServerBaseURL + "/licenses" + VALIDATE_ENDPOINT, "GET", machineId, product.getName(), product.getVersion());
            Result result = this.parseResponse(this.submitRequest(urlConnectionFactory, null), true);
            LOGGER.debug("Validation result: {}", (Object)result);
            if (result.getAccess() == Access.OK) {
                History.setLastPreTrialCheckTime(product, machineId, Instant.now(), ((LicenseServerResult)result).preTrial);
            }
            this.logResponseErrors(result, product, machineId, quiet);
            return result;
        }
        catch (Exception e) {
            if (quiet) {
                LOGGER.debug(VALIDATION_ERROR_MESSAGE, product, machineId, e.getMessage());
            } else {
                LOGGER.error(VALIDATION_ERROR_MESSAGE, product, machineId, "", e);
            }
            return new RequestFailure(Access.CANNOT_REACH_SERVER, null);
        }
    }

    Optional<PreTrial> statusCheck(Product product, String machineId) {
        PreTrialValidator.validateStringParameter("machineId", machineId);
        PreTrialValidator.validateProduct(product);
        PreTrial lastKnownPreTrial = History.getRecentlyValidatedPreTrial(product, machineId);
        if (lastKnownPreTrial != null) {
            return Optional.of(lastKnownPreTrial);
        }
        try {
            LOGGER.trace("Submitting PRE TRIAL validate request for status check for machineId {} and {}", (Object)machineId, (Object)product);
            Callable<HttpURLConnection> urlConnectionFactory = () -> this.getHttpURLConnection(this.licenseServerBaseURL + "/licenses" + VALIDATE_ENDPOINT, "GET", machineId, product.getName(), product.getVersion());
            Result result = this.parseResponse(this.submitRequest(urlConnectionFactory, null), true);
            LOGGER.debug("Validation result: {}", (Object)result);
            this.logResponseErrors(result, product, machineId, true, VALIDATION_STATUS_ERROR_MESSAGE);
            if (result.getAccess() == Access.OK) {
                History.setLastPreTrialCheckTime(product, machineId, Instant.now(), ((LicenseServerResult)result).preTrial);
            }
            if (result instanceof RequestFailure) {
                throw new PreTrialStatusCheckException(((RequestFailure)result).getErrorMessage(), result.getAccess() == Access.CANNOT_REACH_SERVER);
            }
            return result instanceof LicenseServerResult ? Optional.of(((LicenseServerResult)result).preTrial) : Optional.empty();
        }
        catch (Exception e) {
            LOGGER.error(VALIDATION_STATUS_ERROR_MESSAGE, product, machineId, "", e);
            throw new PreTrialStatusCheckException(e.getMessage(), true);
        }
    }

    Optional<PreTrial> check(Product product, String machineId, String userKey) {
        PreTrialValidator.validateStringParameter("machineId", machineId);
        PreTrialValidator.validateStringParameter("userKey", userKey);
        PreTrialValidator.validateProduct(product);
        LOGGER.trace("Submitting PRE TRIAL check request for machineId {}, userKey {} and {}", machineId, userKey, product);
        Callable<HttpURLConnection> urlConnectionFactory = () -> this.getHttpURLConnection(this.licenseServerBaseURL + "/pretrial/" + CHECK_ENDPOINT, "POST", machineId, new String[0]);
        try {
            LicenseCheckerJsonObject requestBody = LicenseCheckerJson.createObject();
            requestBody.put("machineId", machineId);
            requestBody.put("userKey", userKey);
            requestBody.put("productName", product.getName());
            requestBody.put("productVersion", product.getVersion());
            requestBody.put("timestamp", Util.TIMESTAMP_FORMAT.format(Instant.now()));
            Response response = this.submitRequest(urlConnectionFactory, requestBody);
            Result result = this.parseResponse(response, false);
            LOGGER.debug("Pre-trial create result: {}", (Object)result);
            if (result instanceof LicenseServerResult) {
                return Optional.of(((LicenseServerResult)result).preTrial);
            }
            RequestFailure failure = (RequestFailure)result;
            if (response.status == 404) {
                return Optional.empty();
            }
            if (response.status == 401) {
                throw new PreTrialCheckException.Expired(failure.getErrorMessage());
            }
            throw new PreTrialCheckException(failure.getErrorMessage());
        }
        catch (PreTrialCheckException e) {
            throw e;
        }
        catch (Exception e) {
            LOGGER.error(CREATE_ERROR_MESSAGE, (Object)machineId, (Object)e);
            throw new PreTrialCheckException(e.getMessage());
        }
    }

    private static void validateProduct(Product product) {
        if (product == null) {
            throw new IllegalArgumentException("product cannot be null");
        }
    }

    private static void validateStringParameter(String paramName, String paramValue) {
        if (paramValue == null || paramValue.trim().isEmpty()) {
            throw new IllegalArgumentException(paramName + " cannot be null or empty");
        }
    }

    private void logResponseErrors(Result result, Product product, String machineId, boolean quiet) {
        this.logResponseErrors(result, product, machineId, quiet, VALIDATION_ERROR_MESSAGE);
    }

    private void logResponseErrors(Result result, Product product, String machineId, boolean quiet, String messagePattern) {
        String message;
        if (result instanceof RequestFailure && (message = ((RequestFailure)result).getErrorMessage()) != null) {
            if (quiet) {
                LOGGER.debug(messagePattern, product, machineId, message);
            } else {
                LOGGER.error(messagePattern, product, machineId, "");
            }
        }
    }

    private Result parseResponse(Response response, boolean validate) throws IOException {
        LicenseCheckerJsonObject body = response.body;
        LOGGER.trace("License Server response: {}", (Object)body.toJson());
        if (body.hasKey("error") && body.hasKey("message")) {
            return this.createErrorResult(body);
        }
        if (body.hasKey("trialName")) {
            return this.createTrialResult(body, validate);
        }
        throw new IOException("Invalid response from License Server: " + body);
    }

    private Result createTrialResult(LicenseCheckerJsonObject response, boolean validate) throws IOException {
        PreTrial.PreTrialState trialState;
        if (validate) {
            try {
                trialState = PreTrial.PreTrialState.valueOf(response.getString("trialState"));
            }
            catch (IllegalArgumentException e) {
                throw new IOException("Invalid trial state from License Server: " + response);
            }
        } else {
            trialState = PreTrial.PreTrialState.RUNNING;
        }
        PreTrial preTrial = new PreTrial(response.getString("trialName"), trialState, (int)response.getNumber("daysRemaining"), (int)response.getNumber("daysRemainingUntilRenewal"));
        return new LicenseServerResult(preTrial);
    }

    private Result createErrorResult(LicenseCheckerJsonObject response) {
        StringBuilder errorMessageBuilder = new StringBuilder("Error [").append(response.getString("error")).append("] ").append(response.getString("message"));
        if (response.hasKey("detail")) {
            errorMessageBuilder.append(". ").append(response.getString("detail"));
        }
        if (response.hasKey("path")) {
            errorMessageBuilder.append(". (Path: ").append(response.getString("path")).append(")");
        }
        return new RequestFailure(Access.NO_ACCESS, errorMessageBuilder.toString());
    }

    private Response submitRequest(Callable<HttpURLConnection> urlConnectionFactory, LicenseCheckerJsonObject requestBody) throws Exception {
        int status;
        HttpURLConnection http = urlConnectionFactory.call();
        if (requestBody != null) {
            try (OutputStream os = http.getOutputStream();){
                os.write(requestBody.toJson().getBytes(StandardCharsets.UTF_8));
            }
        }
        if ((status = http.getResponseCode()) >= 200 && status < 300) {
            return new Response(status, Util.parseJson(status, http.getInputStream()));
        }
        if (status >= 400 && status < 600) {
            return new Response(status, Util.parseJson(status, http.getErrorStream()));
        }
        throw new IOException("Unexpected response from License Server: " + status);
    }

    private HttpURLConnection getHttpURLConnection(String endpointURL, String httpMethod, String machineId, String ... queryParams) throws IOException {
        URL url = new URL(String.format(endpointURL, queryParams));
        HttpURLConnection http = (HttpURLConnection)url.openConnection();
        http.setRequestMethod(httpMethod);
        http.setRequestProperty("Content-Type", CONTENT_TYPE_JSON);
        http.setRequestProperty("Accept", CONTENT_TYPE_JSON);
        http.setRequestProperty("Authorization", PRETRIAL_AUTH_PREFIX + machineId);
        http.setConnectTimeout(5000);
        http.setReadTimeout(5000);
        http.setDoOutput(true);
        return http;
    }

    private static class Response {
        private final int status;
        private final LicenseCheckerJsonObject body;

        Response(int status, LicenseCheckerJsonObject body) {
            this.status = status;
            this.body = body;
        }
    }

    public static abstract class Result {
        protected abstract Access getAccess();

        protected abstract LicenseValidationException buildError(boolean var1);

        public Access verifyAccess(boolean canValidateOnline) {
            LicenseValidationException error = this.buildError(canValidateOnline);
            if (error != null) {
                throw error;
            }
            return this.getAccess();
        }
    }

    static final class LicenseServerResult
    extends Result {
        private final PreTrial preTrial;
        private final Access access;

        LicenseServerResult(PreTrial preTrial) {
            this.preTrial = Objects.requireNonNull(preTrial, "trial cannot be null");
            this.access = preTrial.trialState == PreTrial.PreTrialState.RUNNING ? Access.OK : Access.NO_ACCESS;
        }

        @Override
        protected Access getAccess() {
            return this.access;
        }

        @Override
        protected LicenseValidationException buildError(boolean canValidateOnline) {
            if (this.access == Access.NO_ACCESS && !canValidateOnline) {
                return LicenseServerResult.createException(this.preTrial);
            }
            if (canValidateOnline && this.access == Access.NO_ACCESS && this.preTrial.trialState != PreTrial.PreTrialState.START_ALLOWED) {
                return LicenseServerResult.createException(this.preTrial);
            }
            return null;
        }

        public String toString() {
            return "Pre-trial: response from License Server [access=" + (Object)((Object)this.access) + ", preTrial=" + this.preTrial + ']';
        }

        private static LicenseValidationException createException(PreTrial preTrial) {
            if (preTrial.getTrialState() == PreTrial.PreTrialState.ACCESS_DENIED) {
                return new LicenseValidationException("Pre trial does not grant access to the product.");
            }
            return new PreTrialLicenseValidationException(preTrial);
        }
    }

    static final class RequestFailure
    extends Result {
        private final Access access;
        private final String errorMessage;

        RequestFailure(Access access, String errorMessage) {
            this.access = access;
            this.errorMessage = errorMessage;
        }

        @Override
        protected Access getAccess() {
            return this.access;
        }

        @Override
        protected LicenseValidationException buildError(boolean canValidateOnline) {
            if (this.access == Access.CANNOT_REACH_SERVER && canValidateOnline) {
                return null;
            }
            return new LicenseValidationException(this.errorMessage);
        }

        public String toString() {
            return "Pre-Trial: request to License Server failed [access=" + (Object)((Object)this.access) + ", errorMessage=" + this.errorMessage + ']';
        }

        String getErrorMessage() {
            return this.errorMessage;
        }
    }

    static final class RecentlyValidated
    extends Result {
        RecentlyValidated() {
        }

        @Override
        protected Access getAccess() {
            return Access.OK;
        }

        @Override
        protected LicenseValidationException buildError(boolean canValidateOnline) {
            return null;
        }

        public String toString() {
            return "Pre-trial: recently validated [access=OK]";
        }
    }

    public static enum Access {
        OK,
        NO_ACCESS,
        CANNOT_REACH_SERVER;

    }
}

