/*
 * Decompiled with CFR 0.152.
 */
package com.vaadin.pro.licensechecker;

import com.vaadin.pro.licensechecker.BuildType;
import com.vaadin.pro.licensechecker.History;
import com.vaadin.pro.licensechecker.LicenseChecker;
import com.vaadin.pro.licensechecker.LicenseException;
import com.vaadin.pro.licensechecker.LicenseValidationException;
import com.vaadin.pro.licensechecker.Platform;
import com.vaadin.pro.licensechecker.ProKey;
import com.vaadin.pro.licensechecker.Product;
import com.vaadin.pro.licensechecker.SubscriptionKey;
import com.vaadin.pro.licensechecker.Util;
import com.vaadin.pro.licensechecker.dau.DauIntegration;
import com.vaadin.pro.licensechecker.json.LicenseCheckerJson;
import com.vaadin.pro.licensechecker.json.LicenseCheckerJsonObject;
import java.io.IOException;
import java.io.InputStream;
import java.net.ConnectException;
import java.net.SocketTimeoutException;
import java.net.URL;
import java.net.URLConnection;
import java.net.UnknownHostException;
import javax.net.ssl.SSLHandshakeException;
import org.slf4j.Logger;

public class OnlineKeyValidator {
    private static final String UNABLE_TO_VALIDATE_SUBSCRIPTION = "Unable to validate subscription. Please go to https://vaadin.com/pro/validate-license to check that your subscription is active. For troubleshooting steps, see https://vaadin.com/licensing-faq-and-troubleshooting.";
    private static final String LICENSE_SERVER_URL = System.getProperty("vaadin.licenseServerBaseUrl", "https://tools.vaadin.com/vaadin-license-server");
    private static final String[] PROPERTIES = new String[]{"java.vendor", "java.version", "os.arch", "os.name", "os.version"};

    private static Logger getLogger() {
        return LicenseChecker.getLogger();
    }

    Result validate(Product product, ProKey proKey, String machineId, BuildType buildType, boolean quiet) {
        OnlineKeyValidator.getLogger().debug("Online validation using proKey for " + product);
        if (proKey == null) {
            OnlineKeyValidator.getLogger().debug("No pro key found");
            return Result.NO_ACCESS;
        }
        if (History.isRecentlyValidated(product, buildType, proKey)) {
            OnlineKeyValidator.getLogger().debug("Skipping check as product license was recently validated.");
            return Result.OK;
        }
        try {
            LicenseCheckerJsonObject response = OnlineKeyValidator.queryServer(product, proKey, machineId, buildType);
            if (OnlineKeyValidator.validateServerResponse(product, response, quiet)) {
                History.setLastCheckTimeNow(product, buildType, proKey);
                History.setLastSubscription(product, response.getString("subscription"), proKey);
                OnlineKeyValidator.getLogger().debug("Pro key OK");
                return Result.OK;
            }
            OnlineKeyValidator.getLogger().debug("Pro key does not have access");
            return Result.NO_ACCESS;
        }
        catch (ConnectException | SocketTimeoutException | UnknownHostException e) {
            OnlineKeyValidator.getLogger().debug("Pro key unable to connect to server");
            return Result.CANNOT_REACH_SERVER;
        }
        catch (SSLHandshakeException e) {
            OnlineKeyValidator.getLogger().warn("Pro key unable to connect to server", e);
            return Result.CANNOT_REACH_SERVER;
        }
        catch (Exception e) {
            if (!quiet) {
                OnlineKeyValidator.getLogger().error(UNABLE_TO_VALIDATE_SUBSCRIPTION, e);
            }
            OnlineKeyValidator.getLogger().debug("Pro key checking failed with exception", e);
            return Result.NO_ACCESS;
        }
    }

    Result validate(Product product, SubscriptionKey subscriptionKey, BuildType buildType, boolean quiet) {
        OnlineKeyValidator.getLogger().debug("Online validation using subscriptionKey for " + product);
        if (subscriptionKey == null) {
            OnlineKeyValidator.getLogger().debug("No subscription key found");
            return Result.NO_ACCESS;
        }
        ProKey subscriptionKeyAsProKey = new ProKey(null, subscriptionKey.getKey());
        if (History.isRecentlyValidated(product, buildType, subscriptionKeyAsProKey)) {
            OnlineKeyValidator.getLogger().debug("Skipping check as product license was recently validated.");
            return Result.OK;
        }
        try {
            DauIntegration.checkSubscriptionKey();
        }
        catch (LicenseValidationException ex) {
            Throwable cause = ex.getCause();
            if (cause instanceof ConnectException || cause instanceof SocketTimeoutException || cause instanceof UnknownHostException) {
                OnlineKeyValidator.getLogger().debug("Subscription key unable to connect to server");
                return Result.CANNOT_REACH_SERVER;
            }
            if (!quiet) {
                OnlineKeyValidator.getLogger().error(UNABLE_TO_VALIDATE_SUBSCRIPTION, cause);
            }
            OnlineKeyValidator.getLogger().debug("Subscription key checking failed with exception", ex);
            return Result.NO_ACCESS;
        }
        catch (LicenseException ex) {
            OnlineKeyValidator.getLogger().debug("Subscription key does not have access");
            return Result.NO_ACCESS;
        }
        History.setLastCheckTimeNow(product, buildType, subscriptionKeyAsProKey);
        return Result.OK;
    }

    static String getSubscription(Product product, ProKey proKey) {
        if (proKey == null) {
            OnlineKeyValidator.getLogger().debug("No pro key found to get subscription");
            return null;
        }
        return History.getLastSubscription(product, proKey);
    }

    private static boolean validateServerResponse(Product product, LicenseCheckerJsonObject response, boolean quiet) {
        OnlineKeyValidator.getLogger().debug("Validating license for " + product.getName() + " " + product.getVersion());
        String result = response.getString("result");
        String message = response.getString("message");
        if (!quiet && message != null && !message.isEmpty()) {
            OnlineKeyValidator.getLogger().info(message);
        }
        return "ok".equals(result);
    }

    private static LicenseCheckerJsonObject queryServer(Product product, ProKey proKey, String machineId, BuildType buildType) throws IOException {
        String validationUrl = Util.removeTrailingSlash(LICENSE_SERVER_URL);
        validationUrl = validationUrl + "/licenses/pro";
        URL url = new URL(Util.validateURL(validationUrl));
        URLConnection http = url.openConnection();
        http.setRequestProperty("check-source", "java");
        http.setRequestProperty("machine-id", machineId);
        Platform.getVaadinVersion().ifPresent(version -> http.setRequestProperty("vaadin-version", (String)version));
        http.setRequestProperty("product-name", product.getName());
        http.setRequestProperty("product-version", product.getVersion());
        if (buildType != null) {
            http.setRequestProperty("build-type", buildType.getKey());
        }
        for (String property : PROPERTIES) {
            String value = System.getProperty(property);
            if (value == null) continue;
            http.setRequestProperty("prop-" + property.replace(".", "-"), value);
        }
        http.setRequestProperty("product-version", product.getVersion());
        http.setRequestProperty("Cookie", "proKey=" + proKey.getProKey());
        http.setConnectTimeout(5000);
        http.setReadTimeout(5000);
        http.connect();
        try (InputStream in = http.getInputStream();){
            String response = Util.toString(in);
            LicenseCheckerJsonObject licenseCheckerJsonObject = LicenseCheckerJson.parse(response);
            return licenseCheckerJsonObject;
        }
    }

    public static enum Result {
        OK,
        NO_ACCESS,
        CANNOT_REACH_SERVER;

    }
}

