/*
 * Decompiled with CFR 0.152.
 */
package com.vaadin.kubernetes.starter.sessiontracker;

import com.vaadin.flow.server.HandlerHelper;
import com.vaadin.kubernetes.starter.KubernetesKitProperties;
import com.vaadin.kubernetes.starter.sessiontracker.CurrentKey;
import com.vaadin.kubernetes.starter.sessiontracker.SameSite;
import com.vaadin.kubernetes.starter.sessiontracker.SessionListener;
import com.vaadin.kubernetes.starter.sessiontracker.SessionSerializer;
import com.vaadin.kubernetes.starter.sessiontracker.SessionTrackerCookie;
import jakarta.servlet.FilterChain;
import jakarta.servlet.ServletException;
import jakarta.servlet.http.Cookie;
import jakarta.servlet.http.HttpFilter;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpServletResponse;
import jakarta.servlet.http.HttpSession;
import java.io.IOException;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.web.util.UriComponentsBuilder;

public class SessionTrackerFilter
extends HttpFilter {
    private final transient SessionSerializer sessionSerializer;
    private final transient KubernetesKitProperties properties;
    private final transient Runnable destroyCallback;
    private final SessionListener sessionListener;
    private final ConcurrentHashMap<String, CompletableFuture<String>> pendingSessionCreation = new ConcurrentHashMap();
    private static final long SESSION_CREATION_TIMEOUT_SECONDS = 30L;
    private static final int SESSION_COOKIE_PROPAGATION_DELAY_MS = 500;

    public SessionTrackerFilter(SessionSerializer sessionSerializer, KubernetesKitProperties properties, SessionListener sessionListener) {
        this.sessionSerializer = sessionSerializer;
        this.properties = properties;
        this.destroyCallback = sessionListener::stop;
        this.sessionListener = sessionListener;
    }

    protected void doFilter(HttpServletRequest request, HttpServletResponse response, FilterChain chain) throws IOException, ServletException {
        String cookieName = this.properties.getClusterKeyCookieName();
        AtomicReference<Boolean> replayRequestRequired = new AtomicReference<Boolean>(false);
        Optional<String> trackerCookie = SessionTrackerCookie.getValue(request, cookieName);
        if (trackerCookie.isPresent() && this.sessionSerializer.isRunning()) {
            String key = trackerCookie.get();
            if (request.getSession(false) == null) {
                this.createOrWaitForSession(request, key, replayRequestRequired);
            } else {
                this.getLogger().trace("Session already exists for cluster key {} on request {}", (Object)key, (Object)request.getRequestURI());
                this.pendingSessionCreation.remove(key);
            }
        }
        if (!this.sessionSerializer.isRunning()) {
            this.forceRequestReply(request, response, "Redirecting current request session ID {} to {} because server is shutting down");
            return;
        }
        if (Boolean.TRUE.equals(replayRequestRequired.get())) {
            this.forceRequestReply(request, response, "Redirecting current request session ID {} to {} to use the new session ID");
            return;
        }
        CompletableFuture<String> futureToComplete = null;
        String clusterKey = CurrentKey.get();
        if (clusterKey != null) {
            futureToComplete = this.pendingSessionCreation.get(clusterKey);
        }
        try {
            HttpSession session = request.getSession(false);
            SessionTrackerCookie.setIfNeeded(session, request, response, cookieName, this.cookieConsumer(request));
            super.doFilter(request, response, chain);
            if (session != null && request.isRequestedSessionIdValid() && HandlerHelper.RequestType.UIDL.getIdentifier().equals(request.getParameter("v-r"))) {
                this.sessionSerializer.serialize(session);
            }
            this.flushResponseAndUnlockPendingRequests(clusterKey, response, session, futureToComplete);
        }
        catch (Exception e) {
            if (futureToComplete != null && !futureToComplete.isDone()) {
                this.getLogger().error("Error processing request for cluster key {}, notifying waiting threads", (Object)clusterKey, (Object)e);
                futureToComplete.completeExceptionally(e);
                this.pendingSessionCreation.remove(clusterKey);
            }
            throw e;
        }
        finally {
            CurrentKey.clear();
        }
    }

    private void forceRequestReply(HttpServletRequest request, HttpServletResponse response, String message) throws IOException {
        String redirectUrl = this.buildRedirectUrl(request);
        this.getLogger().debug(message, (Object)request.getRequestedSessionId(), (Object)redirectUrl);
        response.sendRedirect(redirectUrl, 307);
    }

    private void createOrWaitForSession(HttpServletRequest request, String clusterKey, AtomicReference<Boolean> replayRequestRequired) {
        this.getLogger().debug("Request with cluster key {} with requested session id {}: {}", new Object[]{clusterKey, request.getRequestedSessionId(), request.getRequestURI()});
        AtomicReference sessionCreatingFutureHolder = new AtomicReference();
        CompletableFuture existingFuture = this.pendingSessionCreation.computeIfAbsent(clusterKey, k -> {
            this.getLogger().debug("Initiating session creation for cluster key {}", (Object)clusterKey);
            return this.sessionListener.findExistingSession(clusterKey, request).map(CompletableFuture::completedFuture).orElseGet(() -> {
                CompletableFuture newFuture = new CompletableFuture();
                sessionCreatingFutureHolder.set(newFuture);
                return newFuture;
            });
        });
        if (sessionCreatingFutureHolder.get() != null) {
            HttpSession newSession = this.createOrWaitForSession(request, clusterKey, existingFuture);
            if (newSession == null) {
                replayRequestRequired.set(true);
            }
        } else {
            replayRequestRequired.set(this.waitForSessionCreation(request, clusterKey, existingFuture));
        }
    }

    private void flushResponseAndUnlockPendingRequests(String clusterKey, HttpServletResponse response, HttpSession session, CompletableFuture<String> sessionCreationFuture) {
        if (sessionCreationFuture != null && !sessionCreationFuture.isDone()) {
            String newSessionId = session != null ? session.getId() : null;
            try {
                response.flushBuffer();
                this.getLogger().debug("Response flushed, completing session {} creation future for cluster key {}", (Object)newSessionId, (Object)clusterKey);
            }
            catch (IOException e) {
                this.getLogger().warn("Failed to flush response for cluster key {}, completing future anyway", (Object)clusterKey, (Object)e);
            }
            sessionCreationFuture.completeAsync(() -> newSessionId, CompletableFuture.delayedExecutor(500L, TimeUnit.MILLISECONDS));
            this.pendingSessionCreation.remove(clusterKey);
        }
    }

    private boolean waitForSessionCreation(HttpServletRequest request, String key, CompletableFuture<String> sessionCreationFuture) {
        this.getLogger().debug("Waiting for session creation for cluster key {} on request {} with requested session id {}", new Object[]{key, request.getRequestURI(), request.getRequestedSessionId()});
        try {
            String newSessionId = sessionCreationFuture.get(30L, TimeUnit.SECONDS);
            this.getLogger().debug("Session {} creation completed for cluster key {}, redirecting requested session {}", new Object[]{newSessionId, key, request.getRequestedSessionId()});
            return !newSessionId.equals(request.getRequestedSessionId());
        }
        catch (TimeoutException e) {
            this.getLogger().error("Timeout waiting for session creation for cluster key {}", (Object)key);
            this.pendingSessionCreation.remove(key);
            throw new RuntimeException("Timeout waiting for session creation for cluster key: " + key, e);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            this.getLogger().error("Interrupted while waiting for session creation for cluster key {}", (Object)key);
            throw new RuntimeException("Interrupted while waiting for session creation", e);
        }
        catch (ExecutionException e) {
            this.getLogger().error("Error during session creation for cluster key {}", (Object)key, (Object)e.getCause());
            throw new RuntimeException("Error during session creation", e.getCause());
        }
    }

    private HttpSession createOrWaitForSession(HttpServletRequest request, String key, CompletableFuture<String> existingFuture) {
        CurrentKey.set(key);
        this.getLogger().debug("Creating session for cluster key {} on request {} for requested session id {}", new Object[]{key, request.getRequestURI(), request.getRequestedSessionId()});
        try {
            HttpSession session = this.sessionSerializer.createHttpSession(key, request);
            if (session != null) {
                this.sessionListener.sessionAssociated(key, session, request);
                this.getLogger().debug("Session {} successfully created for cluster key {}", (Object)session.getId(), (Object)key);
            } else {
                this.getLogger().debug("Another request is already processing deserialization for cluster key {}", (Object)key);
            }
            return session;
        }
        catch (RuntimeException e) {
            this.getLogger().error("Failed to create session for cluster key {}", (Object)key, (Object)e);
            this.pendingSessionCreation.remove(key);
            existingFuture.completeExceptionally(e);
            throw e;
        }
    }

    public void destroy() {
        if (this.destroyCallback != null) {
            this.destroyCallback.run();
        }
    }

    private Consumer<Cookie> cookieConsumer(HttpServletRequest request) {
        return cookie -> {
            cookie.setHttpOnly(true);
            String path = request.getContextPath().isEmpty() ? "/" : request.getContextPath();
            cookie.setPath(path);
            SameSite sameSite = this.properties.getClusterKeyCookieSameSite();
            if (sameSite != null && !sameSite.attributeValue().isEmpty()) {
                cookie.setAttribute("SameSite", sameSite.attributeValue());
            }
        };
    }

    private String buildRedirectUrl(HttpServletRequest request) {
        String host;
        String scheme = request.getHeader("X-Forwarded-Proto");
        if (scheme == null || scheme.isEmpty()) {
            scheme = request.getScheme();
        }
        if (((host = request.getHeader("X-Forwarded-Host")) == null || host.isEmpty()) && ((host = request.getHeader("Host")) == null || host.isEmpty())) {
            host = request.getServerName();
        }
        int port = -1;
        String portHeader = request.getHeader("X-Forwarded-Port");
        if (portHeader != null && !portHeader.isEmpty()) {
            try {
                port = Integer.parseInt(portHeader);
            }
            catch (NumberFormatException numberFormatException) {
                // empty catch block
            }
        }
        if (port == -1) {
            port = request.getServerPort();
        }
        UriComponentsBuilder builder = UriComponentsBuilder.newInstance().scheme(scheme).host(host).port(port).path(request.getRequestURI());
        String queryString = request.getQueryString();
        if (queryString != null && !queryString.isEmpty()) {
            builder.query(queryString);
        }
        return builder.build().toUriString();
    }

    private Logger getLogger() {
        return LoggerFactory.getLogger(((Object)((Object)this)).getClass());
    }
}

