/*-
 * Copyright (C) 2022 Vaadin Ltd
 *
 * This program is available under Vaadin Commercial License and Service Terms.
 *
 *
 * See <https://vaadin.com/commercial-license-and-service-terms> for the full
 * license.
 */
package com.vaadin.kubernetes.starter;

import java.util.HashSet;
import java.util.Set;
import java.util.function.Predicate;

import org.springframework.boot.context.properties.ConfigurationProperties;
import org.springframework.boot.context.properties.NestedConfigurationProperty;

import static com.vaadin.kubernetes.starter.SerializationProperties.PREFIX;

/**
 * Definition of configuration properties for Session serialization.
 *
 * @author Vaadin Ltd
 * @since 1.0
 */
@ConfigurationProperties(prefix = PREFIX)
public class SerializationProperties {

    public static final String PREFIX = "vaadin.serialization";

    public static final int DEFAULT_SERIALIZATION_TIMEOUT_MS = 30000;

    public static final int DEFAULT_DESERIALIZATION_LOCK_TIMEOUT_MS = 10000;

    public static final int DEFAULT_OPTIMISTIC_SERIALIZATION_TIMEOUT_MS = 30000;

    public static final int DEFAULT_OPTIMISTIC_SERIALIZATION_DELAY_MS = 10;

    private int timeout = DEFAULT_SERIALIZATION_TIMEOUT_MS;

    private int deserializationLockTimeout = DEFAULT_DESERIALIZATION_LOCK_TIMEOUT_MS;

    private int optimisticTimeout = DEFAULT_OPTIMISTIC_SERIALIZATION_TIMEOUT_MS;

    private int optimisticDelay = DEFAULT_OPTIMISTIC_SERIALIZATION_DELAY_MS;

    @NestedConfigurationProperty
    private final TransientsProperties transients = new TransientsProperties();

    /**
     * Gets the timeout in milliseconds to wait for the serialization to be
     * completed.
     *
     * @return the timeout in milliseconds to wait for the serialization to be
     *         completed, defaults to 30000 ms
     */
    public int getTimeout() {
        return timeout;
    }

    /**
     * Sets the timeout in milliseconds to wait for the serialization to be
     * completed.
     *
     * @param timeout
     *            the timeout in milliseconds to wait for the serialization to
     *            be completed, defaults to 30000 ms
     */
    public void setTimeout(int timeout) {
        this.timeout = timeout;
    }

    /**
     * Gets the timeout in milliseconds to wait for the deserialization lock to
     * be released if the deserialization is not completed.
     *
     * @return the timeout in milliseconds to wait for the deserialization to be
     *         completed, defaults to 10000 ms
     */
    public int getDeserializationLockTimeout() {
        return deserializationLockTimeout;
    }

    /**
     * Sets the timeout in milliseconds to wait for the deserialization lock to
     * be released if the deserialization is not completed.
     *
     * @param timeout
     *            the timeout in milliseconds to wait for the deserialization to
     *            be completed, defaults to 10000 ms
     */
    public void setDeserializationLockTimeout(int timeout) {
        this.deserializationLockTimeout = timeout;
    }

    /**
     * Gets the timeout in milliseconds to wait for the optimistic serialization
     * to be completed.
     * <p>
     * 0 or negative value means that the optimistic serialization is disabled
     * and only the pessimistic serialization is performed. Pessimistic
     * serialization locks the Vaadin session during the serialization process.
     * Disabling the optimistic serialization may affect the UI performance.
     *
     * @return the timeout in milliseconds to wait for the optimistic
     *         serialization to be completed, defaults to 30000 ms
     */
    public int getOptimisticTimeout() {
        return optimisticTimeout;
    }

    /**
     * Sets the timeout in milliseconds to wait for the optimistic serialization
     * to be completed.
     * <p>
     * 0 or negative value means that the optimistic serialization is disabled
     * and only the pessimistic serialization is performed. Pessimistic
     * serialization locks the Vaadin session during the serialization process.
     * Disabling the optimistic serialization may affect the UI performance.
     *
     * @param optimisticTimeout
     *            the timeout in milliseconds to wait for the optimistic
     *            serialization to be completed, defaults to 30000 ms
     */
    public void setOptimisticTimeout(int optimisticTimeout) {
        this.optimisticTimeout = optimisticTimeout;
    }

    /**
     * Gets the delay in milliseconds to wait between optimistic serialization
     * attempts.
     * <p>
     * A value of 0 or negative means no delay is applied between attempts. Note
     * that disabling the delay may increase CPU usage significantly.
     *
     * @return the delay in milliseconds between optimistic serialization
     *         attempts (default: 10)
     */
    public int getOptimisticDelay() {
        return optimisticDelay;
    }

    /**
     * Sets the delay in milliseconds to wait between optimistic serialization
     * attempts.
     * <p>
     * A value of 0 or negative means no delay is applied between attempts. Note
     * that disabling the delay may increase CPU usage significantly.
     *
     * @param delay
     *            the delay in milliseconds between optimistic serialization
     *            attempts (default: 10)
     */
    public void setOptimisticDelay(int delay) {
        this.optimisticDelay = delay;
    }

    /**
     * Gets configuration for transient fields handling during serialization.
     *
     * @return configuration for transient fields handling.
     */
    public TransientsProperties getTransients() {
        return transients;
    }

    static class TransientsProperties {
        private final Set<String> includePackages = new HashSet<>();
        private final Set<String> excludePackages = new HashSet<>();

        /**
         * Gets a set of packages to consider during class inspection for
         * injectable transient fields.
         *
         * @return set of packages included in class inspection.
         */
        public Set<String> getIncludePackages() {
            return includePackages;
        }

        /**
         * Gets a set of packages to exclude from class inspection for
         * injectable transient fields.
         *
         * @return set of packages excluded from class inspection.
         */
        public Set<String> getExcludePackages() {
            return excludePackages;
        }

        /**
         * Gets a predicate that filters classes based on include/exclude
         * packages configuration.
         * <p>
         * An empty inclusion list means all classes are included. Exclusion
         * rules have higher priority over inclusion rules.
         * <p>
         * If no inclusion nor exclusion rules are configured all class are
         * eligible for inspection.
         *
         * @return a predicate that filter classes based on configured package
         *         rules.
         */
        public Predicate<Class<?>> transientInjectableFilter() {
            if (includePackages.isEmpty() && excludePackages.isEmpty()) {
                return type -> true;
            }
            return type -> {
                String packageName = type.getPackageName();
                return excludePackages.stream()
                        .noneMatch(packageName::startsWith)
                        && (includePackages.isEmpty() || includePackages
                                .stream().anyMatch(packageName::startsWith));
            };
        }
    }
}
