/*
 * Decompiled with CFR 0.152.
 */
package com.vaadin.kubernetes.starter.sessiontracker;

import com.vaadin.kubernetes.starter.sessiontracker.CurrentKey;
import com.vaadin.kubernetes.starter.sessiontracker.SessionSerializer;
import com.vaadin.kubernetes.starter.sessiontracker.SessionTrackerCookie;
import com.vaadin.kubernetes.starter.sessiontracker.backend.BackendConnector;
import com.vaadin.kubernetes.starter.sessiontracker.backend.SessionInfo;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpSession;
import jakarta.servlet.http.HttpSessionAttributeListener;
import jakarta.servlet.http.HttpSessionBindingEvent;
import jakarta.servlet.http.HttpSessionEvent;
import jakarta.servlet.http.HttpSessionIdListener;
import jakarta.servlet.http.HttpSessionListener;
import java.util.HashSet;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Predicate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SessionListener
implements HttpSessionListener,
HttpSessionIdListener,
HttpSessionAttributeListener {
    private final BackendConnector sessionBackendConnector;
    private final SessionSerializer sessionSerializer;
    private final Map<String, String> activeSessions = new ConcurrentHashMap<String, String>();
    private final Map<String, SessionCreationRequest> sessionCreationRequestMap = new ConcurrentHashMap<String, SessionCreationRequest>();
    private boolean stopped = false;

    public SessionListener(BackendConnector sessionBackendConnector, SessionSerializer sessionSerializer) {
        this.sessionBackendConnector = sessionBackendConnector;
        this.sessionSerializer = sessionSerializer;
    }

    public void sessionCreated(HttpSessionEvent se) {
        if (this.stopped) {
            return;
        }
        HttpSession session = se.getSession();
        SessionListener.getLogger().debug("Session with id {} created", (Object)session.getId());
        String clusterKey = CurrentKey.get();
        if (clusterKey != null) {
            this.activeSessions.put(session.getId(), clusterKey);
            SessionInfo sessionInfo = this.sessionBackendConnector.getSession(clusterKey);
            if (sessionInfo != null) {
                SessionListener.getLogger().debug("Found session {} on distributed storage", (Object)clusterKey);
                try {
                    this.sessionSerializer.deserialize(sessionInfo, session);
                    SessionListener.getLogger().debug("HTTP session {} populated with data from {} key of distributed storage", (Object)session.getId(), (Object)clusterKey);
                }
                catch (Exception e) {
                    SessionListener.getLogger().error("Unable to deserialize data with key {} from distributed storage into session {}", new Object[]{clusterKey, session.getId(), e});
                }
            }
            this.sessionCreationRequestMap.put(clusterKey, new SessionCreationRequest(session.getId(), new HashSet<String>()));
        } else {
            this.activeSessions.put(session.getId(), "");
        }
    }

    public void sessionDestroyed(HttpSessionEvent se) {
        if (this.stopped) {
            return;
        }
        HttpSession session = se.getSession();
        String sessionId = session.getId();
        String associatedClusterKey = this.activeSessions.remove(sessionId);
        if (associatedClusterKey != null) {
            this.sessionCreationRequestMap.remove(associatedClusterKey);
        }
        SessionListener.getLogger().debug("Session with id {} destroyed", (Object)sessionId);
        SessionTrackerCookie.getFromSession(session).ifPresent(clusterKey -> {
            this.sessionCreationRequestMap.remove(clusterKey);
            SessionListener.getLogger().debug("Deleting data with key {} from distributed storage associated to session {}", clusterKey, (Object)sessionId);
            try {
                this.sessionBackendConnector.deleteSession((String)clusterKey);
                SessionListener.getLogger().debug("Deleted data with key {} from distributed storage associated to session {}", clusterKey, (Object)sessionId);
            }
            catch (Exception e) {
                SessionListener.getLogger().error("Unable to delete data with key {} from distributed storage associated to session {}", new Object[]{clusterKey, sessionId, e});
            }
        });
    }

    public void sessionIdChanged(HttpSessionEvent event, String oldSessionId) {
        String newSessionId = event.getSession().getId();
        this.sessionCreationRequestMap.replaceAll((clusterKey, current) -> {
            if (oldSessionId.equals(current.sessionId())) {
                return new SessionCreationRequest(newSessionId, current.requestedSessionId());
            }
            return current;
        });
        String clusterKey2 = this.activeSessions.remove(oldSessionId);
        this.activeSessions.put(newSessionId, clusterKey2);
    }

    public void attributeAdded(HttpSessionBindingEvent event) {
        this.updateClusterKeyToSessionAssociation(event);
    }

    public void attributeReplaced(HttpSessionBindingEvent event) {
        this.updateClusterKeyToSessionAssociation(event);
    }

    private void updateClusterKeyToSessionAssociation(HttpSessionBindingEvent event) {
        if ("clusterKey".equals(event.getName())) {
            String clusterKey = (String)event.getValue();
            String sessionId = event.getSession().getId();
            SessionListener.getLogger().debug("Associating cluster key {} to session {}", (Object)clusterKey, (Object)sessionId);
            this.updateOrCreateSessionCreationRequest(clusterKey, sessionId);
        }
    }

    private SessionCreationRequest updateOrCreateSessionCreationRequest(String clusterKey, String sessionId) {
        return this.sessionCreationRequestMap.compute(clusterKey, (unused, current) -> {
            if (current == null) {
                return new SessionCreationRequest(sessionId, new HashSet<String>());
            }
            return new SessionCreationRequest(sessionId, current.requestedSessionId());
        });
    }

    Optional<String> findExistingSession(String clusterKey, HttpServletRequest request) {
        String requestedSessionId = request.getRequestedSessionId();
        if (requestedSessionId == null) {
            return Optional.empty();
        }
        return Optional.ofNullable(this.sessionCreationRequestMap.get(clusterKey)).filter(req -> req.requestedSessionId().contains(requestedSessionId)).map(SessionCreationRequest::sessionId);
    }

    void sessionAssociated(String clusterKey, HttpSession session, HttpServletRequest request) {
        Objects.requireNonNull(clusterKey, "clusterKey must not be null");
        Objects.requireNonNull(session, "session must not be null");
        Objects.requireNonNull(request, "HTTP request must not be null");
        String requestedSessionId = request.getRequestedSessionId();
        String sessionId = session.getId();
        if (!clusterKey.equals(this.activeSessions.get(sessionId))) {
            throw new IllegalStateException("Session " + sessionId + " is not associated with cluster key " + clusterKey);
        }
        SessionListener.getLogger().debug("Session {} created for cluster key {} with requested session id {}", new Object[]{sessionId, clusterKey, requestedSessionId});
        if (sessionId != null) {
            this.updateOrCreateSessionCreationRequest(clusterKey, sessionId).requestedSessionId().add(requestedSessionId);
        }
    }

    public Predicate<String> activeSessionChecker() {
        return this.activeSessions::containsKey;
    }

    public void stop() {
        this.stopped = true;
    }

    static Logger getLogger() {
        return LoggerFactory.getLogger(SessionListener.class);
    }

    private record SessionCreationRequest(String sessionId, Set<String> requestedSessionId) {
    }
}

