/*
 * Decompiled with CFR 0.152.
 */
package com.vaadin.kubernetes.starter.sessiontracker;

import com.vaadin.flow.component.UI;
import com.vaadin.flow.server.VaadinSession;
import com.vaadin.flow.server.WrappedHttpSession;
import com.vaadin.flow.server.WrappedSession;
import com.vaadin.kubernetes.starter.ProductUtils;
import com.vaadin.kubernetes.starter.sessiontracker.PessimisticSerializationRequiredException;
import com.vaadin.kubernetes.starter.sessiontracker.SessionSerializationCallback;
import com.vaadin.kubernetes.starter.sessiontracker.UnserializableComponentWrapper;
import com.vaadin.kubernetes.starter.sessiontracker.UnserializableComponentWrapperFoundException;
import com.vaadin.kubernetes.starter.sessiontracker.backend.BackendConnector;
import com.vaadin.kubernetes.starter.sessiontracker.backend.SessionExpirationPolicy;
import com.vaadin.kubernetes.starter.sessiontracker.backend.SessionInfo;
import com.vaadin.kubernetes.starter.sessiontracker.serialization.SerializationInputStream;
import com.vaadin.kubernetes.starter.sessiontracker.serialization.SerializationOutputStream;
import com.vaadin.kubernetes.starter.sessiontracker.serialization.SerializationStreamFactory;
import com.vaadin.kubernetes.starter.sessiontracker.serialization.TransientHandler;
import jakarta.servlet.http.HttpSession;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.NotSerializableException;
import java.time.Duration;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.BiFunction;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.context.ApplicationListener;
import org.springframework.context.event.ContextClosedEvent;

public class SessionSerializer
implements ApplicationListener<ContextClosedEvent> {
    private static final long OPTIMISTIC_SERIALIZATION_TIMEOUT_MS = 30000L;
    private final ExecutorService executorService = Executors.newFixedThreadPool(4, new SerializationThreadFactory());
    private final ConcurrentHashMap<String, Boolean> pending = new ConcurrentHashMap();
    private final BackendConnector backendConnector;
    private final BiFunction<String, String, TransientHandler> handlerProvider;
    private final long optimisticSerializationTimeoutMs;
    private final SessionSerializationCallback sessionSerializationCallback;
    private final SessionExpirationPolicy sessionExpirationPolicy;
    private final SerializationStreamFactory serializationStreamFactory;
    private Predicate<Class<?>> injectableFilter = type -> true;

    public SessionSerializer(BackendConnector backendConnector, TransientHandler transientHandler, SessionExpirationPolicy sessionExpirationPolicy, SessionSerializationCallback sessionSerializationCallback, SerializationStreamFactory serializationStreamFactory) {
        this(backendConnector, (String sessionId, String clusterKey) -> transientHandler, sessionExpirationPolicy, sessionSerializationCallback, serializationStreamFactory);
    }

    public SessionSerializer(BackendConnector backendConnector, BiFunction<String, String, TransientHandler> transientHandlerProvider, SessionExpirationPolicy sessionExpirationPolicy, SessionSerializationCallback sessionSerializationCallback, SerializationStreamFactory serializationStreamFactory) {
        this.backendConnector = backendConnector;
        this.handlerProvider = transientHandlerProvider;
        this.sessionSerializationCallback = sessionSerializationCallback;
        this.sessionExpirationPolicy = sessionExpirationPolicy;
        this.serializationStreamFactory = serializationStreamFactory;
        this.optimisticSerializationTimeoutMs = 30000L;
    }

    SessionSerializer(BackendConnector backendConnector, TransientHandler transientHandler, SessionExpirationPolicy sessionExpirationPolicy, SessionSerializationCallback sessionSerializationCallback, long optimisticSerializationTimeoutMs, SerializationStreamFactory serializationStreamFactory) {
        this.backendConnector = backendConnector;
        this.optimisticSerializationTimeoutMs = optimisticSerializationTimeoutMs;
        this.sessionSerializationCallback = sessionSerializationCallback;
        this.sessionExpirationPolicy = sessionExpirationPolicy;
        this.handlerProvider = (sessionId, clusterKey) -> transientHandler;
        this.serializationStreamFactory = serializationStreamFactory;
    }

    public void setInjectableFilter(Predicate<Class<?>> injectableFilter) {
        this.injectableFilter = injectableFilter;
    }

    public void serialize(HttpSession session) {
        this.serialize((WrappedSession)new WrappedHttpSession(session));
    }

    public void serialize(WrappedSession session) {
        Map<String, Object> values = session.getAttributeNames().stream().collect(Collectors.toMap(Function.identity(), arg_0 -> ((WrappedSession)session).getAttribute(arg_0)));
        Duration timeToLive = this.sessionExpirationPolicy.apply(session.getMaxInactiveInterval());
        this.queueSerialization(session.getId(), timeToLive, values);
    }

    public void deserialize(SessionInfo sessionInfo, HttpSession session) throws Exception {
        Map<String, Object> values = this.doDeserialize(sessionInfo, session.getId());
        for (Map.Entry<String, Object> entry : values.entrySet()) {
            session.setAttribute(entry.getKey(), entry.getValue());
        }
    }

    private void queueSerialization(String sessionId, Duration timeToLive, Map<String, Object> attributes) {
        if (this.pending.containsKey(sessionId)) {
            SessionSerializer.getLogger().debug("Ignoring serialization request for session {} as the session is already being serialized", (Object)sessionId);
            return;
        }
        String clusterKey = this.getClusterKey(attributes);
        SessionSerializer.getLogger().debug("Starting asynchronous serialization of session {} with distributed key {}", (Object)sessionId, (Object)clusterKey);
        this.pending.put(sessionId, true);
        ((CompletableFuture)CompletableFuture.runAsync(() -> this.backendConnector.markSerializationStarted(clusterKey, timeToLive), this.executorService).handle((unused, error) -> {
            if (error != null) {
                SessionSerializer.getLogger().debug("Failed marking serialization start for of session {} with distributed key {}", new Object[]{sessionId, clusterKey, error});
            } else {
                Consumer<SessionInfo> whenSerialized = sessionInfo -> {
                    if (sessionInfo != null) {
                        this.backendConnector.sendSession((SessionInfo)sessionInfo);
                    }
                    this.backendConnector.markSerializationComplete(clusterKey);
                };
                this.handleSessionSerialization(sessionId, timeToLive, attributes, whenSerialized);
            }
            return null;
        })).whenComplete((unused, error) -> {
            this.pending.remove(sessionId);
            if (error != null) {
                if (error instanceof CompletionException && error.getCause() != null) {
                    error = error.getCause();
                }
                this.backendConnector.markSerializationFailed(clusterKey, (Throwable)error);
                SessionSerializer.getLogger().error("Serialization of session {} failed", (Object)sessionId, error);
            }
        });
    }

    private void handleSessionSerialization(String sessionId, Duration timeToLive, Map<String, Object> attributes, Consumer<SessionInfo> whenSerialized) {
        boolean unrecoverableError = false;
        String clusterKey = this.getClusterKey(attributes);
        try {
            this.checkUnserializableWrappers(attributes);
            long start = System.currentTimeMillis();
            long timeout = start + this.optimisticSerializationTimeoutMs;
            SessionSerializer.getLogger().debug("Optimistic serialization of session {} with distributed key {} started", (Object)sessionId, (Object)clusterKey);
            while (System.currentTimeMillis() < timeout) {
                SessionInfo info = this.serializeOptimisticLocking(sessionId, timeToLive, attributes);
                if (info == null) continue;
                this.pending.remove(sessionId);
                SessionSerializer.getLogger().debug("Optimistic serialization of session {} with distributed key {} completed", (Object)sessionId, (Object)clusterKey);
                whenSerialized.accept(info);
                return;
            }
        }
        catch (PessimisticSerializationRequiredException e) {
            if (e instanceof UnserializableComponentWrapperFoundException) {
                SessionSerializer.getLogger().debug(e.getMessage());
            } else {
                SessionSerializer.getLogger().warn("Optimistic serialization of session {} with distributed key {} cannot be completed because VaadinSession lock is required. Switching to pessimistic locking.", new Object[]{sessionId, clusterKey, e});
            }
        }
        catch (NotSerializableException e) {
            SessionSerializer.getLogger().error("Optimistic serialization of session {} with distributed key {} failed, some attribute is not serializable. Giving up immediately since the error is not recoverable", new Object[]{sessionId, clusterKey, e});
            unrecoverableError = true;
        }
        catch (IOException e) {
            SessionSerializer.getLogger().warn("Optimistic serialization of session {} with distributed key {} failed", new Object[]{sessionId, clusterKey, e});
            unrecoverableError = true;
        }
        this.pending.remove(sessionId);
        SessionInfo sessionInfo = null;
        if (!unrecoverableError) {
            sessionInfo = this.serializePessimisticLocking(sessionId, timeToLive, attributes);
        }
        whenSerialized.accept(sessionInfo);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    private SessionInfo serializePessimisticLocking(String sessionId, Duration timeToLive, Map<String, Object> attributes) {
        SessionInfo sessionInfo;
        long start = System.currentTimeMillis();
        String clusterKey = this.getClusterKey(attributes);
        Set<ReentrantLock> locks = this.getLocks(attributes);
        for (ReentrantLock reentrantLock : locks) {
            reentrantLock.lock();
        }
        try {
            this.beforeSerializePessimistic(attributes);
            sessionInfo = this.doSerialize(sessionId, timeToLive, attributes);
            this.afterSerializePessimistic(attributes);
        }
        catch (Exception e) {
            try {
                SessionSerializer.getLogger().error("An error occurred during pessimistic serialization of session {} with distributed key {} ", new Object[]{sessionId, clusterKey, e});
                this.afterSerializePessimistic(attributes);
            }
            catch (Throwable throwable) {
                this.afterSerializePessimistic(attributes);
                for (ReentrantLock lock : locks) {
                    lock.unlock();
                }
                SessionSerializer.getLogger().debug("Pessimistic serialization of session {} with distributed key {} completed in {}ms", new Object[]{sessionId, clusterKey, System.currentTimeMillis() - start});
                throw throwable;
            }
            for (ReentrantLock reentrantLock : locks) {
                reentrantLock.unlock();
            }
            SessionSerializer.getLogger().debug("Pessimistic serialization of session {} with distributed key {} completed in {}ms", new Object[]{sessionId, clusterKey, System.currentTimeMillis() - start});
            return null;
        }
        for (ReentrantLock lock : locks) {
            lock.unlock();
        }
        SessionSerializer.getLogger().debug("Pessimistic serialization of session {} with distributed key {} completed in {}ms", new Object[]{sessionId, clusterKey, System.currentTimeMillis() - start});
        return sessionInfo;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void checkUnserializableWrappers(Map<String, Object> attributes) {
        Consumer<UnserializableComponentWrapper> action = c -> {
            throw new UnserializableComponentWrapperFoundException("Pessimistic serialization required because at least one " + UnserializableComponentWrapper.class.getName() + " is in the UI tree");
        };
        Set<ReentrantLock> locks = this.getLocks(attributes);
        for (ReentrantLock lock : locks) {
            lock.lock();
        }
        try {
            this.getUIs(attributes).forEach(ui -> UnserializableComponentWrapper.doWithWrapper(ui, action));
        }
        finally {
            for (ReentrantLock lock : locks) {
                lock.unlock();
            }
        }
    }

    private void beforeSerializePessimistic(Map<String, Object> attributes) {
        this.getUIs(attributes).forEach(UnserializableComponentWrapper::beforeSerialization);
    }

    private void afterSerializePessimistic(Map<String, Object> attributes) {
        this.getUIs(attributes).forEach(UnserializableComponentWrapper::afterSerialization);
    }

    private List<UI> getUIs(Map<String, Object> attributes) {
        return attributes.values().stream().filter(o -> o instanceof VaadinSession).map(VaadinSession.class::cast).flatMap(s -> s.getUIs().stream()).toList();
    }

    private Set<ReentrantLock> getLocks(Map<String, Object> attributes) {
        HashSet<ReentrantLock> locks = new HashSet<ReentrantLock>();
        for (String key : attributes.keySet()) {
            String serviceName;
            String lockKey;
            Object lockAttribute;
            if (!key.startsWith("com.vaadin.flow.server.VaadinSession") || !((lockAttribute = attributes.get(lockKey = (serviceName = key.substring("com.vaadin.flow.server.VaadinSession".length() + 1)) + ".lock")) instanceof ReentrantLock)) continue;
            ReentrantLock lock = (ReentrantLock)lockAttribute;
            locks.add(lock);
        }
        return locks;
    }

    private SessionInfo serializeOptimisticLocking(String sessionId, Duration timeToLive, Map<String, Object> attributes) throws IOException {
        String clusterKey = this.getClusterKey(attributes);
        try {
            long latestLockTime = this.findNewestLockTime(attributes);
            long latestUnlockTime = this.findNewestUnlockTime(attributes);
            if (latestLockTime > latestUnlockTime) {
                SessionSerializer.getLogger().trace("Optimistic serialization of session {} with distributed key {} failed, session is locked. Will retry", (Object)sessionId, (Object)clusterKey);
                return null;
            }
            SessionInfo info = this.doSerialize(sessionId, timeToLive, attributes);
            long latestUnlockTimeCheck = this.findNewestUnlockTime(attributes);
            if (latestUnlockTime != latestUnlockTimeCheck) {
                SessionSerializer.getLogger().trace("Optimistic serialization of session {} with distributed key {} failed, somebody modified the session during serialization ({} != {}). Will retry", new Object[]{sessionId, clusterKey, latestUnlockTime, latestUnlockTimeCheck});
                return null;
            }
            this.logSessionDebugInfo("Serialized session " + sessionId + " with distributed key " + clusterKey, attributes);
            return info;
        }
        catch (PessimisticSerializationRequiredException | NotSerializableException e) {
            throw e;
        }
        catch (Exception e) {
            SessionSerializer.getLogger().trace("Optimistic serialization of session {} with distributed key {} failed, a problem occurred during serialization. Will retry", new Object[]{sessionId, clusterKey, e});
            return null;
        }
    }

    private void logSessionDebugInfo(String prefix, Map<String, Object> attributes) {
        StringBuilder info = new StringBuilder();
        for (String key : attributes.keySet()) {
            Object value = attributes.get(key);
            if (!(value instanceof VaadinSession)) continue;
            VaadinSession session = (VaadinSession)value;
            try {
                for (UI ui : session.getUIs()) {
                    info.append("[UI ").append(ui.getUIId()).append(", last client message: ").append(ui.getInternals().getLastProcessedClientToServerId()).append(", server sync id: ").append(ui.getInternals().getServerSyncId()).append("]");
                }
            }
            catch (Exception ex) {
                info.append("[ VaadinSession not accessible without locking ]");
            }
        }
        SessionSerializer.getLogger().trace("{} UIs: {}", (Object)prefix, (Object)info);
    }

    private long findNewestLockTime(Map<String, Object> attributes) {
        long latestLock = 0L;
        for (Map.Entry<String, Object> entry : attributes.entrySet()) {
            Object object = entry.getValue();
            if (!(object instanceof VaadinSession)) continue;
            VaadinSession session = (VaadinSession)object;
            latestLock = Math.max(latestLock, session.getLastLocked());
        }
        return latestLock;
    }

    private long findNewestUnlockTime(Map<String, Object> attributes) {
        long latestUnlock = 0L;
        for (Map.Entry<String, Object> entry : attributes.entrySet()) {
            Object object = entry.getValue();
            if (!(object instanceof VaadinSession)) continue;
            VaadinSession session = (VaadinSession)object;
            latestUnlock = Math.max(latestUnlock, session.getLastUnlocked());
        }
        return latestUnlock;
    }

    private SessionInfo doSerialize(String sessionId, Duration timeToLive, Map<String, Object> attributes) throws Exception {
        long start = System.currentTimeMillis();
        String clusterKey = this.getClusterKey(attributes);
        ByteArrayOutputStream out = new ByteArrayOutputStream();
        TransientHandler transientHandler = this.handlerProvider.apply(sessionId, clusterKey);
        try (SerializationOutputStream outStream = this.serializationStreamFactory.createOutputStream(out, transientHandler, this.injectableFilter);){
            outStream.writeWithTransients(attributes);
            this.sessionSerializationCallback.onSerializationSuccess();
        }
        catch (Exception ex) {
            this.sessionSerializationCallback.onSerializationError(ex);
            throw ex;
        }
        SessionInfo info = new SessionInfo(clusterKey, timeToLive, out.toByteArray());
        SessionSerializer.getLogger().debug("Serialization of attributes {} for session {} with distributed key {} completed in {}ms ({} bytes)", new Object[]{attributes.keySet(), sessionId, info.getClusterKey(), System.currentTimeMillis() - start, info.getData().length});
        return info;
    }

    private String getClusterKey(Map<String, Object> attributes) {
        return (String)attributes.get("clusterKey");
    }

    private Map<String, Object> doDeserialize(SessionInfo sessionInfo, String sessionId) throws Exception {
        Map attributes;
        byte[] data = sessionInfo.getData();
        long start = System.currentTimeMillis();
        ClassLoader contextLoader = Thread.currentThread().getContextClassLoader();
        ByteArrayInputStream in = new ByteArrayInputStream(data);
        TransientHandler transientHandler = this.handlerProvider.apply(sessionId, sessionInfo.getClusterKey());
        try (SerializationInputStream inStream = this.serializationStreamFactory.createInputStream(in, transientHandler);){
            attributes = (Map)inStream.readWithTransients();
            this.sessionSerializationCallback.onDeserializationSuccess();
        }
        catch (Exception ex) {
            this.sessionSerializationCallback.onDeserializationError(ex);
            throw ex;
        }
        finally {
            Thread.currentThread().setContextClassLoader(contextLoader);
        }
        this.logSessionDebugInfo("Deserialized session", attributes);
        SessionSerializer.getLogger().debug("Deserialization of attributes {} for session {} with distributed key {} completed in {}ms", new Object[]{attributes.keySet(), sessionId, sessionInfo.getClusterKey(), System.currentTimeMillis() - start});
        return attributes;
    }

    private static Logger getLogger() {
        return LoggerFactory.getLogger(SessionSerializer.class);
    }

    void waitForSerialization() {
        long lastLogTime = System.currentTimeMillis();
        while (!this.pending.isEmpty()) {
            long now = System.currentTimeMillis();
            if (now - lastLogTime >= 5000L) {
                SessionSerializer.getLogger().info("Waiting for {} sessions to be serialized: {}", (Object)this.pending.size(), (Object)this.pending.keySet());
                lastLogTime = now;
            }
            try {
                Thread.sleep(100L);
            }
            catch (InterruptedException interruptedException) {}
        }
    }

    public void onApplicationEvent(ContextClosedEvent event) {
        this.waitForSerialization();
        this.executorService.shutdown();
    }

    static {
        ProductUtils.markAsUsed(SessionSerializer.class.getSimpleName());
    }

    private static class SerializationThreadFactory
    implements ThreadFactory {
        private final AtomicInteger threadNumber = new AtomicInteger(1);

        private SerializationThreadFactory() {
        }

        @Override
        public Thread newThread(Runnable runnable) {
            return new Thread(runnable, "sessionSerializer-worker-" + this.hashCode() + "-" + this.threadNumber.getAndIncrement());
        }
    }
}

