/*
 * Copyright 2000-2025 Vaadin Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package com.vaadin.hilla.engine;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;

import org.jspecify.annotations.NonNull;

import com.vaadin.hilla.parser.utils.ConfigList;

public final class GeneratorConfiguration {
    private Plugins plugins;

    public Optional<Plugins> getPlugins() {
        return Optional.ofNullable(plugins);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        var that = (GeneratorConfiguration) o;
        return Objects.equals(plugins, that.plugins);
    }

    @Override
    public int hashCode() {
        return Objects.hash(plugins);
    }

    void setPlugins(Plugins plugins) {
        this.plugins = plugins;
    }

    public static class Plugin {
        private String path;

        // Maven and Jackson requires a constructor for deserialization
        public Plugin() {
        }

        Plugin(String path) {
            this.path = path;
        }

        @Override
        public boolean equals(Object other) {
            if (this == other) {
                return true;
            }

            if (!(other instanceof Plugin)) {
                return false;
            }

            return Objects.equals(path, ((Plugin) other).path);
        }

        public String getPath() {
            return path;
        }

        @Override
        public int hashCode() {
            return path.hashCode();
        }
    }

    public static class Plugins implements ConfigList<Plugin> {
        private final List<Plugin> disable = new ArrayList<>();
        private final boolean disableAllDefaults;
        private final List<Plugin> use = new ArrayList<>();

        // Maven (and Jackson) require a constructor for deserialization
        public Plugins() {
            disableAllDefaults = false;
        }

        Plugins(@NonNull Collection<Plugin> use,
                @NonNull Collection<Plugin> disable,
                boolean disableAllDefaults) {
            this.use.addAll(use);
            this.disable.addAll(disable);
            this.disableAllDefaults = disableAllDefaults;
        }

        public List<Plugin> getDisable() {
            return disable;
        }

        public List<Plugin> getUse() {
            return use;
        }

        public boolean isDisableAllDefaults() {
            return disableAllDefaults;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            var plugins = (Plugins) o;
            return disableAllDefaults == plugins.disableAllDefaults
                    && Objects.equals(disable, plugins.disable)
                    && Objects.equals(use, plugins.use);
        }

        @Override
        public int hashCode() {
            return Objects.hash(disable, disableAllDefaults, use);
        }

        @Override
        public Collection<Plugin> getDisabledOptions() {
            return disable;
        }

        @Override
        public Collection<Plugin> getUsedOptions() {
            return use;
        }

        @Override
        public boolean shouldAllDefaultsBeDisabled() {
            return disableAllDefaults;
        }
    }

    static class PluginsProcessor extends ConfigList.Processor<Plugin> {
        static private final List<Plugin> DEFAULTS = List.of(
                new Plugin("@vaadin/hilla-generator-plugin-transfertypes"),
                new Plugin("@vaadin/hilla-generator-plugin-backbone"),
                new Plugin("@vaadin/hilla-generator-plugin-client"),
                new Plugin("@vaadin/hilla-generator-plugin-barrel"),
                new Plugin("@vaadin/hilla-generator-plugin-model"),
                new Plugin("@vaadin/hilla-generator-plugin-push"),
                new Plugin("@vaadin/hilla-generator-plugin-signals"),
                new Plugin("@vaadin/hilla-generator-plugin-subtypes"));

        PluginsProcessor() {
            super(DEFAULTS);
        }
    }
}
