/*
 * Decompiled with CFR 0.152.
 */
package com.vaadin.flow.server;

import com.vaadin.flow.WarURLStreamHandlerFactory;
import com.vaadin.flow.function.DeploymentConfiguration;
import com.vaadin.flow.internal.CurrentInstance;
import com.vaadin.flow.internal.ResponseWriter;
import com.vaadin.flow.server.HandlerHelper;
import com.vaadin.flow.server.HttpStatusCode;
import com.vaadin.flow.server.Mode;
import com.vaadin.flow.server.StaticFileServer;
import com.vaadin.flow.server.VaadinService;
import com.vaadin.flow.server.VaadinServletService;
import com.vaadin.tests.util.TestUtil;
import jakarta.servlet.ServletContext;
import jakarta.servlet.ServletOutputStream;
import jakarta.servlet.WriteListener;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpServletResponse;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.Serializable;
import java.lang.reflect.Field;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.net.URLConnection;
import java.net.URLStreamHandler;
import java.nio.charset.StandardCharsets;
import java.nio.file.FileSystem;
import java.nio.file.FileSystemNotFoundException;
import java.nio.file.FileSystems;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.time.LocalDateTime;
import java.time.ZoneOffset;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;
import net.jcip.annotations.NotThreadSafe;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.BeforeAll;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.io.TempDir;
import org.mockito.ArgumentMatchers;
import org.mockito.Mockito;

@NotThreadSafe
class StaticFileServerTest
implements Serializable {
    private OverrideableStaticFileServer fileServer;
    private HttpServletRequest request;
    private HttpServletResponse response;
    private Map<String, String> headers;
    private Map<String, Long> dateHeaders;
    private AtomicInteger responseCode;
    private AtomicLong responseContentLength;
    private VaadinServletService servletService = (VaadinServletService)Mockito.mock(VaadinServletService.class);
    private DeploymentConfiguration configuration;
    private ServletContext servletContext;
    private static final String WEBAPP_RESOURCE_PREFIX = "META-INF/VAADIN/webapp";
    private CapturingServletOutputStream out;
    @TempDir
    Path temporaryFolder;

    StaticFileServerTest() {
    }

    private static URL createFileURLWithDataAndLength(String name, String data) throws MalformedURLException {
        return StaticFileServerTest.createFileURLWithDataAndLength(name, data, -1L);
    }

    private static URL createFileURLWithDataAndLength(String name, final String data, final long lastModificationTime) throws MalformedURLException {
        return new URL("file", "", -1, name, new URLStreamHandler(){

            @Override
            protected URLConnection openConnection(URL u) throws IOException {
                byte[] bytes = data.getBytes(StandardCharsets.UTF_8);
                URLConnection connection = (URLConnection)Mockito.mock(URLConnection.class);
                Mockito.when((Object)connection.getInputStream()).thenReturn((Object)new ByteArrayInputStream(bytes));
                Mockito.when((Object)connection.getContentLengthLong()).thenReturn((Object)bytes.length);
                Mockito.when((Object)connection.getLastModified()).thenReturn((Object)lastModificationTime);
                return connection;
            }
        });
    }

    @BeforeAll
    public static void beforeClass() {
        CurrentInstance.clearAll();
    }

    @BeforeEach
    public void setUp() throws IOException {
        this.request = (HttpServletRequest)Mockito.mock(HttpServletRequest.class);
        this.response = (HttpServletResponse)Mockito.mock(HttpServletResponse.class);
        Mockito.when((Object)this.request.getDateHeader(ArgumentMatchers.anyString())).thenReturn((Object)-1L);
        this.responseCode = new AtomicInteger(-1);
        this.responseContentLength = new AtomicLong(-1L);
        this.headers = new HashMap<String, String>();
        this.dateHeaders = new HashMap<String, Long>();
        ((HttpServletResponse)Mockito.doAnswer(invocation -> {
            this.headers.put((String)invocation.getArguments()[0], (String)invocation.getArguments()[1]);
            return null;
        }).when((Object)this.response)).setHeader(ArgumentMatchers.anyString(), ArgumentMatchers.anyString());
        ((HttpServletResponse)Mockito.doAnswer(invocation -> {
            this.dateHeaders.put((String)invocation.getArguments()[0], (Long)invocation.getArguments()[1]);
            return null;
        }).when((Object)this.response)).setDateHeader(ArgumentMatchers.anyString(), ArgumentMatchers.anyLong());
        ((HttpServletResponse)Mockito.doAnswer(invocation -> {
            this.responseCode.set((Integer)invocation.getArguments()[0]);
            return null;
        }).when((Object)this.response)).setStatus(ArgumentMatchers.anyInt());
        ((HttpServletResponse)Mockito.doAnswer(invocation -> {
            this.responseCode.set((Integer)invocation.getArguments()[0]);
            return null;
        }).when((Object)this.response)).sendError(ArgumentMatchers.anyInt());
        ((HttpServletResponse)Mockito.doAnswer(invocation -> {
            this.responseContentLength.set((Long)invocation.getArguments()[0]);
            return null;
        }).when((Object)this.response)).setContentLengthLong(ArgumentMatchers.anyLong());
        this.configuration = (DeploymentConfiguration)Mockito.mock(DeploymentConfiguration.class);
        Mockito.when((Object)this.configuration.isProductionMode()).thenReturn((Object)true);
        Mockito.when((Object)this.configuration.getMode()).thenAnswer(q -> {
            if (this.configuration.isProductionMode()) {
                return Mode.PRODUCTION_CUSTOM;
            }
            return Mode.DEVELOPMENT_BUNDLE;
        });
        Mockito.when((Object)this.servletService.getDeploymentConfiguration()).thenReturn((Object)this.configuration);
        ((VaadinServletService)Mockito.doAnswer(invocationOnMock -> this.servletService.getClass().getClassLoader()).when((Object)this.servletService)).getClassLoader();
        this.fileServer = new OverrideableStaticFileServer(this.servletService, this.configuration);
        this.servletContext = (ServletContext)Mockito.mock(ServletContext.class);
        Mockito.when((Object)this.request.getServletContext()).thenReturn((Object)this.servletContext);
        this.out = new CapturingServletOutputStream();
        Mockito.when((Object)this.response.getOutputStream()).thenReturn((Object)this.out);
    }

    @Test
    public void getRequestFilename() {
        for (String contextPath : new String[]{"", "/foo", "/foo/bar"}) {
            Assertions.assertEquals((Object)"", (Object)this.getRequestFilename(contextPath, "", null));
            Assertions.assertEquals((Object)"/bar.js", (Object)this.getRequestFilename(contextPath, "", "/bar.js"));
            Assertions.assertEquals((Object)"/foo/bar.js", (Object)this.getRequestFilename(contextPath, "", "/foo/bar.js"));
            Assertions.assertEquals((Object)"/foo", (Object)this.getRequestFilename(contextPath, "/foo", null));
            Assertions.assertEquals((Object)"/foo/bar.js", (Object)this.getRequestFilename(contextPath, "/foo", "/bar.js"));
            Assertions.assertEquals((Object)"/foo/bar/baz.js", (Object)this.getRequestFilename(contextPath, "/foo", "/bar/baz.js"));
            Assertions.assertEquals((Object)"/foo/bar", (Object)this.getRequestFilename(contextPath, "/foo/bar", null));
            Assertions.assertEquals((Object)"/foo/bar/baz.js", (Object)this.getRequestFilename(contextPath, "/foo/bar", "/baz.js"));
            Assertions.assertEquals((Object)"/foo/bar/baz/baz.js", (Object)this.getRequestFilename(contextPath, "/foo/bar", "/baz/baz.js"));
        }
    }

    @Test
    public void getRequestFilename_shouldAlwaysBeResolvedAsRootResourceForServiceWorkerRequest() {
        for (String swFile : new String[]{"/sw.js", "/sw.js.gz"}) {
            Assertions.assertEquals((Object)swFile, (Object)this.getRequestFilename("", "", swFile));
            Assertions.assertEquals((Object)swFile, (Object)this.getRequestFilename("", "/foo", swFile));
            Assertions.assertEquals((Object)swFile, (Object)this.getRequestFilename("", "/foo/bar", swFile));
            Assertions.assertEquals((Object)swFile, (Object)this.getRequestFilename("/ctx", "", swFile));
            Assertions.assertEquals((Object)swFile, (Object)this.getRequestFilename("/ctx", "/foo", swFile));
            Assertions.assertEquals((Object)swFile, (Object)this.getRequestFilename("/ctx", "/foo/bar", swFile));
            Assertions.assertEquals((Object)swFile, (Object)this.getRequestFilename("/ctx/sub", "", swFile));
            Assertions.assertEquals((Object)swFile, (Object)this.getRequestFilename("/ctx/sub", "/foo", swFile));
            Assertions.assertEquals((Object)swFile, (Object)this.getRequestFilename("/ctx/sub", "/foo/bar", swFile));
        }
    }

    private String getRequestFilename(String encodedContextPath, String servletPath, String pathInfo) {
        this.setupRequestURI(encodedContextPath, servletPath, pathInfo);
        return this.fileServer.getRequestFilename(this.request);
    }

    private void setupRequestURI(String encodedContextPath, String servletPath, String pathInfo) {
        assert (!encodedContextPath.equals("/")) : "root context is always \"\"";
        assert (encodedContextPath.equals("") || encodedContextPath.startsWith("/")) : "context always starts with /";
        assert (!encodedContextPath.endsWith("/")) : "context path should start with / but not end with /";
        assert (!servletPath.equals("/")) : "a /* mapped servlet has path \"\"";
        assert (servletPath.equals("") || servletPath.startsWith("/")) : "servlet path always starts with /";
        assert (!servletPath.endsWith("/")) : "servlet path should start with / but not end with /";
        assert (pathInfo == null || pathInfo.startsWith("/"));
        Object requestURI = "";
        if (!encodedContextPath.isEmpty()) {
            requestURI = (String)requestURI + encodedContextPath;
        }
        if (!servletPath.isEmpty()) {
            requestURI = (String)requestURI + servletPath;
        }
        if (pathInfo != null) {
            requestURI = (String)requestURI + pathInfo;
        }
        Mockito.when((Object)this.request.getContextPath()).thenReturn((Object)encodedContextPath);
        Mockito.when((Object)this.request.getServletPath()).thenReturn((Object)servletPath);
        Mockito.when((Object)this.request.getPathInfo()).thenReturn((Object)pathInfo);
        Mockito.when((Object)this.request.getRequestURI()).thenReturn(requestURI);
    }

    @Test
    public void isResourceRequest() throws Exception {
        this.fileServer.writeResponse = false;
        this.setupRequestURI("", "/static", "/file.png");
        Mockito.when((Object)this.servletService.getStaticResource("/static/file.png")).thenReturn((Object)new URL("file:///static/file.png"));
        Assertions.assertTrue((boolean)this.fileServer.serveStaticResource(this.request, this.response));
    }

    @Test
    public void isResourceRequestWithContextPath() throws Exception {
        this.fileServer.writeResponse = false;
        this.setupRequestURI("/foo", "/static", "/file.png");
        Mockito.when((Object)this.servletService.getStaticResource("/static/file.png")).thenReturn((Object)new URL("file:///static/file.png"));
        Assertions.assertTrue((boolean)this.fileServer.serveStaticResource(this.request, this.response));
    }

    @Test
    public void isNotResourceRequest() throws Exception {
        this.setupRequestURI("", "", null);
        Mockito.when((Object)this.servletContext.getResource("/")).thenReturn(null);
        Assertions.assertFalse((boolean)this.fileServer.serveStaticResource(this.request, this.response));
    }

    @Test
    public void directoryIsNotResourceRequest() throws Exception {
        this.fileServer.writeResponse = false;
        Path folder = Files.createTempDirectory("test", new FileAttribute[0]);
        this.setupRequestURI("", "", "/frontend");
        String rootAbsolutePath = folder.toFile().getAbsolutePath().replaceAll("\\\\", "/");
        if (rootAbsolutePath.endsWith("/")) {
            rootAbsolutePath = rootAbsolutePath.substring(0, rootAbsolutePath.length() - 1);
        }
        URL folderPath = new URL("file:///" + rootAbsolutePath);
        Mockito.when((Object)this.servletService.getStaticResource("/frontend")).thenReturn((Object)folderPath);
        Assertions.assertFalse((boolean)this.fileServer.serveStaticResource(this.request, this.response), (String)"Folder on disk should not be a static resource.");
        this.setupRequestURI("", "", "/fake");
        Mockito.when((Object)this.servletService.getStaticResource("/fake")).thenReturn((Object)new URL("file:///fake/"));
        Assertions.assertFalse((boolean)this.fileServer.serveStaticResource(this.request, this.response), (String)"Fake should not check the file system nor be a static resource.");
        Path tempArchive = this.generateZipArchive(folder);
        this.setupRequestURI("", "", "/frontend/.");
        Mockito.when((Object)this.servletService.getStaticResource("/frontend/.")).thenReturn((Object)new URL("jar:file:///" + tempArchive.toString().replaceAll("\\\\", "/") + "!/frontend"));
        Assertions.assertFalse((boolean)this.fileServer.serveStaticResource(this.request, this.response), (String)"Folder 'frontend' in jar should not be a static resource.");
        this.setupRequestURI("", "", "/file.txt");
        Mockito.when((Object)this.servletService.getStaticResource("/file.txt")).thenReturn((Object)new URL("jar:file:///" + tempArchive.toString().replaceAll("\\\\", "/") + "!/file.txt"));
        Assertions.assertTrue((boolean)this.fileServer.serveStaticResource(this.request, this.response), (String)"File 'file.txt' inside jar should be a static resource.");
    }

    @Test
    public void isStaticResource_jarWarFileScheme_detectsAsStaticResources() throws IOException {
        this.fileServer.writeResponse = false;
        Assertions.assertTrue((boolean)StaticFileServer.openFileSystems.isEmpty(), (String)"Can not run concurrently with other test");
        Path folder = Files.createTempDirectory("test", new FileAttribute[0]);
        File archiveFile = new File(folder.toFile(), "fake.jar");
        archiveFile.createNewFile();
        Path tempArchive = archiveFile.toPath();
        File warFile = new File(folder.toFile(), "war.jar");
        warFile.createNewFile();
        Path warArchive = warFile.toPath();
        this.generateJarInJar(archiveFile, tempArchive, warArchive);
        WarURLStreamHandlerFactory.getInstance();
        URL folderResourceURL = new URL("jar:war:" + String.valueOf(warFile.toURI().toURL()) + "!/" + archiveFile.getName() + "!/frontend");
        this.setupRequestURI("", "", "/frontend/.");
        Mockito.when((Object)this.servletService.getStaticResource("/frontend/.")).thenReturn((Object)folderResourceURL);
        Assertions.assertTrue((boolean)this.fileServer.serveStaticResource(this.request, this.response), (String)"Request should return as static request as we can not determine non file resources in jar files.");
    }

    @Test
    public void isStaticResource_jarInAJar_detectsAsStaticResources() throws IOException {
        this.fileServer.writeResponse = false;
        Assertions.assertTrue((boolean)StaticFileServer.openFileSystems.isEmpty(), (String)"Can not run concurrently with other test");
        Path folder = Files.createTempDirectory("test", new FileAttribute[0]);
        File archiveFile = new File(folder.toFile(), "fake.jar");
        archiveFile.createNewFile();
        Path tempArchive = archiveFile.toPath();
        File warFile = new File(folder.toFile(), "war.jar");
        warFile.createNewFile();
        Path warArchive = warFile.toPath();
        this.generateJarInJar(archiveFile, tempArchive, warArchive);
        this.setupRequestURI("", "", "/frontend/.");
        Mockito.when((Object)this.servletService.getStaticResource("/frontend/.")).thenReturn((Object)new URL("jar:" + String.valueOf(warFile.toURI().toURL()) + "!/" + archiveFile.getName() + "!/frontend"));
        Assertions.assertTrue((boolean)this.fileServer.serveStaticResource(this.request, this.response), (String)"Request should return as static request as we can not determine non file resources in jar files.");
        this.setupRequestURI("", "", "/file.txt");
        Mockito.when((Object)this.servletService.getStaticResource("/file.txt")).thenReturn((Object)new URL("jar:" + String.valueOf(warFile.toURI().toURL()) + "!/" + archiveFile.getName() + "!/file.txt"));
        Assertions.assertTrue((boolean)this.fileServer.serveStaticResource(this.request, this.response), (String)"Request should return as static request as we can not determine non file resources in jar files.");
    }

    private Path generateZipArchive(Path folder) throws IOException {
        File archiveFile = new File(folder.toFile(), "fake.jar");
        archiveFile.createNewFile();
        Path tempArchive = archiveFile.toPath();
        try (ZipOutputStream zipOutputStream = new ZipOutputStream(Files.newOutputStream(tempArchive, new OpenOption[0]));){
            zipOutputStream.putNextEntry(new ZipEntry("/file"));
            zipOutputStream.closeEntry();
            zipOutputStream.putNextEntry(new ZipEntry("frontend/"));
            zipOutputStream.closeEntry();
        }
        return tempArchive;
    }

    private void generateJarInJar(File archiveFile, Path tempArchive, Path warArchive) throws IOException {
        try (ZipOutputStream zipOutputStream = new ZipOutputStream(Files.newOutputStream(tempArchive, new OpenOption[0]));){
            zipOutputStream.putNextEntry(new ZipEntry("/file"));
            zipOutputStream.closeEntry();
            zipOutputStream.putNextEntry(new ZipEntry("frontend/"));
            zipOutputStream.closeEntry();
        }
        try (ZipOutputStream warOutputStream = new ZipOutputStream(Files.newOutputStream(warArchive, new OpenOption[0]));){
            warOutputStream.putNextEntry(new ZipEntry(archiveFile.getName()));
            warOutputStream.write(Files.readAllBytes(tempArchive));
            warOutputStream.closeEntry();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void openFileServerExistsForZip_openingNewDoesNotFail() throws IOException, URISyntaxException {
        Assertions.assertTrue((boolean)StaticFileServer.openFileSystems.isEmpty(), (String)"Can not run concurrently with other test");
        Path folder = Files.createTempDirectory("test", new FileAttribute[0]);
        Path tempArchive = this.generateZipArchive(folder);
        FileSystem fileSystem = FileSystems.newFileSystem(new URL("jar:file:///" + tempArchive.toString().replaceAll("\\\\", "/") + "!/").toURI(), Collections.emptyMap());
        URL folderResourceURL = new URL("jar:file:///" + tempArchive.toString().replaceAll("\\\\", "/") + "!/frontend");
        try {
            this.fileServer.getFileSystem(folderResourceURL.toURI());
        }
        finally {
            this.fileServer.closeFileSystem(folderResourceURL.toURI());
            fileSystem.close();
        }
    }

    @Test
    public void openingJarFileSystemForDifferentFilesInSameJar_existingFileSystemIsUsed() throws IOException, URISyntaxException {
        Assertions.assertTrue((boolean)StaticFileServer.openFileSystems.isEmpty(), (String)"Can not run concurrently with other test");
        Path folder = Files.createTempDirectory("test", new FileAttribute[0]);
        Path tempArchive = this.generateZipArchive(folder);
        URL folderResourceURL = new URL("jar:file:///" + tempArchive.toString().replaceAll("\\\\", "/") + "!/frontend");
        URL fileResourceURL = new URL("jar:file:///" + tempArchive.toString().replaceAll("\\\\", "/") + "!/file.txt");
        this.fileServer.getFileSystem(folderResourceURL.toURI());
        this.fileServer.getFileSystem(fileResourceURL.toURI());
        Assertions.assertEquals((Integer)2, (Integer)((Integer)StaticFileServer.openFileSystems.entrySet().iterator().next().getValue()), (String)"Same file should be marked for both resources");
        this.fileServer.closeFileSystem(folderResourceURL.toURI());
        Assertions.assertEquals((Integer)1, (Integer)((Integer)StaticFileServer.openFileSystems.entrySet().iterator().next().getValue()), (String)"Closing resource should be removed from jar uri");
        this.fileServer.closeFileSystem(fileResourceURL.toURI());
        Assertions.assertTrue((boolean)StaticFileServer.openFileSystems.isEmpty(), (String)"Closing last resource should clear marking");
        try {
            FileSystems.getFileSystem(folderResourceURL.toURI());
            Assertions.fail((String)"Jar FileSystem should have been closed");
        }
        catch (FileSystemNotFoundException fileSystemNotFoundException) {
            // empty catch block
        }
    }

    @Test
    public void concurrentRequestsToJarResources_checksAreCorrect() throws IOException, InterruptedException, ExecutionException, URISyntaxException {
        this.fileServer.writeResponse = false;
        Assertions.assertTrue((boolean)StaticFileServer.openFileSystems.isEmpty(), (String)"Can not run concurrently with other test");
        Path folder = Files.createTempDirectory("test", new FileAttribute[0]);
        Path tempArchive = this.generateZipArchive(folder);
        this.setupRequestURI("", "", "/frontend/.");
        URL folderResourceURL = new URL("jar:file:///" + tempArchive.toString().replaceAll("\\\\", "/") + "!/frontend");
        Mockito.when((Object)this.servletService.getStaticResource("/frontend/.")).thenReturn((Object)folderResourceURL);
        int THREADS = 5;
        List folderNotResource = IntStream.range(0, THREADS).mapToObj(i -> {
            Callable<Result> callable = () -> {
                try {
                    if (this.fileServer.serveStaticResource(this.request, this.response)) {
                        throw new IllegalArgumentException("Folder 'frontend' in jar should not be a static resource.");
                    }
                }
                catch (Exception e) {
                    return new Result(e);
                }
                return new Result(null);
            };
            return callable;
        }).collect(Collectors.toList());
        ExecutorService executor = Executors.newFixedThreadPool(THREADS);
        List futures = executor.invokeAll(folderNotResource);
        ArrayList<String> exceptions = new ArrayList<String>();
        executor.shutdown();
        for (Future resultFuture : futures) {
            Result result = (Result)resultFuture.get();
            if (result.exception == null) continue;
            exceptions.add(result.exception.getMessage());
        }
        Assertions.assertTrue((boolean)exceptions.isEmpty(), (String)("There were exceptions in concurrent requests {" + String.valueOf(exceptions) + "}"));
        Assertions.assertFalse((boolean)StaticFileServer.openFileSystems.containsKey(folderResourceURL.toURI()), (String)"Folder URI should have been cleared");
        try {
            FileSystems.getFileSystem(folderResourceURL.toURI());
            Assertions.fail((String)"FileSystem for folder resource should be closed");
        }
        catch (FileSystemNotFoundException fileSystemNotFoundException) {
            // empty catch block
        }
        this.setupRequestURI("", "", "/file.txt");
        URL fileResourceURL = new URL("jar:file:///" + tempArchive.toString().replaceAll("\\\\", "/") + "!/file.txt");
        Mockito.when((Object)this.servletService.getStaticResource("/file.txt")).thenReturn((Object)fileResourceURL);
        List fileIsResource = IntStream.range(0, THREADS).mapToObj(i -> {
            Callable<Result> callable = () -> {
                try {
                    if (!this.fileServer.serveStaticResource(this.request, this.response)) {
                        throw new IllegalArgumentException("File 'file.txt' inside jar should be a static resource.");
                    }
                }
                catch (Exception e) {
                    return new Result(e);
                }
                return new Result(null);
            };
            return callable;
        }).collect(Collectors.toList());
        executor = Executors.newFixedThreadPool(THREADS);
        futures = executor.invokeAll(fileIsResource);
        exceptions = new ArrayList();
        executor.shutdown();
        for (Future resultFuture : futures) {
            Result result = (Result)resultFuture.get();
            if (result.exception == null) continue;
            exceptions.add(result.exception.getMessage());
        }
        Assertions.assertTrue((boolean)exceptions.isEmpty(), (String)("There were exceptions in concurrent requests {" + String.valueOf(exceptions) + "}"));
        Assertions.assertFalse((boolean)OverrideableStaticFileServer.openFileSystems.containsKey(fileResourceURL.toURI()), (String)"URI should have been cleared");
        try {
            FileSystems.getFileSystem(fileResourceURL.toURI());
            Assertions.fail((String)"FileSystem for file resource should be closed");
        }
        catch (FileSystemNotFoundException fileSystemNotFoundException) {
            // empty catch block
        }
    }

    @Test
    public void isNotResourceRequestWithContextPath() throws Exception {
        this.setupRequestURI("/context", "", "/");
        Mockito.when((Object)this.servletContext.getResource("/")).thenReturn((Object)new URL("file", "", -1, "flow/flow-tests/non-root-context-test/src/main/webapp/", new URLStreamHandler(this){

            @Override
            protected URLConnection openConnection(URL u) throws IOException {
                URLConnection mock = (URLConnection)Mockito.mock(URLConnection.class);
                return mock;
            }
        }));
        Assertions.assertFalse((boolean)this.fileServer.serveStaticResource(this.request, this.response));
    }

    @Test
    public void writeModificationTimestampBrowserHasLatest() throws MalformedURLException {
        this.fileServer.overrideBrowserHasNewestVersion = true;
        Long modificationTimestamp = this.writeModificationTime();
        Assertions.assertEquals((Long)modificationTimestamp, (Long)this.dateHeaders.get("Last-Modified"));
    }

    @Test
    public void writeModificationTimestampBrowserDoesNotHaveLatest() throws MalformedURLException {
        this.fileServer.overrideBrowserHasNewestVersion = false;
        Long modificationTimestamp = this.writeModificationTime();
        Assertions.assertEquals((Long)modificationTimestamp, (Long)this.dateHeaders.get("Last-Modified"));
    }

    private Long writeModificationTime() throws MalformedURLException {
        LocalDateTime modificationTime = LocalDateTime.of(2016, 2, 2, 0, 0, 0);
        final Long modificationTimestamp = modificationTime.toEpochSecond(ZoneOffset.UTC) * 1000L;
        String filename = "modified-1d-ago.txt";
        URL resourceUrl = new URL("file", "", -1, filename, new URLStreamHandler(this){

            @Override
            protected URLConnection openConnection(URL u) throws IOException {
                URLConnection mock = (URLConnection)Mockito.mock(URLConnection.class);
                Mockito.when((Object)mock.getLastModified()).thenReturn((Object)modificationTimestamp);
                return mock;
            }
        });
        this.fileServer.writeModificationTimestamp(resourceUrl, this.request, this.response);
        return modificationTimestamp;
    }

    @Test
    public void browserHasNewestVersionUnknownModificiationTime() {
        Assertions.assertFalse((boolean)this.fileServer.browserHasNewestVersion(this.request, -1L));
    }

    @Test
    public void browserHasNewestVersionInvalidModificiationTime() {
        Assertions.assertThrows(AssertionError.class, () -> this.fileServer.browserHasNewestVersion(this.request, -2L));
    }

    @Test
    public void browserHasNewestVersionNoIfModifiedSinceHeader() {
        long fileModifiedTime = 0L;
        Assertions.assertFalse((boolean)this.fileServer.browserHasNewestVersion(this.request, fileModifiedTime));
    }

    @Test
    public void browserHasNewestVersionOlderIfModifiedSinceHeader() {
        long browserIfModifiedSince = 123L;
        long fileModifiedTime = 124L;
        Mockito.when((Object)this.request.getDateHeader("If-Modified-Since")).thenReturn((Object)browserIfModifiedSince);
        Assertions.assertFalse((boolean)this.fileServer.browserHasNewestVersion(this.request, fileModifiedTime));
    }

    @Test
    public void browserHasNewestVersionNewerIfModifiedSinceHeader() {
        long browserIfModifiedSince = 125L;
        long fileModifiedTime = 124L;
        Mockito.when((Object)this.request.getDateHeader("If-Modified-Since")).thenReturn((Object)browserIfModifiedSince);
        Assertions.assertTrue((boolean)this.fileServer.browserHasNewestVersion(this.request, fileModifiedTime));
    }

    @Test
    public void writeCacheHeadersCacheResource() {
        this.fileServer.overrideCacheTime = 12;
        this.fileServer.writeCacheHeaders("/folder/myfile.txt", this.response);
        Assertions.assertTrue((boolean)this.headers.get("Cache-Control").contains("max-age=12"));
    }

    @Test
    public void nonProductionMode_writeCacheHeadersCacheResource_noCache() {
        Mockito.when((Object)this.configuration.isProductionMode()).thenReturn((Object)false);
        this.fileServer.overrideCacheTime = 12;
        this.fileServer.writeCacheHeaders("/folder/myfile.txt", this.response);
        Assertions.assertTrue((boolean)this.headers.get("Cache-Control").equals("no-cache"));
    }

    @Test
    public void writeCacheHeadersDoNotCacheResource() {
        this.fileServer.overrideCacheTime = 0;
        this.fileServer.writeCacheHeaders("/folder/myfile.txt", this.response);
        Assertions.assertTrue((boolean)this.headers.get("Cache-Control").contains("max-age=0"));
        Assertions.assertTrue((boolean)this.headers.get("Cache-Control").contains("must-revalidate"));
    }

    @Test
    public void nonProductionMode_writeCacheHeadersDoNotCacheResource() {
        Mockito.when((Object)this.configuration.isProductionMode()).thenReturn((Object)false);
        this.fileServer.overrideCacheTime = 0;
        this.fileServer.writeCacheHeaders("/folder/myfile.txt", this.response);
        Assertions.assertTrue((boolean)this.headers.get("Cache-Control").equals("no-cache"));
    }

    @Test
    public void productionMode_writeCacheHeaders_withVersionParam_oneYearCache() {
        Mockito.when((Object)this.configuration.isProductionMode()).thenReturn((Object)true);
        Mockito.when((Object)this.request.getParameter("v-c")).thenReturn((Object)"abcd1234");
        this.fileServer.writeCacheHeaders("/folder/myfile.css", this.request, this.response);
        Assertions.assertEquals((Object)"max-age=31536000, immutable", (Object)this.headers.get("Cache-Control"));
    }

    @Test
    public void getCacheTime() {
        int oneYear = 31536000;
        Assertions.assertEquals((int)oneYear, (int)this.fileServer.getCacheTime("somefile.cache.js"));
        Assertions.assertEquals((int)oneYear, (int)this.fileServer.getCacheTime("folder/somefile.cache.js"));
        Assertions.assertEquals((int)0, (int)this.fileServer.getCacheTime("somefile.nocache.js"));
        Assertions.assertEquals((int)0, (int)this.fileServer.getCacheTime("folder/somefile.nocache.js"));
        Assertions.assertEquals((int)3600, (int)this.fileServer.getCacheTime("randomfile.js"));
        Assertions.assertEquals((int)3600, (int)this.fileServer.getCacheTime("folder/randomfile.js"));
    }

    @Test
    public void serveNonExistingStaticResource() throws IOException {
        this.setupRequestURI("", "", "/nonexisting/file.js");
        Assertions.assertFalse((boolean)this.fileServer.serveStaticResource(this.request, this.response));
    }

    @Test
    public void serveStaticResource() throws IOException {
        this.setupRequestURI("", "/some", "/file.js");
        String fileData = "function() {eval('foo');};";
        Mockito.when((Object)this.servletService.getStaticResource("/some/file.js")).thenReturn((Object)StaticFileServerTest.createFileURLWithDataAndLength("/some/file.js", fileData));
        Assertions.assertTrue((boolean)this.fileServer.serveStaticResource(this.request, this.response));
        Assertions.assertEquals((Object)fileData, (Object)this.out.getOutputString());
    }

    @Test
    public void contextAndServletPath_serveStaticBundleBuildResource() throws IOException {
        String pathInfo = "/VAADIN/build/vaadin-bundle-1234.cache.js";
        this.setupRequestURI("/context", "/servlet", pathInfo);
        this.assertBundleBuildResource(pathInfo);
    }

    @Test
    public void ServletPath_serveStaticBundleBuildResource() throws IOException {
        String pathInfo = "/VAADIN/build/vaadin-bundle-1234.cache.js";
        this.setupRequestURI("", "/servlet", pathInfo);
        this.assertBundleBuildResource(pathInfo);
    }

    @Test
    public void contextPath_serveStaticBundleBuildResource() throws IOException {
        String pathInfo = "/VAADIN/build/vaadin-bundle-1234.cache.js";
        this.setupRequestURI("/context", "", pathInfo);
        this.assertBundleBuildResource(pathInfo);
    }

    @Test
    public void serveStaticBundleBuildResource() throws IOException {
        String pathInfo = "/VAADIN/build/vaadin-bundle-1234.cache.js";
        this.setupRequestURI("", "", pathInfo);
        this.assertBundleBuildResource(pathInfo);
    }

    @Test
    public void contextAndServletPath_serveStaticFileResource() throws IOException {
        String pathInfo = "/VAADIN/static/img/bg.jpg";
        this.setupRequestURI("/context", "/servlet", pathInfo);
        this.assertBundleBuildResource(pathInfo);
    }

    @Test
    public void ServletPath_serveStaticFileResource() throws IOException {
        String pathInfo = "/VAADIN/static/img/bg.jpg";
        this.setupRequestURI("", "/servlet", pathInfo);
        this.assertBundleBuildResource(pathInfo);
    }

    @Test
    public void contextPath_serveStaticFileResource() throws IOException {
        String pathInfo = "/VAADIN/static/img/bg.jpg";
        this.setupRequestURI("/context", "", pathInfo);
        this.assertBundleBuildResource(pathInfo);
    }

    @Test
    public void serveStaticFileResource() throws IOException {
        String pathInfo = "/VAADIN/static/img/bg.jpg";
        this.setupRequestURI("", "", pathInfo);
        this.assertBundleBuildResource(pathInfo);
    }

    public void assertBundleBuildResource(String pathInfo) throws IOException {
        String fileData = "function() {eval('foo');};";
        ClassLoader mockLoader = (ClassLoader)Mockito.mock(ClassLoader.class);
        Mockito.when((Object)this.servletService.getClassLoader()).thenReturn((Object)mockLoader);
        Mockito.when((Object)mockLoader.getResource(WEBAPP_RESOURCE_PREFIX + pathInfo)).thenReturn((Object)StaticFileServerTest.createFileURLWithDataAndLength("/META-INF/VAADIN/webapp" + pathInfo, fileData));
        this.mockStatsBundles(mockLoader);
        this.mockConfigurationPolyfills();
        Assertions.assertTrue((boolean)this.fileServer.serveStaticResource(this.request, this.response));
        Assertions.assertEquals((Object)fileData, (Object)this.out.getOutputString());
    }

    private void staticBuildResourceWithDirectoryChange_nothingServed(String pathInfo) throws IOException {
        this.setupRequestURI("/context", "/servlet", pathInfo);
        String fileData = "function() {eval('foo');};";
        ClassLoader mockLoader = (ClassLoader)Mockito.mock(ClassLoader.class);
        Mockito.when((Object)this.servletService.getClassLoader()).thenReturn((Object)mockLoader);
        Mockito.when((Object)mockLoader.getResource(WEBAPP_RESOURCE_PREFIX + pathInfo)).thenReturn((Object)StaticFileServerTest.createFileURLWithDataAndLength("/META-INF/VAADIN/webapp" + pathInfo, fileData));
        Mockito.when((Object)mockLoader.getResource(WEBAPP_RESOURCE_PREFIX + pathInfo.replace("build/../", ""))).thenReturn((Object)StaticFileServerTest.createFileURLWithDataAndLength(WEBAPP_RESOURCE_PREFIX + pathInfo.replace("build/../", ""), fileData));
        this.mockStatsBundles(mockLoader);
        this.mockConfigurationPolyfills();
        Assertions.assertTrue((boolean)this.fileServer.serveStaticResource(this.request, this.response));
        Assertions.assertEquals((int)0, (int)this.out.getOutput().length);
        Assertions.assertEquals((int)HttpStatusCode.BAD_REQUEST.getCode(), (int)this.responseCode.get());
    }

    @Test
    public void serveStaticResource_uriWithDirectoryChangeWithSlash_returnsImmediatelyAndSetsBadRequestStatus() throws IOException {
        this.staticBuildResourceWithDirectoryChange_nothingServed("/VAADIN/build/../vaadin-bundle-1234.cache.js");
    }

    @Test
    public void serveStaticResource_uriWithDirectoryChangeWithBackslash_returnsImmediatelyAndSetsBadRequestStatus() throws IOException {
        this.staticBuildResourceWithDirectoryChange_nothingServed("/VAADIN/build/something\\..\\vaadin-bundle-1234.cache.js");
    }

    @Test
    public void serveStaticResource_uriWithDirectoryChangeWithEncodedBackslashUpperCase_returnsImmediatelyAndSetsBadRequestStatus() throws IOException {
        this.staticBuildResourceWithDirectoryChange_nothingServed("/VAADIN/build/something%5C..%5Cvaadin-bundle-1234.cache.js");
    }

    @Test
    public void serveStaticResource_uriWithDirectoryChangeWithEncodedBackslashLowerCase_returnsImmediatelyAndSetsBadRequestStatus() throws IOException {
        this.staticBuildResourceWithDirectoryChange_nothingServed("/VAADIN/build/something%5c..%5cvaadin-bundle-1234.cache.js");
    }

    @Test
    public void serveStaticResource_uriWithDirectoryChangeInTheEndWithSlash_returnsImmediatelyAndSetsBadRequestStatus() throws IOException {
        this.staticBuildResourceWithDirectoryChange_nothingServed("/VAADIN/build/..");
    }

    @Test
    public void serveStaticResource_uriWithDirectoryChangeInTheEndWithBackslash_returnsImmediatelyAndSetsBadRequestStatus() throws IOException {
        this.staticBuildResourceWithDirectoryChange_nothingServed("/VAADIN/build/something\\..");
    }

    @Test
    public void serveStaticResource_uriWithDirectoryChangeInTheEndWithEncodedBackslashUpperCase_returnsImmediatelyAndSetsBadRequestStatus() throws IOException {
        this.staticBuildResourceWithDirectoryChange_nothingServed("/VAADIN/build/something%5C..");
    }

    @Test
    public void serveStaticResource_uriWithDirectoryChangeInTheEndWithEncodedBackslashLowerCase_returnsImmediatelyAndSetsBadRequestStatus() throws IOException {
        this.staticBuildResourceWithDirectoryChange_nothingServed("/VAADIN/build/something%5c..");
    }

    @Test
    public void serveStaticResource_uriWithPercent_isServed() throws IOException {
        String pathInfo = "/VAADIN/build/100%.pdf";
        this.setupRequestURI("", "", pathInfo);
        String fileData = "contents";
        ClassLoader mockLoader = (ClassLoader)Mockito.mock(ClassLoader.class);
        Mockito.when((Object)this.servletService.getClassLoader()).thenReturn((Object)mockLoader);
        Mockito.when((Object)mockLoader.getResource(WEBAPP_RESOURCE_PREFIX + pathInfo)).thenReturn((Object)StaticFileServerTest.createFileURLWithDataAndLength("/META-INF/VAADIN/webapp" + pathInfo, fileData));
        this.mockStatsBundles(mockLoader);
        this.mockConfigurationPolyfills();
        Assertions.assertTrue((boolean)this.fileServer.serveStaticResource(this.request, this.response));
        Assertions.assertEquals((Object)fileData, (Object)this.out.getOutputString());
    }

    @Test
    public void customStaticBuildResource_isServed() throws IOException {
        String pathInfo = "/VAADIN/build/my-text.txt";
        this.setupRequestURI("", "", pathInfo);
        String fileData = "function() {eval('foo');};";
        ClassLoader mockLoader = (ClassLoader)Mockito.mock(ClassLoader.class);
        Mockito.when((Object)this.servletService.getClassLoader()).thenReturn((Object)mockLoader);
        Mockito.when((Object)mockLoader.getResource(WEBAPP_RESOURCE_PREFIX + pathInfo)).thenReturn((Object)StaticFileServerTest.createFileURLWithDataAndLength("/META-INF/VAADIN/webapp" + pathInfo, fileData));
        this.mockStatsBundles(mockLoader);
        this.mockConfigurationPolyfills();
        Assertions.assertTrue((boolean)this.fileServer.serveStaticResource(this.request, this.response));
        Assertions.assertEquals((Object)fileData, (Object)this.out.getOutputString());
    }

    @Test
    public void nonexistingStaticBuildResource_notServed() throws IOException {
        String pathInfo = "/VAADIN/build/my-text.txt";
        this.setupRequestURI("", "", pathInfo);
        ClassLoader mockLoader = (ClassLoader)Mockito.mock(ClassLoader.class);
        Mockito.when((Object)this.servletService.getClassLoader()).thenReturn((Object)mockLoader);
        this.mockStatsBundles(mockLoader);
        this.mockConfigurationPolyfills();
        Assertions.assertFalse((boolean)this.fileServer.serveStaticResource(this.request, this.response));
    }

    @Test
    public void staticManifestPathResource_isServed() throws IOException {
        String pathInfo = "/sw.js";
        this.setupRequestURI("", "", pathInfo);
        String fileData = "function() {eval('foo');};";
        ClassLoader mockLoader = (ClassLoader)Mockito.mock(ClassLoader.class);
        Mockito.when((Object)this.servletService.getClassLoader()).thenReturn((Object)mockLoader);
        Mockito.when((Object)mockLoader.getResource(WEBAPP_RESOURCE_PREFIX + pathInfo)).thenReturn((Object)StaticFileServerTest.createFileURLWithDataAndLength("/META-INF/VAADIN/webapp" + pathInfo, fileData));
        Assertions.assertTrue((boolean)this.fileServer.serveStaticResource(this.request, this.response));
        Assertions.assertEquals((Object)fileData, (Object)this.out.getOutputString());
    }

    @Test
    public void staticManifestPathIndexHtmlResource_notServed() throws IOException {
        String pathInfo = "/index.html";
        this.setupRequestURI("", "", pathInfo);
        String fileData = "function() {eval('foo');};";
        ClassLoader mockLoader = (ClassLoader)Mockito.mock(ClassLoader.class);
        Mockito.when((Object)this.servletService.getClassLoader()).thenReturn((Object)mockLoader);
        Mockito.when((Object)mockLoader.getResource(WEBAPP_RESOURCE_PREFIX + pathInfo)).thenReturn((Object)StaticFileServerTest.createFileURLWithDataAndLength("/META-INF/VAADIN/webapp" + pathInfo, fileData));
        Assertions.assertFalse((boolean)this.fileServer.serveStaticResource(this.request, this.response));
    }

    @Test
    public void customStatsJson_isServedFromServlet() throws IOException {
        String pathInfo = "/VAADIN/build/stats.json";
        this.setupRequestURI("", "/servlet", pathInfo);
        String fileData = "function() {eval('foo');};";
        ClassLoader mockLoader = (ClassLoader)Mockito.mock(ClassLoader.class);
        Mockito.when((Object)this.servletService.getClassLoader()).thenReturn((Object)mockLoader);
        this.mockStatsBundles(mockLoader);
        this.mockConfigurationPolyfills();
        Mockito.when((Object)this.servletService.getStaticResource(pathInfo)).thenReturn((Object)StaticFileServerTest.createFileURLWithDataAndLength(pathInfo, fileData));
        Assertions.assertTrue((boolean)this.fileServer.serveStaticResource(this.request, this.response));
        Assertions.assertEquals((Object)fileData, (Object)this.out.getOutputString());
    }

    public void mockConfigurationPolyfills() {
        Mockito.when((Object)this.configuration.getPolyfills()).thenReturn(Arrays.asList("".split("[, ]+")));
    }

    public void mockStatsBundles(ClassLoader mockLoader) {
        Mockito.when((Object)this.configuration.getStringProperty("statistics.file.path", "META-INF/VAADIN/config/stats.json")).thenReturn((Object)"META-INF/VAADIN/config/stats.json");
        Mockito.when((Object)mockLoader.getResourceAsStream("META-INF/VAADIN/config/stats.json")).thenReturn((Object)new ByteArrayInputStream(this.getStatsData()));
    }

    public byte[] getStatsData() {
        return "{ \"assetsByChunkName\" :{ \"index\": \"build/vaadin-bundle-1234.cache.js\"} }".getBytes(StandardCharsets.UTF_8);
    }

    @Test
    public void serveStaticResourceBrowserHasLatest() throws IOException {
        long browserLatest = 123L;
        long fileModified = 123L;
        this.setupRequestURI("", "/some", "/file.js");
        Mockito.when((Object)this.request.getDateHeader("If-Modified-Since")).thenReturn((Object)browserLatest);
        String fileData = "function() {eval('foo');};";
        Mockito.when((Object)this.servletService.getStaticResource("/some/file.js")).thenReturn((Object)StaticFileServerTest.createFileURLWithDataAndLength("/some/file.js", fileData, fileModified));
        Assertions.assertTrue((boolean)this.fileServer.serveStaticResource(this.request, this.response));
        Assertions.assertEquals((int)0, (int)this.out.getOutput().length);
        Assertions.assertEquals((int)HttpStatusCode.NOT_MODIFIED.getCode(), (int)this.responseCode.get());
    }

    @Test
    public void serveStaticResourceFromWebjarWithIncorrectPath() throws IOException {
        Mockito.when((Object)this.configuration.getBooleanProperty("vaadin.fixIncorrectWebjarPaths", false)).thenReturn((Object)true);
        String fileData = "function() {eval('foo');};";
        Mockito.when((Object)this.servletService.getStaticResource("/webjars/foo/bar.js")).thenReturn((Object)StaticFileServerTest.createFileURLWithDataAndLength("/webjars/foo/bar.js", fileData));
        this.setupRequestURI("", "", "/frontend/src/webjars/foo/bar.js");
        Assertions.assertTrue((boolean)this.fileServer.serveStaticResource(this.request, this.response));
        Assertions.assertEquals((Object)fileData, (Object)this.out.getOutputString());
    }

    @Test
    public void serveStaticResourceFromWebjarWithIncorrectPathAndFixingDisabled() throws IOException {
        Mockito.when((Object)this.configuration.getBooleanProperty("vaadin.fixIncorrectWebjarPaths", false)).thenReturn((Object)false);
        Mockito.when((Object)this.servletService.getStaticResource("/frontend/src/webjars/foo/bar.js")).thenReturn(null);
        this.setupRequestURI("", "", "/frontend/src/webjars/foo/bar.js");
        Assertions.assertFalse((boolean)this.fileServer.serveStaticResource(this.request, this.response));
    }

    @Test
    public void getStaticResource_delegateToVaadinService() throws MalformedURLException {
        URL url = new URL("http://bar");
        Mockito.when((Object)this.servletService.getStaticResource("foo")).thenReturn((Object)url);
        URL result = this.fileServer.getStaticResource("foo");
        ((VaadinServletService)Mockito.verify((Object)this.servletService)).getStaticResource("foo");
        Assertions.assertSame((Object)url, (Object)result);
    }

    @Test
    public void serveStaticResource_projectThemeResourceRequest_serveFromFrontend() throws IOException {
        File projectRootFolder = Files.createTempDirectory(this.temporaryFolder, "temp", new FileAttribute[0]).toFile();
        String styles = "body { background: black; }";
        TestUtil.createStyleCssStubInFrontend(projectRootFolder, "my-theme", "body { background: black; }");
        Mockito.when((Object)this.configuration.isProductionMode()).thenReturn((Object)false);
        Mockito.when((Object)this.configuration.getProjectFolder()).thenReturn((Object)projectRootFolder);
        Mockito.when((Object)this.configuration.getBuildFolder()).thenReturn((Object)"target");
        Mockito.when((Object)this.configuration.getFrontendFolder()).thenReturn((Object)new File(projectRootFolder, "./src/main/frontend/"));
        this.setupRequestURI("", "", "/VAADIN/themes/my-theme/styles.css");
        Assertions.assertTrue((boolean)this.fileServer.serveStaticResource(this.request, this.response));
        Assertions.assertEquals((Object)"body { background: black; }", (Object)this.out.getOutputString());
    }

    @Test
    public void serveStaticResource_externalThemeResourceRequest_serveFromBundle() throws IOException {
        File projectRootFolder = Files.createTempDirectory(this.temporaryFolder, "temp", new FileAttribute[0]).toFile();
        String styles = "body { background: black; }";
        TestUtil.createStylesCssStubInBundle(projectRootFolder, "my-theme", "body { background: black; }");
        Mockito.when((Object)this.configuration.isProductionMode()).thenReturn((Object)false);
        Mockito.when((Object)this.configuration.getProjectFolder()).thenReturn((Object)projectRootFolder);
        Mockito.when((Object)this.configuration.getBuildFolder()).thenReturn((Object)"target");
        Mockito.when((Object)this.configuration.getStringProperty("vaadin.frontend.folder", "./src/main/frontend/")).thenReturn((Object)"./src/main/frontend/");
        this.setupRequestURI("", "", "/VAADIN/themes/my-theme/styles.css");
        Assertions.assertTrue((boolean)this.fileServer.serveStaticResource(this.request, this.response));
        Assertions.assertEquals((Object)"body { background: black; }", (Object)this.out.getOutputString());
    }

    @Test
    public void serveStaticResource_themeResourceRequest_productionMode_notServeFromBundleNorFromFrontend() throws IOException {
        File projectRootFolder = Files.createTempDirectory(this.temporaryFolder, "temp", new FileAttribute[0]).toFile();
        String styles = "body { background: black; }";
        TestUtil.createStylesCssStubInBundle(projectRootFolder, "my-theme", "body { background: black; }");
        Mockito.when((Object)this.configuration.isProductionMode()).thenReturn((Object)true);
        Mockito.when((Object)this.configuration.getProjectFolder()).thenReturn((Object)projectRootFolder);
        Mockito.when((Object)this.configuration.getBuildFolder()).thenReturn((Object)"target");
        this.setupRequestURI("", "", "/themes/my-theme/styles.css");
        Assertions.assertFalse((boolean)this.fileServer.serveStaticResource(this.request, this.response));
    }

    @Test
    public void frameworkStaticFolder_withoutEndingSlash_doesNotServeStaticResource() throws IOException {
        for (String publicInternalFolderPath : HandlerHelper.getPublicInternalFolderPaths()) {
            Assertions.assertTrue((boolean)publicInternalFolderPath.startsWith("/"));
            Assertions.assertFalse((boolean)publicInternalFolderPath.endsWith("/**"));
            this.setupRequestURI("", "", publicInternalFolderPath);
            Mockito.when((Object)this.servletService.getStaticResource(publicInternalFolderPath)).thenReturn((Object)URI.create("file:///" + publicInternalFolderPath + "/").toURL());
            Assertions.assertFalse((boolean)this.fileServer.serveStaticResource(this.request, this.response), (String)(publicInternalFolderPath + " should not be a static resource."));
        }
    }

    private static class OverrideableStaticFileServer
    extends StaticFileServer {
        public boolean writeResponse = true;
        private Boolean overrideBrowserHasNewestVersion;
        private Integer overrideCacheTime;
        private DeploymentConfiguration configuration;

        OverrideableStaticFileServer(VaadinServletService servletService, DeploymentConfiguration configuration) {
            super((VaadinService)servletService);
            this.configuration = configuration;
        }

        protected boolean browserHasNewestVersion(HttpServletRequest request, long resourceLastModifiedTimestamp) {
            if (this.overrideBrowserHasNewestVersion != null) {
                return this.overrideBrowserHasNewestVersion;
            }
            return super.browserHasNewestVersion(request, resourceLastModifiedTimestamp);
        }

        protected int getCacheTime(String filenameWithPath) {
            if (this.overrideCacheTime != null) {
                return this.overrideCacheTime;
            }
            return super.getCacheTime(filenameWithPath);
        }

        public boolean serveStaticResource(HttpServletRequest request, HttpServletResponse response) throws IOException {
            if (!this.writeResponse) {
                try {
                    ResponseWriter fakeWriter = new ResponseWriter(this.configuration){

                        public void writeResponseContents(String filenameWithPath, URL resourceUrl, HttpServletRequest request, HttpServletResponse response) {
                        }
                    };
                    Field f = StaticFileServer.class.getDeclaredField("responseWriter");
                    f.setAccessible(true);
                    f.set((Object)this, fakeWriter);
                }
                catch (IllegalAccessException | IllegalArgumentException | NoSuchFieldException | SecurityException e) {
                    throw new RuntimeException(e);
                }
            }
            return super.serveStaticResource(request, response);
        }
    }

    private static class CapturingServletOutputStream
    extends ServletOutputStream {
        ByteArrayOutputStream baos = new ByteArrayOutputStream();

        private CapturingServletOutputStream() {
        }

        public void write(int b) throws IOException {
            this.baos.write(b);
        }

        public void setWriteListener(WriteListener writeListener) {
        }

        public boolean isReady() {
            return true;
        }

        public byte[] getOutput() {
            return this.baos.toByteArray();
        }

        public String getOutputString() {
            return new String(this.getOutput(), StandardCharsets.UTF_8);
        }
    }

    private static class Result {
        final Exception exception;

        Result(Exception exception) {
            this.exception = exception;
        }
    }
}

