/*
 * Copyright 2000-2026 Vaadin Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package com.vaadin.flow.component;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;

import org.jspecify.annotations.Nullable;
import org.slf4j.LoggerFactory;

import com.vaadin.flow.dom.Element;
import com.vaadin.flow.dom.ElementEffect;
import com.vaadin.flow.function.SerializableFunction;
import com.vaadin.flow.internal.nodefeature.SignalBindingFeature;
import com.vaadin.flow.signals.BindingActiveException;
import com.vaadin.flow.signals.Signal;
import com.vaadin.flow.signals.impl.Effect;

/**
 * A component to which the user can add and remove child components.
 * {@link Component} in itself provides basic support for child components that
 * are manually added as children of an element belonging to the component. This
 * interface provides an explicit API for components that explicitly support
 * adding and removing arbitrary child components.
 * <p>
 * {@link HasComponents} is generally implemented by layouts or components whose
 * primary function is to host child components. It isn't, for example,
 * implemented by non-layout components such as fields.
 * <p>
 * The default implementations assume that children are attached to
 * {@link #getElement()}. Override all methods in this interface if the
 * components should be added to some other element.
 *
 * @author Vaadin Ltd
 * @since 1.0
 */
public interface HasComponents extends HasElement, HasEnabled {
    /**
     * Adds the given components as children of this component.
     * <p>
     * In case any of the specified components has already been added to another
     * parent, it will be removed from there and added to this one.
     *
     * @param components
     *            the components to add
     */
    default void add(Component... components) {
        throwIfChildrenBindingIsActive("add");
        Objects.requireNonNull(components, "Components should not be null");
        add(Arrays.asList(components));
    }

    /**
     * Adds the given components as children of this component.
     * <p>
     * In case any of the specified components has already been added to another
     * parent, it will be removed from there and added to this one.
     *
     * @param components
     *            the components to add
     */
    default void add(Collection<Component> components) {
        throwIfChildrenBindingIsActive("add");
        Objects.requireNonNull(components, "Components should not be null");
        components.stream()
                .map(component -> Objects.requireNonNull(component,
                        "Component to add cannot be null"))
                .map(Component::getElement).forEach(getElement()::appendChild);
    }

    /**
     * Add the given text as a child of this component.
     *
     * @param text
     *            the text to add, not <code>null</code>
     */
    default void add(String text) {
        throwIfChildrenBindingIsActive("add");
        add(new Text(text));
    }

    /**
     * Removes the given child components from this component.
     *
     * @param components
     *            the components to remove
     * @throws IllegalArgumentException
     *             if there is a component whose non {@code null} parent is not
     *             this component
     */
    default void remove(Component... components) {
        throwIfChildrenBindingIsActive("remove");
        Objects.requireNonNull(components, "Components should not be null");
        remove(Arrays.asList(components));
    }

    /**
     * Removes the given child components from this component.
     *
     * @param components
     *            the components to remove
     * @throws IllegalArgumentException
     *             if there is a component whose non {@code null} parent is not
     *             this component
     */
    default void remove(Collection<Component> components) {
        throwIfChildrenBindingIsActive("remove");
        Objects.requireNonNull(components, "Components should not be null");
        List<Component> toRemove = new ArrayList<>(components.size());
        for (Component component : components) {
            Objects.requireNonNull(component,
                    "Component to remove cannot be null");
            Element parent = component.getElement().getParent();
            if (parent == null) {
                LoggerFactory.getLogger(HasComponents.class).debug(
                        "Remove of a component with no parent does nothing.");
                continue;
            }
            if (getElement().equals(parent)) {
                toRemove.add(component);
            } else {
                throw new IllegalArgumentException("The given component ("
                        + component + ") is not a child of this component");
            }
        }
        toRemove.stream().map(Component::getElement)
                .forEach(getElement()::removeChild);
    }

    /**
     * Removes all contents from this component, including child components,
     * text content as well as child elements that have been added directly to
     * this component using the {@link Element} API. It also removes the
     * children added only at the client-side.
     */
    default void removeAll() {
        throwIfChildrenBindingIsActive("removeAll");
        getElement().removeAllChildren();
    }

    /**
     * Adds the given component as a child of this component at the specific
     * index.
     * <p>
     * In case the specified component has already been added to another parent,
     * it will be removed from there and added to this one.
     *
     * @param index
     *            the index, where the component will be added. The index must
     *            be non-negative and may not exceed the children count
     * @param component
     *            the component to add, value should not be null
     */
    default void addComponentAtIndex(int index, Component component) {
        throwIfChildrenBindingIsActive("addComponentAtIndex");
        Objects.requireNonNull(component, "Component should not be null");
        if (index < 0) {
            throw new IllegalArgumentException(
                    "Cannot add a component with a negative index");
        }
        // The case when the index is bigger than the children count is handled
        // inside the method below
        getElement().insertChild(index, component.getElement());
    }

    /**
     * Adds the given component as the first child of this component.
     * <p>
     * In case the specified component has already been added to another parent,
     * it will be removed from there and added to this one.
     *
     * @param component
     *            the component to add, value should not be null
     */
    default void addComponentAsFirst(Component component) {
        throwIfChildrenBindingIsActive("addComponentAsFirst");
        addComponentAtIndex(0, component);
    }

    /**
     * Binds a list {@link Signal} to this component using a child component
     * factory. Each item {@link Signal} in the list corresponds to a child
     * component within this component.
     * <p>
     * This component is automatically updated to reflect the structure of the
     * list. Changes to the list, such as additions, removals, or reordering,
     * will update this component's children accordingly.
     * <p>
     * This component must not contain any children that are not part of the
     * list. If this component has existing children when this method is called,
     * or if it contains unrelated children after the list changes, an
     * {@link IllegalStateException} will be thrown.
     * <p>
     * New child components are created using the provided
     * <code>childFactory</code> function. This function takes a {@link Signal}
     * from the list and returns a corresponding {@link Component}. It shouldn't
     * return <code>null</code>. The {@link Signal} can be further bound to the
     * returned component as needed. Note that <code>childFactory</code> is run
     * inside a {@link Effect}, and therefore {@link Signal#get()} calls makes
     * effect re-run automatically on signal value change.
     * <p>
     * Example of usage:
     *
     * <pre>
     * SharedListSignal&lt;String&gt; taskList = new SharedListSignal&lt;&gt;(String.class);
     *
     * UnorderedList component = new UnorderedList();
     *
     * component.bindChildren(taskList, ListItem::new);
     * </pre>
     * <p>
     * Note: The default implementation adds children directly to the
     * component’s element using the Element API and does not invoke
     * {@link #add(Component...)}. Components that override {@code add} or
     * manage children indirectly must override this method to provide a
     * suitable implementation or explicitly disable it.
     *
     * @param list
     *            list signal to bind to the parent, must not be
     *            <code>null</code>
     * @param childFactory
     *            factory to create new component, must not be <code>null</code>
     * @param <T>
     *            the value type of the {@link Signal}s in the list
     * @param <S>
     *            the type of the {@link Signal}s in the list
     * @throws IllegalStateException
     *             thrown if this component isn't empty
     * @throws BindingActiveException
     *             thrown if a binding for children already exists
     */
    default <T extends @Nullable Object, S extends Signal<T>> void bindChildren(
            Signal<List<S>> list,
            SerializableFunction<S, Component> childFactory) {
        var self = (Component & HasComponents) this;
        var node = self.getElement().getNode();
        var feature = node.getFeature(SignalBindingFeature.class);
        if (feature.hasBinding(SignalBindingFeature.CHILDREN)) {
            throw new BindingActiveException();
        }
        Objects.requireNonNull(list, "Signal cannot be null");
        Objects.requireNonNull(childFactory,
                "Child component factory cannot be null");
        var binding = ElementEffect.bindChildren(self.getElement(), list,
                // wrap childFactory to convert Component to Element
                signalValue -> Optional
                        .ofNullable(childFactory.apply(signalValue))
                        .map(Component::getElement)
                        .orElseThrow(() -> new IllegalStateException(
                                "HasComponents.bindChildren childFactory must not return null")));

        feature.setBinding(SignalBindingFeature.CHILDREN, binding, list);
    }

    private void throwIfChildrenBindingIsActive(String methodName) {
        getElement().getNode()
                .getFeatureIfInitialized(SignalBindingFeature.class)
                .ifPresent(feature -> {
                    if (feature.hasBinding(SignalBindingFeature.CHILDREN)) {
                        throw new BindingActiveException(methodName
                                + " is not allowed while a binding for children exists.");
                    }
                });
    }
}
