/*
 * Copyright 2000-2026 Vaadin Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package com.vaadin.flow.signals.shared;

import java.io.Serializable;
import java.util.Objects;
import java.util.concurrent.Executor;

import org.jspecify.annotations.Nullable;
import tools.jackson.core.JacksonException;
import tools.jackson.databind.JsonNode;
import tools.jackson.databind.ObjectMapper;

import com.vaadin.flow.shared.Registration;
import com.vaadin.flow.signals.Id;
import com.vaadin.flow.signals.Node;
import com.vaadin.flow.signals.Node.Data;
import com.vaadin.flow.signals.Signal;
import com.vaadin.flow.signals.SignalCommand;
import com.vaadin.flow.signals.SignalEnvironment;
import com.vaadin.flow.signals.function.CommandValidator;
import com.vaadin.flow.signals.impl.Transaction;
import com.vaadin.flow.signals.impl.TransientListener;
import com.vaadin.flow.signals.impl.UsageTracker;
import com.vaadin.flow.signals.impl.UsageTracker.Usage;
import com.vaadin.flow.signals.operations.InsertOperation;
import com.vaadin.flow.signals.operations.SignalOperation;
import com.vaadin.flow.signals.shared.impl.CommandResult;
import com.vaadin.flow.signals.shared.impl.SignalTree;
import com.vaadin.flow.signals.shared.impl.StagedTransaction;
import com.vaadin.flow.signals.shared.impl.TreeRevision;

/**
 * Base type for full-featured signals that are backed by a transactional signal
 * tree.
 * <p>
 * This signal may be synchronized across a cluster. In that case, changes to
 * the signal value are only confirmed asynchronously. The regular signal
 * {@link #get()} returns the assumed value based on local modifications whereas
 * {@link #peekConfirmed()} gives access to the confirmed value.
 *
 * @param <T>
 *            the signal value type
 */
public abstract class AbstractSignal<T> implements Signal<T> {
    /**
     * Converts a command result into a specific value type.
     *
     * @param <T>
     *            the result value type
     */
    @FunctionalInterface
    protected interface ResultConverter<T> extends Serializable {
        /**
         * Converts an accepted command result into a result value.
         *
         * @param accept
         *            the accepted command result, not <code>null</code>
         * @return the converted value, may be <code>null</code>
         */
        @Nullable
        T convert(CommandResult.Accept accept);
    }

    /**
     * Creates a child signal instance from a node ID.
     *
     * @param <I>
     *            the child signal type
     */
    @FunctionalInterface
    protected interface ChildSignalFactory<I extends AbstractSignal<?>> {
        /**
         * Creates a child signal instance for the given node ID.
         *
         * @param childId
         *            the node ID for the child signal, not <code>null</code>
         * @return the child signal instance, not <code>null</code>
         */
        I create(Id childId);
    }

    private final SignalTree tree;
    private final Id id;
    private final CommandValidator validator;

    private static final ObjectMapper OBJECT_MAPPER;
    static {
        OBJECT_MAPPER = new ObjectMapper();
    }

    /**
     * Signal validator that accepts anything. This is defined as a constant to
     * enable using <code>==</code> to detect and optimize cases where no
     * validation is applied.
     */
    protected static final CommandValidator ANYTHING_GOES = CommandValidator.ACCEPT_ALL;

    /**
     * Creates a new signal instance with the given id and validator for the
     * given signal tree.
     *
     * @param tree
     *            the signal tree that contains the value for this signal, not
     *            <code>null</code>
     * @param id
     *            the id of the signal node within the signal tree, not
     *            <code>null</code>
     * @param validator
     *            the validator to check operations submitted to this singal,
     *            not <code>null</code>
     */
    protected AbstractSignal(SignalTree tree, Id id,
            CommandValidator validator) {
        this.tree = Objects.requireNonNull(tree);
        this.validator = Objects.requireNonNull(validator);
        this.id = Objects.requireNonNull(id);
    }

    /**
     * Gets the data node for this signal in the given tree revision.
     *
     * @param revision
     *            the tree revision to read from, not <code>null</code>
     * @return the data node, or <code>null</code> if there is no node for this
     *         signal in the revision
     */
    protected @Nullable Data data(TreeRevision revision) {
        return revision.data(id()).orElse(null);
    }

    /**
     * Gets the data node for this signal in the given transaction.
     *
     * @param transaction
     *            the transaction to read from, not <code>null</code>
     * @return the data node, or <code>null</code> if there is no node for this
     *         signal in the transaction
     */
    protected @Nullable Data data(Transaction transaction) {
        return data(transaction.read(tree()));
    }

    @Override
    public @Nullable T get() {
        if (!UsageTracker.isGetAllowed() && !Transaction.inTransaction()) {
            throw new IllegalStateException(
                    "Signal.get() was called outside a reactive context. "
                            + "Use peek() to read the value without setting up "
                            + "dependency tracking, or use "
                            + "Signal.untracked(() -> signal.get()) to "
                            + "explicitly opt out.");
        }
        Transaction transaction = Transaction.getCurrent();
        Data data = data(transaction);

        if (transaction instanceof StagedTransaction && data != null) {
            /*
             * This could be optimized to avoid creating the command if
             * lastUpdate has already been set in the same transaction
             */
            submit(new SignalCommand.LastUpdateCondition(Id.random(), id(),
                    data.lastUpdate()));
        }

        /*
         * Extract value before registering since extracting sets up state that
         * is used by registerDepedency in the case of computed signals.
         */
        T value = extractValue(data);
        if (UsageTracker.isActive()) {
            UsageTracker.registerUsage(createUsage(transaction));
        }
        return value;
    }

    @Override
    public @Nullable T peek() {
        return extractValue(data(Transaction.getCurrent()));
    }

    /**
     * Reads the confirmed value without setting up any dependencies. The
     * confirmed value doesn't consider changes in the current transaction or
     * changes that have been submitted but not yet confirmed in a cluster.
     *
     * @return the confirmed signal value
     */
    public @Nullable T peekConfirmed() {
        return extractValue(data(tree().confirmed()));
    }

    /**
     * Gets the validator used by this signal instance.
     *
     * @return the used validator, not <code>null</code>
     */
    protected CommandValidator validator() {
        return validator;
    }

    /**
     * Merges the validator used by this signal with the given validator. This
     * chains the two validators so that both must accept any change but it
     * additionally avoids redundant chaining in case either validator is
     * {@link #ANYTHING_GOES}.
     *
     * @param validator
     *            the validator to merge, not <code>null</code>
     * @return a combined validator, not <code>null</code>
     */
    protected CommandValidator mergeValidators(CommandValidator validator) {
        CommandValidator own = validator();
        if (own == ANYTHING_GOES) {
            return validator;
        } else if (validator == ANYTHING_GOES) {
            return own;
        } else {
            return own.and(validator);
        }
    }

    /**
     * Extracts the value for this signal from the given signal data node.
     *
     * @param data
     *            the data node to extract the value from, or <code>null</code>
     *            if the node doesn't exist in the tree
     * @return the signal value
     */
    protected abstract @Nullable T extractValue(@Nullable Data data);

    /**
     * Gets a reference value that will be used to determine whether a
     * dependency based on previous usage should be invalidated. This is done by
     * getting one reference value when the dependency occurs and then comparing
     * that to the current value to determine if the value has changed.
     * <p>
     * The implementation should return an object that changes if and only if
     * the {@link #get()} of this signal changes.
     *
     * @param data
     *            the data node to read from, not <code>null</code>
     * @return a reference value to use for validity checks, may be
     *         <code>null</code>
     */
    protected abstract @Nullable Object usageChangeValue(Data data);

    /**
     * Checks if the given command is valid according to this signal's
     * validator. Condition commands are always considered valid, transaction
     * commands are valid if all nested commands are valid, and other commands
     * are validated using the configured validator.
     *
     * @param command
     *            the command to validate, not <code>null</code>
     * @return <code>true</code> if the command is valid, <code>false</code>
     *         otherwise
     */
    boolean isValid(SignalCommand command) {
        if (command instanceof SignalCommand.ConditionCommand) {
            return true;
        } else if (command instanceof SignalCommand.TransactionCommand tx) {
            return tx.commands().stream().allMatch(this::isValid);
        } else {
            return validator().isValid(command);
        }
    }

    /**
     * Submits a command for this signal and updates the given operation using
     * the given result converter once the command result is confirmed. The
     * command is submitted through the current {@link Transaction} and it uses
     * {@link SignalEnvironment#getCurrentResultNotifier()} for delivering the
     * result update.
     *
     * @param <R>
     *            the result type
     * @param <O>
     *            the operation type
     * @param command
     *            the command to submit, not <code>null</code>
     * @param resultConverter
     *            a callback for creating an operation result value based on the
     *            command result, not <code>null</code>
     * @param operation
     *            the operation to update with the eventual result, not
     *            <code>null</code>
     * @return the provided operation, for chaining
     */
    protected <R, O extends SignalOperation<R>> O submit(SignalCommand command,
            ResultConverter<R> resultConverter, O operation) {
        // Remove is issued through the parent but targets the child
        assert command instanceof SignalCommand.RemoveCommand
                || id().equals(command.targetNodeId());

        if (!isValid(command)) {
            throw new UnsupportedOperationException();
        }

        Executor notifier = SignalEnvironment.getCurrentResultNotifier();

        Transaction.getCurrent().include(tree(), command, result -> {
            operation.result().completeAsync(() -> {
                if (result instanceof CommandResult.Accept accept) {
                    return new SignalOperation.Result<>(
                            resultConverter.convert(accept));
                } else if (result instanceof CommandResult.Reject reject) {
                    return new SignalOperation.Error<>(reject.reason());
                } else {
                    throw new RuntimeException(
                            "Unsupported result type: " + result);
                }
            }, notifier);
        });

        return operation;
    }

    /**
     * Submits a command for this signal and updates the given operation without
     * a value once the command result is confirmed. This is a shorthand for
     * {@link #submit(SignalCommand, ResultConverter, SignalOperation)} in the
     * case of operations that don't have a result value.
     *
     * @param <O>
     *            the operation type
     * @param command
     *            the command to submit, not <code>null</code>
     * @param operation
     *            the operation to update with the eventual result, not
     *            <code>null</code>
     * @return the provided operation, for chaining
     */
    protected <O extends SignalOperation<Void>> O submitVoidOperation(
            SignalCommand command, O operation) {
        return submit(command, success -> null, operation);
    }

    /**
     * Submits a command for this signal and creates and insert operation that
     * is updated once the command result is confirmed. This is a shorthand for
     * {@link #submit(SignalCommand, ResultConverter, SignalOperation)} in the
     * case of insert operations.
     *
     * @param <I>
     *            the insert operation type
     * @param command
     *            the command to submit, not <code>null</code>
     * @param childFactory
     *            callback used to create a signal instance in the insert
     *            operation, not <code>null</code>
     * @return the created insert operation, not <code>null</code>
     */
    protected <I extends AbstractSignal<?>> InsertOperation<I> submitInsert(
            SignalCommand command, ChildSignalFactory<I> childFactory) {
        return submitVoidOperation(command, new InsertOperation<>(
                childFactory.create(command.commandId())));
    }

    /**
     * Submits a command for this signal and uses the provided result converter
     * to updates the created operation once the command result is confirmed.
     * This is a shorthand for
     * {@link #submit(SignalCommand, ResultConverter, SignalOperation)} in the
     * case of using the default operation type.
     *
     * @param <R>
     *            the operation result value
     * @param command
     *            the command to submit, not <code>null</code>
     * @param resultConverter
     *            a callback for creating an operation result value based on the
     *            command result, not <code>null</code>
     * @return the created operation instance, not <code>null</code>
     */
    protected <R> SignalOperation<R> submit(SignalCommand command,
            ResultConverter<R> resultConverter) {
        return submit(command, resultConverter, new SignalOperation<R>());
    }

    /**
     * Submits a command for this signal and updates the created operation
     * without a value once the command result is confirmed. This is a shorthand
     * for {@link #submit(SignalCommand, ResultConverter, SignalOperation)} in
     * the case of using the default operation type and no result value.
     *
     * @param command
     *            the command to submit, not <code>null</code>
     * @return the created operation instance, not <code>null</code>
     */
    protected SignalOperation<Void> submit(SignalCommand command) {
        return submitVoidOperation(command, new SignalOperation<Void>());
    }

    /**
     * Gets the unique id of this signal instance. The id will be the same for
     * other signal instances backed by the same data, e.g. in the case of using
     * {@link #asNode()} to create a signal of different type.
     *
     * @return the signal id, not null
     */
    public Id id() {
        return id;
    }

    /**
     * Gets the signal tree that stores the value for this signal.
     *
     * @return the signal tree, not <code>null</code>
     */
    protected SignalTree tree() {
        return tree;
    }

    /**
     * Creates a usage instance based on the current state of this signal.
     *
     * @param transaction
     *            the transaction for which the usage occurs, not
     *            <code>null</code>
     * @return a usage instance, not <code>null</code>
     */
    protected Usage createUsage(Transaction transaction) {
        Data data = data(transaction);
        if (data == null) {
            // Node is removed so no usage to track
            return UsageTracker.NO_USAGE;
        }

        // Capture so that we can use it later
        Object originalValue = usageChangeValue(data);

        return new Usage() {
            @Override
            public boolean hasChanges() {
                Data currentData = data(Transaction.getCurrent());

                return currentData != null && !Objects.equals(originalValue,
                        usageChangeValue(currentData));
            }

            @Override
            public Registration onNextChange(TransientListener listener) {
                SignalTree tree = tree();

                /*
                 * Lock the tree to eliminate the risk that it processes a
                 * change after checking for previous changes but before adding
                 * a listener to the tree, since the listener would in that case
                 * miss that change
                 */
                tree.getLock().lock();
                try {
                    /*
                     * Run the listener right away if there's already a change.
                     */
                    if (hasChanges()) {
                        boolean listenToNext = listener.invoke(true);
                        /*
                         * If the listener is no longer interested in changes
                         * after an initial invocation, then return without
                         * adding a listener to the tree and thus without
                         * anything to clean up.
                         */
                        if (!listenToNext) {
                            return () -> {
                            };
                        }
                    }

                    // avoid lambda to allow proper deserialization
                    TransientListener transientListener = new TransientListener() {
                        @Override
                        public boolean invoke(boolean immediate) {
                            /*
                             * Only invoke the listener if the tree change is
                             * relevant in the context of this usage instance
                             */
                            if (hasChanges()) {
                                /*
                                 * Run listener and let it decide if we should
                                 * keep listening to the tree
                                 */
                                return listener.invoke(immediate);
                            } else {
                                /*
                                 * Keep listening to the tree since the listener
                                 * hasn't yet been invoked
                                 */
                                return true;
                            }
                        }
                    };
                    return tree.observeNextChange(id(), transientListener);
                } finally {
                    tree.getLock().unlock();
                }

            }
        };
    }

    /**
     * Converts this signal into a node signal. This allows further conversion
     * into any specific signal type through the methods in
     * {@link SharedNodeSignal}. The converted signal is backed by the same
     * underlying data and uses the same validator as this signal.
     *
     * @return this signal as a node signal, not <code>null</code>
     */
    protected SharedNodeSignal asNode() {
        // This method is protected to avoid exposing in cases where it doesn't
        // make sense
        assert (!(this instanceof SharedNodeSignal));

        return new SharedNodeSignal(tree(), id(), validator());
    }

    /**
     * Helper to submit a clear command. This is a helper is re-defined as
     * public in the signal types where a clear operation makes sense.
     *
     * @return the created signal operation instance, not <code>null</code>
     */
    protected SignalOperation<Void> clear() {
        return submit(new SignalCommand.ClearCommand(Id.random(), id()));
    }

    /**
     * Helper to submit a remove command. This is a helper is re-defined as
     * public in the signal types where a remove operation makes sense.
     *
     * @param child
     *            the child signal to remove, not <code>null</code>
     *
     * @return the created signal operation instance, not <code>null</code>
     */
    protected SignalOperation<Void> remove(AbstractSignal<?> child) {
        return submit(
                new SignalCommand.RemoveCommand(Id.random(), child.id(), id()));
    }

    /**
     * Helper to convert the given object to JSON using the global signal object
     * mapper.
     *
     * @see SignalEnvironment
     *
     * @param value
     *            the object to convert to JSON
     * @return the converted JSON node, not <code>null</code>
     */
    protected static JsonNode toJson(@Nullable Object value) {
        return OBJECT_MAPPER.valueToTree(value);
    }

    /**
     * Helper to convert the given JSON to a Java instance of the given type
     * using the global signal object mapper.
     *
     * @see SignalEnvironment
     *
     * @param <T>
     *            the target type
     * @param value
     *            the JSON value to convert
     * @param targetType
     *            the target type, not <code>null</code>
     * @return the converted Java instance
     */
    protected static <T> @Nullable T fromJson(@Nullable JsonNode value,
            Class<T> targetType) {
        try {
            return OBJECT_MAPPER.treeToValue(value, targetType);
        } catch (JacksonException e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * Helper to convert the value of the given node into Java object of the
     * given type.
     *
     * @param <T>
     *            the Java object type
     * @param node
     *            the signal node to read the value from, not <code>null</code>
     * @param valueType
     *            the type to convert to, not <code>null</code>
     * @return the converted Java instance
     */
    protected static <T> @Nullable T nodeValue(Node node, Class<T> valueType) {
        assert node instanceof Data;

        return fromJson(((Data) node).value(), valueType);
    }
}
