/*
 * Decompiled with CFR 0.152.
 */
package com.vaadin.flow.server.frontend;

import com.vaadin.flow.server.frontend.JarContentsManager;
import com.vaadin.flow.testutil.TestUtils;
import java.io.File;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.jar.JarFile;
import java.util.stream.Collectors;
import java.util.zip.ZipEntry;
import org.apache.commons.io.FileUtils;
import org.hamcrest.CoreMatchers;
import org.hamcrest.Matcher;
import org.hamcrest.MatcherAssert;
import org.junit.Assert;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;
import org.junit.rules.TemporaryFolder;

public class JarContentsManagerTest {
    @Rule
    public TemporaryFolder testDirectory = new TemporaryFolder();
    @Rule
    public final ExpectedException expectedException = ExpectedException.none();
    private final JarContentsManager jarContentsManager = new JarContentsManager();
    private final File testJar = TestUtils.getTestJar();

    @Test
    public void getFileContents_directoryInsteadOfJar() {
        this.expectedException.expect(IllegalArgumentException.class);
        this.expectedException.expectMessage(String.format("Expect '%s' to be an existing file", this.testDirectory.getRoot()));
        this.jarContentsManager.getFileContents(this.testDirectory.getRoot(), "test");
    }

    @Test
    public void getFileContents_notAJarFile() throws IOException {
        File testFile = this.testDirectory.newFile("test");
        this.expectedException.expect(UncheckedIOException.class);
        this.expectedException.expectMessage(String.format("Failed to retrieve file '%s' from jar '%s'", "test", testFile));
        this.jarContentsManager.getFileContents(testFile, "test");
    }

    @Test
    public void getFileContents_nonExistingJarFile() {
        File test = new File("test");
        this.expectedException.expect(IllegalArgumentException.class);
        this.expectedException.expectMessage(String.format("Expect '%s' to be an existing file", test));
        this.jarContentsManager.getFileContents(test, "test");
    }

    @Test
    public void getFileContents_nonExistingFile() {
        byte[] fileContents = this.jarContentsManager.getFileContents(this.testJar, "blah");
        Assert.assertNull((String)"Expect to have non-empty file from jar", (Object)fileContents);
    }

    @Test
    public void getFileContents_existingFile() {
        byte[] fileContents = this.jarContentsManager.getFileContents(this.testJar, "META-INF/resources/webjars/paper-button/2.0.0/bower.json");
        Assert.assertNotNull((String)"Expect to have non-empty file from jar", (Object)fileContents);
        Assert.assertTrue((String)"Expect to have non-empty file from jar", (fileContents.length > 0 ? 1 : 0) != 0);
    }

    @Test
    public void containsPath_directoryInsteadOfJar() {
        this.expectedException.expect(IllegalArgumentException.class);
        this.expectedException.expectMessage(String.format("Expect '%s' to be an existing file", this.testDirectory.getRoot()));
        this.jarContentsManager.containsPath(this.testDirectory.getRoot(), "test");
    }

    @Test
    public void containsPath_notAJarFile() throws IOException {
        File testFile = this.testDirectory.newFile("test");
        this.expectedException.expect(UncheckedIOException.class);
        this.expectedException.expectMessage(String.format("Failed to retrieve file '%s' from jar '%s'", "test", testFile));
        this.jarContentsManager.containsPath(testFile, "test");
    }

    @Test
    public void containsPath_nonExistingJarFile() {
        File test = new File("test");
        this.expectedException.expect(IllegalArgumentException.class);
        this.expectedException.expectMessage(String.format("Expect '%s' to be an existing file", test));
        this.jarContentsManager.containsPath(test, "test");
    }

    @Test
    public void containsPath_nonExistingPath() {
        String nonExistingPath = "should not exist";
        Assert.assertFalse((String)String.format("Test jar '%s' should not contain path '%s'", this.testJar, nonExistingPath), (boolean)this.jarContentsManager.containsPath(this.testJar, nonExistingPath));
    }

    @Test
    public void containsPath_existingFile() {
        String existingPath = "META-INF/resources/webjars/";
        Assert.assertTrue((String)String.format("Test jar '%s' should contain path '%s'", this.testJar, existingPath), (boolean)this.jarContentsManager.containsPath(this.testJar, existingPath));
    }

    @Test
    public void containsPath_missingDirectoryStructure_scansForMatch() {
        String existingPathLocal = "META-INF/resources/webjars/";
        File testJarLocal = TestUtils.getTestJar((String)"test-jar-issue-6241.jar");
        Assert.assertTrue((String)String.format("Test jar '%s' should contain path '%s'", this.testJar, existingPathLocal), (boolean)this.jarContentsManager.containsPath(testJarLocal, existingPathLocal));
    }

    @Test
    public void findFiles_directoryInsteadOfJar() {
        this.expectedException.expect(IllegalArgumentException.class);
        this.expectedException.expectMessage(String.format("Expect '%s' to be an existing file", this.testDirectory.getRoot()));
        this.jarContentsManager.findFiles(this.testDirectory.getRoot(), "test", "test");
    }

    @Test
    public void findFiles_notAJarFile() throws IOException {
        File testFile = this.testDirectory.newFile("test");
        this.expectedException.expect(UncheckedIOException.class);
        this.expectedException.expectMessage("java.util.zip.ZipException: zip file is empty");
        this.jarContentsManager.findFiles(testFile, "test", "test");
    }

    @Test
    public void findFiles_nonExistingJarFile() {
        File test = new File("test");
        this.expectedException.expect(IllegalArgumentException.class);
        this.expectedException.expectMessage(String.format("Expect '%s' to be an existing file", test));
        this.jarContentsManager.findFiles(test, "test", "test");
    }

    @Test
    public void findFiles_nonExistingFile() {
        List result = this.jarContentsManager.findFiles(this.testJar, "blah", "nope");
        Assert.assertTrue((String)"Expect to have empty results for non-existing file", (boolean)result.isEmpty());
    }

    @Test
    public void findFiles_existingFiles() {
        String resourceName = "vaadin-charts-webjar-6.0.0-alpha3.jar";
        String searchName = "bower.json";
        List bowerJsons = this.jarContentsManager.findFiles(TestUtils.getTestJar((String)resourceName), "", searchName);
        Assert.assertEquals((String)String.format("Expect '%s' WebJar to contain two '%s' files", resourceName, searchName), (long)2L, (long)bowerJsons.size());
        Assert.assertTrue((String)String.format("Expect all found paths to end with the file name searched for: '%s'", searchName), (boolean)bowerJsons.stream().allMatch(path -> path.endsWith("/" + searchName)));
    }

    @Test
    public void findFiles_existingFiles_baseDirectoryMatters() {
        String resourceName = "vaadin-charts-webjar-6.0.0-alpha3.jar";
        String testPath = "META-INF/resources/webjars/highcharts/5.0.14/";
        String searchName = "bower.json";
        List bowerJson = this.jarContentsManager.findFiles(TestUtils.getTestJar((String)resourceName), testPath, searchName);
        Assert.assertEquals((String)String.format("Expect '%s' WebJar to contain one '%s' file in directory '%s'", resourceName, searchName, testPath), (long)1L, (long)bowerJson.size());
    }

    @Test
    public void copyFilesFromJar_nullJarFile() {
        this.expectedException.expect(NullPointerException.class);
        this.jarContentsManager.copyFilesFromJarTrimmingBasePath(null, null, this.testDirectory.getRoot(), new String[0]);
    }

    @Test
    public void copyFilesFromJar_notAJarFile() throws IOException {
        File testFile = this.testDirectory.newFile("test");
        this.expectedException.expect(UncheckedIOException.class);
        this.expectedException.expectMessage(String.format("Failed to extract files from jarFile '%s' to directory '%s'", testFile, this.testDirectory.getRoot()));
        this.jarContentsManager.copyFilesFromJarTrimmingBasePath(testFile, null, this.testDirectory.getRoot(), new String[0]);
    }

    @Test
    public void copyFilesFromJar_nonExistingJarFile() {
        File test = new File("test");
        this.expectedException.expect(IllegalArgumentException.class);
        this.expectedException.expectMessage(String.format("Expect '%s' to be an existing file", test));
        this.jarContentsManager.copyFilesFromJarTrimmingBasePath(test, null, this.testDirectory.getRoot(), new String[0]);
    }

    @Test
    public void copyFilesFromJar_directoryInsteadOfJar() {
        this.expectedException.expect(IllegalArgumentException.class);
        this.expectedException.expectMessage(String.format("Expect '%s' to be an existing file", this.testDirectory.getRoot()));
        this.jarContentsManager.copyFilesFromJarTrimmingBasePath(this.testDirectory.getRoot(), null, this.testDirectory.getRoot(), new String[0]);
    }

    @Test
    public void copyFilesFromJar_nullOutputDirectory() {
        this.expectedException.expect(NullPointerException.class);
        this.jarContentsManager.copyFilesFromJarTrimmingBasePath(this.testJar, null, null, new String[0]);
    }

    @Test
    public void copyFilesFromJar_fileInsteadOfDirectory() {
        this.expectedException.expect(IllegalArgumentException.class);
        this.expectedException.expectMessage(String.format("Expect '%s' to be an existing directory", this.testJar));
        this.jarContentsManager.copyFilesFromJarTrimmingBasePath(this.testJar, null, this.testJar, new String[0]);
    }

    @Test
    public void copyFilesFromJar_noBasePath_noExclusions() {
        MatcherAssert.assertThat((String)"Do not expect any files in temporary directory before the test", (Object)TestUtils.listFilesRecursively((File)this.testDirectory.getRoot()).size(), (Matcher)CoreMatchers.is((Object)0));
        this.jarContentsManager.copyFilesFromJarTrimmingBasePath(this.testJar, null, this.testDirectory.getRoot(), new String[0]);
        MatcherAssert.assertThat((String)"Temporary directory should have files after jar copied", (Object)TestUtils.listFilesRecursively((File)this.testDirectory.getRoot()).size(), (Matcher)CoreMatchers.is((Matcher)CoreMatchers.not((Object)0)));
    }

    @Test
    public void copyFilesFromJar_noBasePath_excludeEverything() {
        this.jarContentsManager.copyFilesFromJarTrimmingBasePath(this.testJar, null, this.testDirectory.getRoot(), new String[]{"*"});
        MatcherAssert.assertThat((String)"Do not expect any files with filter that excludes everything", (Object)TestUtils.listFilesRecursively((File)this.testDirectory.getRoot()).size(), (Matcher)CoreMatchers.is((Object)0));
    }

    @Test
    public void copyFilesFromJar_withBasePath_noExclusions() throws IOException {
        String basePath = "META-INF/maven/";
        this.jarContentsManager.copyFilesFromJarTrimmingBasePath(this.testJar, basePath, this.testDirectory.getRoot(), new String[0]);
        List resultingPaths = TestUtils.listFilesRecursively((File)this.testDirectory.getRoot());
        MatcherAssert.assertThat((String)String.format("Expect jar '%s' to contain files with base path '%s'", this.testJar, basePath), (Object)resultingPaths.size(), (Matcher)CoreMatchers.is((Matcher)CoreMatchers.not((Object)0)));
        Assert.assertTrue((String)("Resulting paths should not contain base path = " + basePath), (boolean)resultingPaths.stream().noneMatch(path -> path.contains(basePath)));
    }

    @Test
    public void copyFilesFromJar_exclusionsWork() throws IOException {
        String basePath = "META-INF/maven";
        File notFilteredCopyingDirectory = this.testDirectory.newFolder("notFiltered");
        this.jarContentsManager.copyFilesFromJarTrimmingBasePath(this.testJar, basePath, notFilteredCopyingDirectory, new String[0]);
        List notFilteredPaths = TestUtils.listFilesRecursively((File)notFilteredCopyingDirectory);
        File filteredCopyingDirectory = this.testDirectory.newFolder("filtered");
        this.jarContentsManager.copyFilesFromJarTrimmingBasePath(this.testJar, basePath, filteredCopyingDirectory, new String[]{"*.xml"});
        List filteredPaths = TestUtils.listFilesRecursively((File)filteredCopyingDirectory);
        Assert.assertTrue((String)"Filtered paths' count should be less than non filtered paths' count", (filteredPaths.size() < notFilteredPaths.size() ? 1 : 0) != 0);
        Assert.assertTrue((String)"Not filtered paths should contain xml files", (boolean)notFilteredPaths.stream().anyMatch(path -> path.endsWith(".xml")));
        Assert.assertTrue((String)"Paths with '*.xml' exclusion should not contain xml files", (boolean)filteredPaths.stream().noneMatch(path -> path.endsWith(".xml")));
    }

    @Test
    public void copyFilesFromJar_basePathAppendedWithTrailingSlash() throws IOException {
        String basePath1 = "META-INF/maven";
        File basePath1Directory = this.testDirectory.newFolder("basePath1");
        this.jarContentsManager.copyFilesFromJarTrimmingBasePath(this.testJar, basePath1, basePath1Directory, new String[0]);
        List basePath1Paths = TestUtils.listFilesRecursively((File)basePath1Directory);
        String basePath2 = basePath1 + "/";
        File basePath2Directory = this.testDirectory.newFolder("basePath2");
        this.jarContentsManager.copyFilesFromJarTrimmingBasePath(this.testJar, basePath2, basePath2Directory, new String[0]);
        List basePath2Paths = TestUtils.listFilesRecursively((File)basePath2Directory);
        Assert.assertEquals((String)"Base path without trailing slash should be treated the same as base path with one", (Object)basePath1Paths, (Object)basePath2Paths);
    }

    @Test
    public void copyFilesFromJar_copiedFromBasePathResultsAreContainedInAllPaths() throws IOException {
        File allFilesDirectory = this.testDirectory.newFolder("all");
        this.jarContentsManager.copyFilesFromJarTrimmingBasePath(this.testJar, null, allFilesDirectory, new String[0]);
        List allPaths = TestUtils.listFilesRecursively((File)allFilesDirectory);
        String basePath = "/META-INF/maven";
        File filteredFilesDirectory = this.testDirectory.newFolder("filtered");
        this.jarContentsManager.copyFilesFromJarTrimmingBasePath(this.testJar, basePath, filteredFilesDirectory, new String[0]);
        List filteredPaths = TestUtils.listFilesRecursively((File)filteredFilesDirectory);
        List filteredPathsPrefixedByBasePath = filteredPaths.stream().map(path -> basePath + path).collect(Collectors.toList());
        Assert.assertTrue((String)"Filtered paths' count should be less than all paths' count", (filteredPaths.size() < allPaths.size() ? 1 : 0) != 0);
        Assert.assertTrue((String)"base path + filtered path should be contained in all paths", (boolean)allPaths.containsAll(filteredPathsPrefixedByBasePath));
    }

    @Test
    public void copyFilesFromJar_casePreserved() {
        File outputDirectory = this.testDirectory.getRoot();
        String jarDirectory = "META-INF/resources/webjars/paper-button/2.0.0/.github/";
        File testJar = TestUtils.getTestJar((String)"paper-button-2.0.0.jar");
        List<String> originalFiles = this.listFilesInJar(testJar, jarDirectory);
        this.jarContentsManager.copyFilesFromJarTrimmingBasePath(testJar, jarDirectory, outputDirectory, new String[0]);
        HashSet copiedFiles = new HashSet(TestUtils.listFilesRecursively((File)outputDirectory));
        Assert.assertEquals((String)String.format("Number of files in jar '%s' in jar directory '%s' and number of copied files should match.", testJar, jarDirectory), (long)originalFiles.size(), (long)copiedFiles.size());
        copiedFiles.forEach(copiedFile -> Assert.assertTrue((String)String.format("Failed to find copied file '%s' in files '%s' from jar '%s'", copiedFile, originalFiles, testJar), (boolean)originalFiles.stream().anyMatch(file -> file.endsWith((String)copiedFile))));
    }

    @Test
    public void copyFilesFromJar_doNotUpdateFileIfContentIsTheSame() {
        File outputDirectory = this.testDirectory.getRoot();
        String jarDirectory = "META-INF/resources/webjars/paper-button";
        File testJar = TestUtils.getTestJar((String)"paper-button-2.0.0.jar");
        File jsonFile = this.copyFilesFromJar(outputDirectory, jarDirectory, testJar);
        long timestamp = jsonFile.lastModified();
        this.jarContentsManager.copyFilesFromJarTrimmingBasePath(testJar, jarDirectory, outputDirectory, new String[0]);
        Assert.assertEquals((long)timestamp, (long)jsonFile.lastModified());
    }

    @Test
    public void copyFilesFromJar_updateFileIfContentIsNotTheSame() throws IOException {
        File outputDirectory = this.testDirectory.getRoot();
        String jarDirectory = "META-INF/resources/webjars/paper-button";
        File testJar = TestUtils.getTestJar((String)"paper-button-2.0.0.jar");
        File jsonFile = this.copyFilesFromJar(outputDirectory, jarDirectory, testJar);
        String originalContent = FileUtils.readLines((File)jsonFile, (Charset)StandardCharsets.UTF_8).stream().collect(Collectors.joining(""));
        String content = "{}";
        FileUtils.write((File)jsonFile, (CharSequence)content, (Charset)StandardCharsets.UTF_8);
        this.jarContentsManager.copyFilesFromJarTrimmingBasePath(testJar, jarDirectory, outputDirectory, new String[0]);
        Assert.assertNotEquals((Object)content, (Object)FileUtils.readLines((File)jsonFile, (Charset)StandardCharsets.UTF_8).stream().collect(Collectors.joining("")));
        Assert.assertEquals((Object)originalContent, (Object)FileUtils.readLines((File)jsonFile, (Charset)StandardCharsets.UTF_8).stream().collect(Collectors.joining("")));
    }

    private File copyFilesFromJar(File outputDirectory, String jarDirectory, File testJar) {
        List<String> originalFiles = this.listFilesInJar(testJar, jarDirectory);
        Optional<String> json = originalFiles.stream().filter(fileName -> fileName.endsWith(".json")).findFirst();
        assert (json.isPresent());
        String jsonPath = json.get();
        jsonPath = jsonPath.substring(jarDirectory.length() + 1);
        this.jarContentsManager.copyFilesFromJarTrimmingBasePath(testJar, jarDirectory, outputDirectory, new String[0]);
        File jsonFile = new File(outputDirectory, jsonPath);
        assert (jsonFile.exists());
        return jsonFile;
    }

    private List<String> listFilesInJar(File jar, String jarDirectory) {
        List<String> list;
        JarFile jarFile = new JarFile(jar, false);
        try {
            list = jarFile.stream().filter(file -> !file.isDirectory()).filter(file -> file.getName().startsWith(jarDirectory)).map(ZipEntry::getName).collect(Collectors.toList());
        }
        catch (Throwable throwable) {
            try {
                try {
                    jarFile.close();
                }
                catch (Throwable throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
            catch (IOException e) {
                throw new RuntimeException(String.format("Failed to list files in jarFile '%s'", jar), e);
            }
        }
        jarFile.close();
        return list;
    }
}

