/*
 * Decompiled with CFR 0.152.
 */
package com.vaadin.flow.internal;

import com.vaadin.flow.function.DeploymentConfiguration;
import com.vaadin.flow.internal.Pair;
import com.vaadin.flow.internal.ResponseWriter;
import com.vaadin.flow.internal.SimpleMultipartParser;
import com.vaadin.tests.util.MockDeploymentConfiguration;
import jakarta.servlet.ServletContext;
import jakarta.servlet.ServletOutputStream;
import jakarta.servlet.ServletRequest;
import jakarta.servlet.ServletResponse;
import jakarta.servlet.WriteListener;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpServletResponse;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.Serializable;
import java.io.UncheckedIOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLConnection;
import java.net.URLStreamHandler;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;
import java.util.zip.GZIPOutputStream;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;
import org.mockito.ArgumentMatchers;
import org.mockito.Mockito;

public class ResponseWriterTest {
    private static final String PATH_JS = "/static/file.js";
    private static final String PATH_GZ = "/static/file.js.gz";
    private static final String PATH_BR = "/static/file.js.br";
    private static final String CLASS_PATH_JS = "/VAADIN/build/file.js";
    private static final String CLASS_PATH_GZ = "/VAADIN/build/file.js.gz";
    private static final String FAULTY_CLASS_PATH_JS = "/VAADIN/config/file.js";
    private static final String FAULTY_CLASS_PATH_GZ = "/VAADIN/config/file.js.gz";
    private static final byte[] fileJsContents = "File.js contents".getBytes(StandardCharsets.UTF_8);
    private static final byte[] fileJsGzippedContents = ResponseWriterTest.gzip(fileJsContents);
    private static final byte[] fileJsBrotliContents = "Fake brotli".getBytes();
    private static final Map<String, URL> pathToUrl = new HashMap<String, URL>();
    private ServletContext servletContext;
    private HttpServletRequest request;
    private HttpServletResponse response;
    private AtomicLong responseContentLength;
    private OverrideableResponseWriter responseWriter;

    @Before
    public void setUp() {
        MockDeploymentConfiguration deploymentConfiguration = new MockDeploymentConfiguration();
        deploymentConfiguration.setBrotli(true);
        this.responseWriter = new OverrideableResponseWriter((DeploymentConfiguration)deploymentConfiguration);
        this.servletContext = (ServletContext)Mockito.mock(ServletContext.class);
        this.request = (HttpServletRequest)Mockito.mock(HttpServletRequest.class);
        Mockito.when((Object)this.request.getServletContext()).thenReturn((Object)this.servletContext);
        Mockito.when((Object)this.request.getDateHeader(ArgumentMatchers.anyString())).thenReturn((Object)-1L);
        this.response = (HttpServletResponse)Mockito.mock(HttpServletResponse.class);
        this.responseContentLength = new AtomicLong(-1L);
        ((HttpServletResponse)Mockito.doAnswer(invocation -> {
            this.responseContentLength.set((Long)invocation.getArguments()[0]);
            return null;
        }).when((Object)this.response)).setContentLengthLong(ArgumentMatchers.anyLong());
    }

    @Test
    public void contentType() {
        AtomicReference<Object> contentType = new AtomicReference<Object>(null);
        ((HttpServletResponse)Mockito.doAnswer(invocation -> {
            contentType.set(((String)invocation.getArguments()[0]));
            return null;
        }).when((Object)this.response)).setContentType(ArgumentMatchers.anyString());
        Mockito.when((Object)this.servletContext.getMimeType("/file.png")).thenReturn((Object)"image/png");
        this.responseWriter.writeContentType("/file.png", (ServletRequest)this.request, (ServletResponse)this.response);
        Assert.assertEquals((Object)"image/png", contentType.get());
    }

    @Test
    public void noContentType() {
        AtomicReference<Object> contentType = new AtomicReference<Object>(null);
        ((HttpServletResponse)Mockito.doAnswer(invocation -> {
            contentType.set(((String)invocation.getArguments()[0]));
            return null;
        }).when((Object)this.response)).setContentType(ArgumentMatchers.anyString());
        Mockito.when((Object)this.servletContext.getMimeType("/file")).thenReturn(null);
        this.responseWriter.writeContentType("/file", (ServletRequest)this.request, (ServletResponse)this.response);
        Assert.assertNull(contentType.get());
    }

    @Test
    public void acceptsGzippedResource() {
        Assert.assertTrue((boolean)this.acceptsGzippedResource("compress, gzip"));
        Assert.assertTrue((boolean)this.acceptsGzippedResource("brotli, gzip"));
        Assert.assertTrue((boolean)this.acceptsGzippedResource("gzip"));
        Assert.assertTrue((boolean)this.acceptsGzippedResource("gzip;"));
        Assert.assertTrue((boolean)this.acceptsGzippedResource("gzip;q"));
        Assert.assertFalse((boolean)this.acceptsGzippedResource("compress; q=1 , gzip;q=0"));
        Assert.assertFalse((boolean)this.acceptsGzippedResource(""));
        Assert.assertFalse((boolean)this.acceptsGzippedResource("compress"));
        Assert.assertTrue((boolean)this.acceptsGzippedResource("compress;q = 0.5, gzip;q=0.6"));
        Assert.assertTrue((boolean)this.acceptsGzippedResource("gzip;q=1.0, identity;q=0.5, *;q=0"));
        Assert.assertTrue((boolean)this.acceptsGzippedResource("*"));
        Assert.assertTrue((boolean)this.acceptsGzippedResource("*;q=0;gzip"));
        Assert.assertFalse((boolean)this.acceptsGzippedResource("*;q=0"));
        Assert.assertFalse((boolean)this.acceptsGzippedResource("*;q=0.0"));
        Assert.assertFalse((boolean)this.acceptsGzippedResource("*;q=0.00"));
        Assert.assertFalse((boolean)this.acceptsGzippedResource("*;q=0.000"));
    }

    private boolean acceptsGzippedResource(String acceptEncodingHeader) {
        Mockito.when((Object)this.request.getHeader("Accept-Encoding")).thenReturn((Object)acceptEncodingHeader);
        return this.responseWriter.acceptsGzippedResource(this.request);
    }

    @Test
    public void acceptsBrotliResource() {
        Assert.assertTrue((boolean)this.acceptsBrotliResource("compress, brotli"));
        Assert.assertFalse((boolean)this.acceptsBrotliResource("gzip"));
        Assert.assertTrue((boolean)this.acceptsBrotliResource("compress;q = 0.5, brotli;q=0.6"));
        Assert.assertTrue((boolean)this.acceptsBrotliResource("*"));
        Assert.assertFalse((boolean)this.acceptsBrotliResource("*;q=0"));
    }

    private boolean acceptsBrotliResource(String acceptEncodingHeader) {
        Mockito.when((Object)this.request.getHeader("Accept-Encoding")).thenReturn((Object)acceptEncodingHeader);
        return this.responseWriter.acceptsBrotliResource(this.request);
    }

    @Test
    public void writeDataGzipped() throws IOException {
        this.responseWriter.overrideAcceptsGzippedResource = true;
        this.makePathsAvailable(PATH_JS, PATH_GZ);
        this.assertResponse(fileJsGzippedContents);
    }

    @Test
    public void writeDataGzippedClassPathResource() throws IOException {
        this.responseWriter.overrideAcceptsGzippedResource = true;
        this.makePathsAvailable(CLASS_PATH_JS);
        this.makeClassPathAvailable(CLASS_PATH_GZ);
        this.assertResponse(CLASS_PATH_JS, fileJsGzippedContents);
    }

    @Test
    public void writeDataNotGzippedClassPathNotAcceptedPath() throws IOException {
        this.responseWriter.overrideAcceptsGzippedResource = true;
        this.makePathsAvailable(FAULTY_CLASS_PATH_JS);
        this.makeClassPathAvailable(FAULTY_CLASS_PATH_GZ);
        this.assertResponse(FAULTY_CLASS_PATH_JS, fileJsContents);
    }

    @Test
    public void writeDataNoGzippedVersion() throws IOException {
        this.responseWriter.overrideAcceptsGzippedResource = true;
        this.makePathsAvailable(PATH_JS);
        this.assertResponse(fileJsContents);
    }

    @Test
    public void writeDataBrowserDoesNotAcceptGzippedVersion() throws IOException {
        this.responseWriter.overrideAcceptsGzippedResource = false;
        this.makePathsAvailable(PATH_JS, PATH_GZ);
        this.assertResponse(fileJsContents);
    }

    @Test
    public void writeDataBrotli() throws IOException {
        this.responseWriter.overrideAcceptsBrotliResource = Boolean.TRUE;
        this.responseWriter.overrideAcceptsGzippedResource = Boolean.TRUE;
        this.makePathsAvailable(PATH_JS, PATH_GZ, PATH_BR);
        this.assertResponse(fileJsBrotliContents);
    }

    @Test
    public void writeDataNoBrotliVersion() throws IOException {
        this.responseWriter.overrideAcceptsBrotliResource = Boolean.TRUE;
        this.makePathsAvailable(PATH_JS);
        this.assertResponse(fileJsContents);
    }

    @Test
    public void writeDataBrowserDoesNotAcceptBrotli() throws IOException {
        this.responseWriter.overrideAcceptsBrotliResource = Boolean.FALSE;
        this.makePathsAvailable(PATH_JS, PATH_BR);
        this.assertResponse(fileJsContents);
    }

    @Test
    public void writeDataBrotliDisabled() throws IOException {
        MockDeploymentConfiguration configuration = new MockDeploymentConfiguration();
        configuration.setBrotli(false);
        this.responseWriter = new OverrideableResponseWriter((DeploymentConfiguration)configuration);
        this.responseWriter.overrideAcceptsBrotliResource = Boolean.TRUE;
        this.makePathsAvailable(PATH_JS, PATH_BR);
        this.assertResponse(fileJsContents);
    }

    @Test
    public void writeByteRangeFromStart() throws IOException {
        this.makePathsAvailable(PATH_JS);
        this.mockRequestHeaders(new Pair((Serializable)((Object)"Range"), (Serializable)((Object)"bytes=0-1")));
        this.assertResponse(Arrays.copyOfRange(fileJsContents, 0, 2));
        this.assertResponseHeaders(new Pair((Serializable)((Object)"Accept-Ranges"), (Serializable)((Object)"bytes")), new Pair((Serializable)((Object)"Content-Range"), (Serializable)((Object)("bytes 0-1/" + fileJsContents.length))));
        Assert.assertEquals((long)2L, (long)this.responseContentLength.get());
        this.assertStatus(206);
    }

    @Test
    public void writeByteRangeSubset() throws IOException {
        this.makePathsAvailable(PATH_JS);
        this.mockRequestHeaders(new Pair((Serializable)((Object)"Range"), (Serializable)((Object)"bytes=10-11")));
        this.assertResponse(Arrays.copyOfRange(fileJsContents, 10, 12));
        this.assertResponseHeaders(new Pair((Serializable)((Object)"Accept-Ranges"), (Serializable)((Object)"bytes")), new Pair((Serializable)((Object)"Content-Range"), (Serializable)((Object)("bytes 10-11/" + fileJsContents.length))));
        Assert.assertEquals((long)2L, (long)this.responseContentLength.get());
        this.assertStatus(206);
    }

    @Test
    public void writeByteRangeStartOmitted() throws IOException {
        this.makePathsAvailable(PATH_JS);
        this.mockRequestHeaders(new Pair((Serializable)((Object)"Range"), (Serializable)((Object)"bytes=-10")));
        this.assertResponse(Arrays.copyOfRange(fileJsContents, 0, 11));
        this.assertResponseHeaders(new Pair((Serializable)((Object)"Accept-Ranges"), (Serializable)((Object)"bytes")), new Pair((Serializable)((Object)"Content-Range"), (Serializable)((Object)("bytes 0-10/" + fileJsContents.length))));
        Assert.assertEquals((long)11L, (long)this.responseContentLength.get());
        this.assertStatus(206);
    }

    @Test
    public void writeByteRangeEndOmitted() throws IOException {
        this.makePathsAvailable(PATH_JS);
        this.mockRequestHeaders(new Pair((Serializable)((Object)"Range"), (Serializable)((Object)"bytes=10-")));
        this.assertResponse(Arrays.copyOfRange(fileJsContents, 10, fileJsContents.length));
        this.assertResponseHeaders(new Pair((Serializable)((Object)"Accept-Ranges"), (Serializable)((Object)"bytes")), new Pair((Serializable)((Object)"Content-Range"), (Serializable)((Object)("bytes 10-15/" + fileJsContents.length))));
        Assert.assertEquals((long)6L, (long)this.responseContentLength.get());
        this.assertStatus(206);
    }

    @Test
    public void writeByteRangePastFileSize() throws IOException {
        this.makePathsAvailable(PATH_JS);
        this.mockRequestHeaders(new Pair((Serializable)((Object)"Range"), (Serializable)((Object)"bytes=10-100000")));
        this.assertResponse(Arrays.copyOfRange(fileJsContents, 10, fileJsContents.length));
        this.assertResponseHeaders(new Pair((Serializable)((Object)"Accept-Ranges"), (Serializable)((Object)"bytes")), new Pair((Serializable)((Object)"Content-Range"), (Serializable)((Object)("bytes 10-15/" + fileJsContents.length))));
        Assert.assertEquals((long)6L, (long)this.responseContentLength.get());
        this.assertStatus(206);
    }

    @Test
    public void writeByteRangeEmpty() throws IOException {
        this.makePathsAvailable(PATH_JS);
        this.mockRequestHeaders(new Pair((Serializable)((Object)"Range"), (Serializable)((Object)"bytes=10-9")));
        this.assertResponse(new byte[0]);
        this.assertStatus(416);
    }

    @Test
    public void writeByteRangeMalformed() throws IOException {
        this.makePathsAvailable(PATH_JS);
        this.mockRequestHeaders(new Pair((Serializable)((Object)"Range"), (Serializable)((Object)"f-d-d___")));
        this.assertResponse(new byte[0]);
        this.assertStatus(416);
    }

    @Test
    public void writeByteRangeBothEndsOpen() throws IOException {
        this.makePathsAvailable(PATH_JS);
        this.mockRequestHeaders(new Pair((Serializable)((Object)"Range"), (Serializable)((Object)"-")));
        this.assertResponse(new byte[0]);
        this.assertStatus(416);
    }

    @Test
    public void writeByteRangeMultiPartSequential() throws IOException {
        this.makePathsAvailable(PATH_JS);
        this.mockRequestHeaders(new Pair((Serializable)((Object)"Range"), (Serializable)((Object)"bytes=1-4, 5-6, 10-12")));
        this.assertMultipartResponse(PATH_JS, Arrays.asList(new Pair((Serializable)new String[]{"Content-Range: bytes 1-4/16"}, (Serializable)"ile.".getBytes()), new Pair((Serializable)new String[]{"Content-Range: bytes 5-6/16"}, (Serializable)"js".getBytes()), new Pair((Serializable)new String[]{"Content-Range: bytes 10-12/16"}, (Serializable)"nte".getBytes())));
        this.assertStatus(206);
    }

    @Test
    public void writeByteRangeMultiPartNonSequential() throws IOException {
        this.makePathsAvailable(PATH_JS);
        this.mockRequestHeaders(new Pair((Serializable)((Object)"Range"), (Serializable)((Object)"bytes=10-12, 1-4, 5-6")));
        this.assertMultipartResponse(PATH_JS, Arrays.asList(new Pair((Serializable)new String[]{"Content-Range: bytes 10-12/16"}, (Serializable)"nte".getBytes()), new Pair((Serializable)new String[]{"Content-Range: bytes 1-4/16"}, (Serializable)"ile.".getBytes()), new Pair((Serializable)new String[]{"Content-Range: bytes 5-6/16"}, (Serializable)"js".getBytes())));
        this.assertStatus(206);
    }

    @Test
    public void writeByteRangeMultiPartOverlapping() throws IOException {
        this.makePathsAvailable(PATH_JS);
        this.mockRequestHeaders(new Pair((Serializable)((Object)"Range"), (Serializable)((Object)"bytes=0-15, 1-4")));
        this.assertMultipartResponse(PATH_JS, Arrays.asList(new Pair((Serializable)new String[]{"Content-Range: bytes 0-15/16"}, (Serializable)"File.js contents".getBytes()), new Pair((Serializable)new String[]{"Content-Range: bytes 1-4/16"}, (Serializable)"ile.".getBytes())));
        this.assertStatus(206);
    }

    @Test
    public void writeByteRangeMultiPartTooManyRequested() throws IOException {
        this.makePathsAvailable(PATH_JS);
        this.mockRequestHeaders(new Pair((Serializable)((Object)"Range"), (Serializable)((Object)"bytes=0-0, 0-0, 1-1, 2-2, 3-3, 4-4, 5-5, 6-6, 7-7, 8-8, 9-9, 10-10, 11-11, 12-12, 13-13, 14-14, 15-15, 16-16")));
        this.assertMultipartResponse(PATH_JS, Arrays.asList(new Pair((Serializable)new String[]{"Content-Range: bytes 0-0/16"}, (Serializable)"F".getBytes()), new Pair((Serializable)new String[]{"Content-Range: bytes 0-0/16"}, (Serializable)"F".getBytes()), new Pair((Serializable)new String[]{"Content-Range: bytes 1-1/16"}, (Serializable)"i".getBytes()), new Pair((Serializable)new String[]{"Content-Range: bytes 2-2/16"}, (Serializable)"l".getBytes()), new Pair((Serializable)new String[]{"Content-Range: bytes 3-3/16"}, (Serializable)"e".getBytes()), new Pair((Serializable)new String[]{"Content-Range: bytes 4-4/16"}, (Serializable)".".getBytes()), new Pair((Serializable)new String[]{"Content-Range: bytes 5-5/16"}, (Serializable)"j".getBytes()), new Pair((Serializable)new String[]{"Content-Range: bytes 6-6/16"}, (Serializable)"s".getBytes()), new Pair((Serializable)new String[]{"Content-Range: bytes 7-7/16"}, (Serializable)" ".getBytes()), new Pair((Serializable)new String[]{"Content-Range: bytes 8-8/16"}, (Serializable)"c".getBytes()), new Pair((Serializable)new String[]{"Content-Range: bytes 9-9/16"}, (Serializable)"o".getBytes()), new Pair((Serializable)new String[]{"Content-Range: bytes 10-10/16"}, (Serializable)"n".getBytes()), new Pair((Serializable)new String[]{"Content-Range: bytes 11-11/16"}, (Serializable)"t".getBytes()), new Pair((Serializable)new String[]{"Content-Range: bytes 12-12/16"}, (Serializable)"e".getBytes()), new Pair((Serializable)new String[]{"Content-Range: bytes 13-13/16"}, (Serializable)"n".getBytes()), new Pair((Serializable)new String[]{"Content-Range: bytes 14-14/16"}, (Serializable)"t".getBytes())));
        this.assertStatus(206);
    }

    @Test
    public void writeByteRangeMultiPartTooManyOverlappingRequested() throws IOException {
        this.makePathsAvailable(PATH_JS);
        this.mockRequestHeaders(new Pair((Serializable)((Object)"Range"), (Serializable)((Object)"bytes=2-4, 0-4, 3-14")));
        this.assertMultipartResponse(PATH_JS, Arrays.asList(new Pair((Serializable)new String[]{"Content-Range: bytes 2-4/16"}, (Serializable)"le.".getBytes()), new Pair((Serializable)new String[]{"Content-Range: bytes 0-4/16"}, (Serializable)"File.".getBytes())));
        this.assertStatus(206);
    }

    private void assertResponse(byte[] expectedResponse) throws IOException {
        this.assertResponse(PATH_JS, expectedResponse);
    }

    private void assertResponse(String path, byte[] expectedResponse) throws IOException {
        CapturingServletOutputStream out = new CapturingServletOutputStream();
        Mockito.when((Object)this.response.getOutputStream()).thenReturn((Object)out);
        this.responseWriter.writeResponseContents(path, pathToUrl.get(path), this.request, this.response);
        Assert.assertArrayEquals((byte[])expectedResponse, (byte[])out.getOutput());
        Assert.assertEquals((long)expectedResponse.length, (long)this.responseContentLength.get());
    }

    private void assertMultipartResponse(String path, List<Pair<String[], byte[]>> expectedHeadersAndBytes) throws IOException {
        CapturingServletOutputStream out = new CapturingServletOutputStream();
        Mockito.when((Object)this.response.getOutputStream()).thenReturn((Object)out);
        AtomicReference<Object> contentType = new AtomicReference<Object>(null);
        ((HttpServletResponse)Mockito.doAnswer(invocation -> {
            contentType.set(((String)invocation.getArguments()[0]));
            return null;
        }).when((Object)this.response)).setContentType(ArgumentMatchers.anyString());
        this.responseWriter.writeResponseContents(path, pathToUrl.get(path), this.request, this.response);
        byte[] output = out.getOutput();
        Assert.assertNotNull(contentType.get());
        Assert.assertTrue((boolean)((String)contentType.get()).startsWith("multipart/byteranges; boundary="));
        String boundary = ((String)contentType.get()).substring(((String)contentType.get()).indexOf("=") + 1);
        SimpleMultipartParser parser = new SimpleMultipartParser(output, boundary);
        for (Pair<String[], byte[]> expected : expectedHeadersAndBytes) {
            String[] expectedHeaders = (String[])expected.getFirst();
            String actualHeaders = parser.readHeaders();
            for (String expectedHeader : expectedHeaders) {
                Assert.assertTrue((String)String.format("Headers:\n%s\ndid not contain:\n%s", actualHeaders, expectedHeader), (boolean)actualHeaders.contains(expectedHeader));
            }
            byte[] expectedBytes = (byte[])expected.getSecond();
            ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
            parser.readBodyData(outputStream);
            byte[] bytes = outputStream.toByteArray();
            Assert.assertArrayEquals((byte[])expectedBytes, (byte[])bytes);
        }
        try {
            parser.readHeaders();
            ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
            parser.readBodyData(outputStream);
            Assert.assertTrue((String)"excess bytes in multipart response", (outputStream.toByteArray().length == 0 ? 1 : 0) != 0);
        }
        catch (IOException iOException) {
            // empty catch block
        }
    }

    private void assertStatus(int status) {
        ((HttpServletResponse)Mockito.verify((Object)this.response)).setStatus(status);
    }

    private void makePathsAvailable(String ... paths) throws MalformedURLException {
        for (String path : paths) {
            URL url = pathToUrl.get(path);
            if (url == null) {
                throw new IllegalArgumentException("Unsupported path: " + path);
            }
            Mockito.when((Object)this.servletContext.getResource(path)).thenReturn((Object)url);
        }
    }

    private void makeClassPathAvailable(String ... paths) {
        for (String path : paths) {
            URL url = pathToUrl.get(path);
            if (url == null) {
                throw new IllegalArgumentException("Unsupported path: " + path);
            }
            ClassLoader classLoader = (ClassLoader)Mockito.mock(ClassLoader.class);
            Mockito.when((Object)this.servletContext.getClassLoader()).thenReturn((Object)classLoader);
            Mockito.when((Object)classLoader.getResource("META-INF/VAADIN/webapp" + path)).thenReturn((Object)url);
        }
    }

    @SafeVarargs
    private final void assertResponseHeaders(Pair<String, String> ... headers) {
        for (Pair<String, String> header : headers) {
            ((HttpServletResponse)Mockito.verify((Object)this.response)).setHeader((String)((Object)header.getFirst()), (String)((Object)header.getSecond()));
        }
    }

    @SafeVarargs
    private final void mockRequestHeaders(Pair<String, String> ... headers) {
        for (Pair<String, String> header : headers) {
            Mockito.when((Object)this.request.getHeader((String)((Object)header.getFirst()))).thenReturn((Object)((String)((Object)header.getSecond())));
            Mockito.when((Object)this.request.getHeaders((String)((Object)header.getFirst()))).thenReturn(Collections.enumeration(Collections.singleton((String)((Object)header.getSecond()))));
        }
        Mockito.when((Object)this.request.getHeaderNames()).thenReturn(Collections.enumeration(Arrays.stream(headers).map(Pair::getFirst).collect(Collectors.toList())));
    }

    private static byte[] gzip(byte[] input) {
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        try (GZIPOutputStream stream = new GZIPOutputStream(baos);){
            stream.write(input);
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
        return baos.toByteArray();
    }

    private static URL createFileURLWithDataAndLength(String name, byte[] data) {
        return ResponseWriterTest.createFileURLWithDataAndLength(name, data, -1L);
    }

    private static URL createFileURLWithDataAndLength(String name, final byte[] data, final long lastModificationTime) {
        try {
            return new URL("file", "", -1, name, new URLStreamHandler(){

                @Override
                protected URLConnection openConnection(URL u) throws IOException {
                    URLConnection connection = (URLConnection)Mockito.mock(URLConnection.class);
                    Mockito.when((Object)connection.getInputStream()).thenReturn((Object)new ByteArrayInputStream(data));
                    Mockito.when((Object)connection.getContentLengthLong()).thenReturn((Object)data.length);
                    Mockito.when((Object)connection.getLastModified()).thenReturn((Object)lastModificationTime);
                    return connection;
                }
            });
        }
        catch (MalformedURLException e) {
            throw new UncheckedIOException(e);
        }
    }

    static {
        pathToUrl.put(PATH_JS, ResponseWriterTest.createFileURLWithDataAndLength(PATH_JS, fileJsContents));
        pathToUrl.put(PATH_GZ, ResponseWriterTest.createFileURLWithDataAndLength(PATH_GZ, fileJsGzippedContents));
        pathToUrl.put(PATH_BR, ResponseWriterTest.createFileURLWithDataAndLength(PATH_BR, fileJsBrotliContents));
        pathToUrl.put(CLASS_PATH_JS, ResponseWriterTest.createFileURLWithDataAndLength(CLASS_PATH_JS, fileJsContents));
        pathToUrl.put(CLASS_PATH_GZ, ResponseWriterTest.createFileURLWithDataAndLength(CLASS_PATH_GZ, fileJsGzippedContents));
        pathToUrl.put(FAULTY_CLASS_PATH_JS, ResponseWriterTest.createFileURLWithDataAndLength(FAULTY_CLASS_PATH_JS, fileJsContents));
        pathToUrl.put(FAULTY_CLASS_PATH_GZ, ResponseWriterTest.createFileURLWithDataAndLength(FAULTY_CLASS_PATH_GZ, fileJsGzippedContents));
    }

    private static class OverrideableResponseWriter
    extends ResponseWriter {
        private Boolean overrideAcceptsGzippedResource;
        private Boolean overrideAcceptsBrotliResource;

        public OverrideableResponseWriter(DeploymentConfiguration deploymentConfiguration) {
            super(deploymentConfiguration);
        }

        protected boolean acceptsGzippedResource(HttpServletRequest request) {
            if (this.overrideAcceptsGzippedResource != null) {
                return this.overrideAcceptsGzippedResource;
            }
            return super.acceptsGzippedResource(request);
        }

        protected boolean acceptsBrotliResource(HttpServletRequest request) {
            if (this.overrideAcceptsBrotliResource != null) {
                return this.overrideAcceptsBrotliResource;
            }
            return super.acceptsBrotliResource(request);
        }
    }

    public static class CapturingServletOutputStream
    extends ServletOutputStream {
        ByteArrayOutputStream baos = new ByteArrayOutputStream();

        public void write(int b) throws IOException {
            this.baos.write(b);
        }

        public void setWriteListener(WriteListener writeListener) {
        }

        public boolean isReady() {
            return true;
        }

        public byte[] getOutput() {
            return this.baos.toByteArray();
        }
    }
}

