/*
 * Decompiled with CFR 0.152.
 */
package com.vaadin.flow.internal;

import com.vaadin.flow.internal.CssBundler;
import com.vaadin.flow.internal.JacksonUtils;
import java.io.File;
import java.io.IOException;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.attribute.FileAttribute;
import org.apache.commons.io.FileUtils;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;
import tools.jackson.databind.JsonNode;
import tools.jackson.databind.ObjectMapper;
import tools.jackson.databind.node.ObjectNode;

public class CssBundlerTest {
    private static final String TEST_CSS = "body {background: blue};";
    private File themesFolder;
    private File themeFolder;

    @Before
    public void setup() throws IOException {
        this.themesFolder = Files.createTempDirectory("cssbundlertest", new FileAttribute[0]).toFile();
        this.themeFolder = new File(this.themesFolder, "my-theme");
    }

    @Test
    public void differentImportSyntaxesSupported() throws Exception {
        String[] validImports;
        for (String valid : validImports = new String[]{"@import url(foo.css);", "@import url(foo.css?ts=1234);", "@import url('foo.css');", "@import url(\"foo.css\");", "@import 'foo.css';", "@import \"foo.css\";", "@import url(foo.css);", "  @import    url  (  '  foo.css  '  )  ;   ", "   @import   '  foo.css   '   ;"}) {
            this.assertImportWorks(valid);
        }
    }

    @Test
    public void layerImportsNotHandled() throws IOException {
        this.assertImportNotHandled("@import url('foo.css') layer(foo);");
        this.assertImportNotHandled("@import url('foo.css') layer(foo) ;");
        this.assertImportNotHandled("@import 'theme.css' layer(utilities);");
        this.assertImportNotHandled("@import \"theme.css\" layer();");
        this.assertImportNotHandled("@import \"style.css\" layer;");
        this.assertImportNotHandled("@import \"style.css\" layer print;");
    }

    @Test
    public void conditionalImportsNotHandled() throws IOException {
        this.assertImportNotHandled("@import url('foo.css') print;");
        this.assertImportNotHandled("@import url('bluish.css') print, screen;");
        this.assertImportNotHandled("@import \"common.css\" screen;");
        this.assertImportNotHandled("@import url('landscape.css') screen and (orientation: landscape);");
    }

    @Test
    public void relativeUrlsRewritten() throws IOException {
        this.writeCss("background-image: url('foo/bar.png');", "styles.css");
        this.createThemeFile("foo/bar.png");
        Assert.assertEquals((Object)"background-image: url('VAADIN/themes/my-theme/foo/bar.png');", (Object)CssBundler.inlineImportsForThemes((File)this.themeFolder, (File)this.getThemeFile("styles.css"), (JsonNode)this.getThemeJson()));
        this.writeCss("background-image: url(\"foo/bar.png\");", "styles.css");
        Assert.assertEquals((Object)"background-image: url('VAADIN/themes/my-theme/foo/bar.png');", (Object)CssBundler.inlineImportsForThemes((File)this.themeFolder, (File)this.getThemeFile("styles.css"), (JsonNode)this.getThemeJson()));
        this.writeCss("background-image: url(foo/bar.png);", "styles.css");
        Assert.assertEquals((Object)"background-image: url('VAADIN/themes/my-theme/foo/bar.png');", (Object)CssBundler.inlineImportsForThemes((File)this.themeFolder, (File)this.getThemeFile("styles.css"), (JsonNode)this.getThemeJson()));
    }

    @Test
    public void relativeUrlsWithExtraInfoRewritten() throws IOException {
        this.writeCss("@font-face {\n    font-family: \"Ostrich\";\n    src: url(\"./fonts/ostrich-sans-regular.ttf\") format(\"TrueType\");\n}\n", "styles.css");
        this.createThemeFile("fonts/ostrich-sans-regular.ttf");
        Assert.assertEquals((Object)"                           @font-face {\n    font-family: \"Ostrich\";\n    src: url('VAADIN/themes/my-theme/fonts/ostrich-sans-regular.ttf') format(\"TrueType\");\n}".trim(), (Object)CssBundler.inlineImportsForThemes((File)this.themeFolder, (File)this.getThemeFile("styles.css"), (JsonNode)this.getThemeJson()).trim());
    }

    @Test
    public void relativeUrlsInSubFolderRewritten() throws IOException {
        this.writeCss("@import url('sub/sub.css');", "styles.css");
        this.writeCss("background-image: url('./file.png');", "sub/sub.css");
        this.createThemeFile("sub/file.png");
        Assert.assertEquals((Object)"background-image: url('VAADIN/themes/my-theme/sub/file.png');", (Object)CssBundler.inlineImportsForThemes((File)this.themeFolder, (File)this.getThemeFile("styles.css"), (JsonNode)this.getThemeJson()));
    }

    @Test
    public void dollarAndBackslashWorks() throws IOException {
        String css = "body { content: '$\\'}";
        this.writeCss("@import 'other.css';", "styles.css");
        this.writeCss(css, "other.css");
        Assert.assertEquals((Object)"body { content: '$\\'}", (Object)CssBundler.inlineImportsForThemes((File)this.themeFolder, (File)this.getThemeFile("styles.css"), (JsonNode)this.getThemeJson()));
    }

    @Test
    public void unhandledImportsAreMovedToTop() throws IOException {
        this.writeCss(TEST_CSS, "other.css");
        this.writeCss("@import url('https://cdn.jsdelivr.net/fontsource/css/inter@latest/index.css');\n@import url('other.css');\n@import url('https://cdn.jsdelivr.net/fontsource/css/aclonica@latest/index.css');\n@import url('https://cdn.jsdelivr.net/fontsource/css/aclonica@latest/index.css?ts=1234');\n@import url('https://cdn.jsdelivr.net/fontsource/css/aclonica@latest/index.css#foo');\n@import url('foo.css') layer(foo);\n@import url('bluish.css') print, screen;\n@import url('landscape.css') screen and (orientation: landscape);\n", "styles.css");
        Assert.assertEquals((Object)"@import url('https://cdn.jsdelivr.net/fontsource/css/inter@latest/index.css');\n@import url('https://cdn.jsdelivr.net/fontsource/css/aclonica@latest/index.css');\n@import url('https://cdn.jsdelivr.net/fontsource/css/aclonica@latest/index.css?ts=1234');\n@import url('https://cdn.jsdelivr.net/fontsource/css/aclonica@latest/index.css#foo');\n@import url('foo.css') layer(foo);\n@import url('bluish.css') print, screen;\n@import url('landscape.css') screen and (orientation: landscape);\n\nbody {background: blue};\n".trim(), (Object)CssBundler.inlineImportsForThemes((File)this.themeFolder, (File)this.getThemeFile("styles.css"), (JsonNode)this.getThemeJson()).trim());
    }

    @Test
    public void themeAssetsRelativeUrlsRewritten() throws IOException {
        this.createThemeJson("{\n  \"assets\": {\n    \"@some/pkg\": {\n      \"svgs/regular/**\": \"my/icons\"\n    }\n  }\n}\n");
        this.writeCss("background-image: url('my/icons/file1.png');\nbackground-image: url('./my/icons/file2.png');\nbackground-image: url('../my/icons/file3.png');\n", "styles.css");
        Assert.assertEquals((Object)"background-image: url('VAADIN/themes/my-theme/my/icons/file1.png');\nbackground-image: url('VAADIN/themes/my-theme/my/icons/file2.png');\nbackground-image: url('../my/icons/file3.png');\n", (Object)CssBundler.inlineImportsForThemes((File)this.themeFolder, (File)this.getThemeFile("styles.css"), (JsonNode)this.getThemeJson()));
    }

    @Test
    public void themeAssetsRelativeUrlsInSubFolderRewritten() throws IOException {
        this.createThemeJson("{\n  \"assets\": {\n    \"@some/pkg\": {\n      \"svgs/regular/**\": \"my/icons\"\n    }\n  }\n}\n");
        this.writeCss("@import url('sub/sub.css');\n@import url('sub/nested/two.css');\n", "styles.css");
        this.writeCss("@import url('nested/one.css');\nbackground-image: url('../my/icons/file.png');\n", "sub/sub.css");
        this.writeCss("background-image: url('../../my/icons/file1.png');", "sub/nested/one.css");
        this.writeCss("background-image: url('../../my/icons/file2.png');", "sub/nested/two.css");
        String actualCss = CssBundler.inlineImportsForThemes((File)this.themeFolder, (File)this.getThemeFile("styles.css"), (JsonNode)this.getThemeJson());
        Assert.assertEquals((Object)"background-image: url('VAADIN/themes/my-theme/my/icons/file1.png');\nbackground-image: url('VAADIN/themes/my-theme/my/icons/file.png');\n\nbackground-image: url('VAADIN/themes/my-theme/my/icons/file2.png');\n", (Object)actualCss);
    }

    @Test
    public void relativeUrl_notThemeResourceNotAssets_notRewritten() throws IOException {
        this.createThemeJson("{\n  \"assets\": {\n    \"@some/pkg\": {\n      \"svgs/regular/**\": \"my/icons\"\n    }\n  }\n}\n");
        this.writeCss("@import url('sub/sub.css');\n@import url('sub/nested/two.css');\nbackground-image: url('unknown/icons/file-root.png');\n", "styles.css");
        this.writeCss("@import url('nested/one.css');\nbackground-image: url('../unknown/icons/file-sub.png');\n", "sub/sub.css");
        this.writeCss("background-image: url('../../unknown/icons/file1.png');", "sub/nested/one.css");
        this.writeCss("background-image: url('../../unknown/icons/file2.png');", "sub/nested/two.css");
        String actualCss = CssBundler.inlineImportsForThemes((File)this.themeFolder, (File)this.getThemeFile("styles.css"), (JsonNode)this.getThemeJson());
        Assert.assertEquals((Object)"background-image: url('../../unknown/icons/file1.png');\nbackground-image: url('../unknown/icons/file-sub.png');\n\nbackground-image: url('../../unknown/icons/file2.png');\nbackground-image: url('unknown/icons/file-root.png');\n", (Object)actualCss);
    }

    @Test
    public void ignoreCommentedRules() throws Exception {
        File cssFile = this.writeCss("/*\n@import url('a.css');\n*/\n@import url('b.css');\n/*@import url('c.css');*/\n@import url('d.css');\n", "styles.css");
        String output = CssBundler.inlineImportsForPublicResources((File)cssFile.getParentFile(), (File)cssFile, null);
        Assert.assertFalse((boolean)output.contains("a.css"));
        Assert.assertTrue((boolean)output.contains("b.css"));
        Assert.assertFalse((boolean)output.contains("c.css"));
        Assert.assertTrue((boolean)output.contains("d.css"));
    }

    private boolean createThemeFile(String filename) throws IOException {
        File f = this.getThemeFile(filename);
        f.getParentFile().mkdirs();
        return f.createNewFile();
    }

    private File getThemeFile(String filename) {
        return new File(this.themeFolder, filename);
    }

    private void assertImportWorks(String importCss) throws IOException {
        File f = this.writeFileWithImport(importCss, "foo.css");
        Assert.assertEquals((String)importCss, (Object)TEST_CSS.trim(), (Object)CssBundler.inlineImportsForThemes((File)f.getParentFile(), (File)f, (JsonNode)new ObjectMapper().createArrayNode()).trim());
    }

    private void assertImportNotHandled(String importCss) throws IOException {
        File f = this.writeFileWithImport(importCss, "foo.css");
        Assert.assertEquals((Object)importCss, (Object)CssBundler.inlineImportsForThemes((File)f.getParentFile(), (File)f, (JsonNode)new ObjectMapper().createArrayNode()));
    }

    private File writeFileWithImport(String css, String otherFilename) throws IOException {
        this.writeCss(TEST_CSS, otherFilename);
        return this.writeCss(css, "styles.css");
    }

    private File writeCss(String css, String filename) throws IOException {
        File file = this.getThemeFile(filename);
        FileUtils.writeStringToFile((File)file, (String)css, (Charset)StandardCharsets.UTF_8);
        return file;
    }

    private JsonNode getThemeJson() throws IOException {
        File file = this.getThemeFile("theme.json");
        if (file.exists()) {
            return JacksonUtils.readTree((String)Files.readString(file.toPath()));
        }
        return null;
    }

    private void createThemeJson(String json) throws IOException {
        ObjectNode jsonObject = JacksonUtils.readTree((String)json);
        File file = this.getThemeFile("theme.json");
        FileUtils.writeStringToFile((File)file, (String)jsonObject.toString(), (Charset)StandardCharsets.UTF_8);
    }

    @Test
    public void minifyCss_removesComments() {
        String css = "/* comment */ .class { color: red; }";
        String result = CssBundler.minifyCss((String)css);
        Assert.assertEquals((Object)".class{color: red}", (Object)result);
    }

    @Test
    public void minifyCss_doesNotRemoveContentComment() {
        String css = ".selector {\n  content: \"/* not a comment, should not remove */\";\n}\n";
        String result = CssBundler.minifyCss((String)css);
        Assert.assertEquals((Object)".selector{content: \"/* not a comment,should not remove */\"}", (Object)result);
    }

    @Test
    public void minifyCss_removesMultilineComments() {
        String css = "/* This is a\n   multiline comment */\n.class { color: red; }\n";
        String result = CssBundler.minifyCss((String)css);
        Assert.assertEquals((Object)".class{color: red}", (Object)result);
    }

    @Test
    public void minifyCss_collapsesWhitespace() {
        String css = ".class   {   color:   red;   }";
        String result = CssBundler.minifyCss((String)css);
        Assert.assertEquals((Object)".class{color: red}", (Object)result);
    }

    @Test
    public void minifyCss_removesTrailingSemicolons() {
        String css = ".class { color: red; }";
        String result = CssBundler.minifyCss((String)css);
        Assert.assertEquals((Object)".class{color: red}", (Object)result);
    }

    @Test
    public void minifyCss_handlesMultipleRules() {
        String css = ".class1 { color: red; }\n.class2 { background: blue; }\n";
        String result = CssBundler.minifyCss((String)css);
        Assert.assertEquals((Object)".class1{color: red}.class2{background: blue}", (Object)result);
    }

    @Test
    public void minifyCss_handlesCalc_noRemovalOfWhitespace() {
        String css = "span.test::before {\n    content: \"\";\n    height: calc(100% + 6px);\n    width: calc(10px - 100px);\n    length: calc(var(--variable-width) + 20px);\n    size: calc(2em * 5);\n    border-left: 1px solid red;\n    border: var(--bs-border-width) solid var(--bs-border-color);\n}\n";
        String result = CssBundler.minifyCss((String)css);
        Assert.assertEquals((Object)"span.test::before{content: \"\";height: calc(100% + 6px);width: calc(10px - 100px);length: calc(var(--variable-width) + 20px);size: calc(2em * 5);border-left: 1px solid red;border: var(--bs-border-width) solid var(--bs-border-color)}", (Object)result);
    }

    @Test
    public void minifyCss_preservesSelectorsWithCombinators() {
        String css = ".parent > .child { color: red; }";
        String result = CssBundler.minifyCss((String)css);
        Assert.assertEquals((Object)".parent>.child{color: red}", (Object)result);
    }

    @Test
    public void minifyCss_keyframes() {
        String content = "@-webkit-keyframes configuratorFlyLeft {\n\t0% {left: 0;}\n\t100% {left: -100%;}\n }\n\n @-webkit-keyframes configuratorFlyRight {\n\t0% {right: 0;}\n\t100% {right: -100%;}\n}\n";
        Assert.assertEquals((Object)"@-webkit-keyframes configuratorFlyLeft{0%{left: 0}100%{left: -100%}}@-webkit-keyframes configuratorFlyRight{0%{right: 0}100%{right: -100%}}", (Object)CssBundler.minifyCss((String)content));
    }

    @Test
    public void minifyCss_CSSMediaQueriesHandled() {
        String content = "@media (max-width: 1200px) {\n    legend {\n      font-size: calc(1.275rem + 0.3vw);\n    }\n}\n";
        Assert.assertEquals((Object)"@media (max-width: 1200px){legend{font-size: calc(1.275rem + 0.3vw)}}", (Object)CssBundler.minifyCss((String)content));
    }

    @Test
    public void minifyCss_handlesEmptyInput() {
        Assert.assertEquals((Object)"", (Object)CssBundler.minifyCss((String)""));
        Assert.assertEquals((Object)"", (Object)CssBundler.minifyCss((String)"   "));
        Assert.assertEquals((Object)"", (Object)CssBundler.minifyCss((String)"/* only comment */"));
    }

    @Test
    public void inlineImports_resolvesNodeModulesImport() throws IOException {
        File nodeModules = new File(this.themesFolder, "node_modules");
        File packageDir = new File(nodeModules, "some-package");
        packageDir.mkdirs();
        File nodeModulesCss = new File(packageDir, "styles.css");
        FileUtils.writeStringToFile((File)nodeModulesCss, (String)".from-node-modules { color: blue; }", (Charset)StandardCharsets.UTF_8);
        this.writeCss("@import 'some-package/styles.css';\n.main { color: red; }", "styles.css");
        String result = CssBundler.inlineImports((File)this.themeFolder, (File)this.getThemeFile("styles.css"), null, (File)nodeModules);
        Assert.assertTrue((String)"Should start with node_modules CSS inlined", (boolean)result.startsWith(".from-node-modules { color: blue; }"));
        Assert.assertTrue((String)"Should contain main CSS", (boolean)result.contains(".main { color: red; }"));
    }

    @Test
    public void inlineImports_prefersRelativeOverNodeModules() throws IOException {
        File nodeModules = new File(this.themesFolder, "node_modules");
        File packageDir = new File(nodeModules, "local");
        packageDir.mkdirs();
        File nodeModulesCss = new File(packageDir, "styles.css");
        FileUtils.writeStringToFile((File)nodeModulesCss, (String)".from-node-modules { color: blue; }", (Charset)StandardCharsets.UTF_8);
        File localDir = new File(this.themeFolder, "local");
        localDir.mkdirs();
        File localCss = new File(localDir, "styles.css");
        FileUtils.writeStringToFile((File)localCss, (String)".from-local { color: green; }", (Charset)StandardCharsets.UTF_8);
        this.writeCss("@import 'local/styles.css';\n.main { color: red; }", "styles.css");
        String result = CssBundler.inlineImports((File)this.themeFolder, (File)this.getThemeFile("styles.css"), null, (File)nodeModules);
        Assert.assertTrue((String)"Should start with local CSS inlined", (boolean)result.startsWith(".from-local { color: green; }"));
        Assert.assertFalse((String)"Should not contain node_modules CSS", (boolean)result.contains(".from-node-modules"));
    }

    @Test
    public void inlineImports_handlesNullNodeModulesFolder() throws IOException {
        this.writeCss("@import 'nonexistent/styles.css';\n.main { color: red; }", "styles.css");
        String result = CssBundler.inlineImports((File)this.themeFolder, (File)this.getThemeFile("styles.css"), null, null);
        Assert.assertTrue((String)"Should preserve unresolved import", (boolean)result.contains("@import 'nonexistent/styles.css'"));
        Assert.assertTrue((String)"Should contain main CSS", (boolean)result.contains(".main { color: red; }"));
    }
}

