/*
 * Copyright 2000-2025 Vaadin Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package com.vaadin.flow.server.streams;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.UncheckedIOException;

/**
 * Abstract class for file upload handler.
 *
 * @param <R>
 *            type of file upload handler
 *
 * @since 24.8
 */
public abstract class AbstractFileUploadHandler<R extends AbstractFileUploadHandler>
        extends TransferProgressAwareHandler<UploadEvent, R>
        implements UploadHandler {
    private final FileUploadCallback successCallback;
    private final FileFactory fileFactory;

    /**
     * Creates an upload callback that stores the incoming stream into a
     * {@link File} that is generated by the given {@link FileFactory}.
     *
     * @param successCallback
     *            consumer to be called for successful upload
     * @param fileFactory
     *            factory that generates the {@link File} to store data
     */
    public AbstractFileUploadHandler(FileUploadCallback successCallback,
            FileFactory fileFactory) {
        this.successCallback = successCallback;
        this.fileFactory = fileFactory;
    }

    @Override
    public void handleUploadRequest(UploadEvent event) throws IOException {
        UploadMetadata metadata = new UploadMetadata(event.getFileName(),
                event.getContentType(), event.getFileSize());
        setTransferUI(event.getUI());
        File file;
        try {
            file = fileFactory.createFile(metadata);
            try (InputStream inputStream = event.getInputStream();
                    FileOutputStream outputStream = new FileOutputStream(
                            file)) {
                TransferUtil.transfer(inputStream, outputStream,
                        getTransferContext(event), getListeners());
            }
        } catch (IOException e) {
            notifyError(event, e);
            throw e;
        }
        event.getUI().access(() -> {
            try {
                successCallback.complete(metadata, file);
            } catch (IOException e) {
                throw new UncheckedIOException("Error in file upload callback",
                        e);
            }
        });
    }

    @Override
    protected TransferContext getTransferContext(UploadEvent transferEvent) {
        return new TransferContext(transferEvent.getRequest(),
                transferEvent.getResponse(), transferEvent.getSession(),
                transferEvent.getFileName(), transferEvent.getOwningElement(),
                transferEvent.getFileSize());
    }
}
