/*
 * Decompiled with CFR 0.152.
 */
package com.vaadin.flow.server.frontend;

import com.vaadin.flow.server.frontend.FrontendTools;
import com.vaadin.flow.server.frontend.FrontendToolsLocator;
import com.vaadin.flow.server.frontend.FrontendUtils;
import com.vaadin.flow.server.frontend.FrontendVersion;
import com.vaadin.flow.server.frontend.installer.InstallationException;
import com.vaadin.flow.server.frontend.installer.NodeInstaller;
import com.vaadin.flow.server.frontend.installer.ProxyConfig;
import java.io.File;
import java.io.Serializable;
import java.net.URI;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class NodeResolver
implements Serializable {
    private final FrontendToolsLocator frontendToolsLocator = new FrontendToolsLocator();
    private final String alternativeDir;
    private final String nodeVersion;
    private final URI nodeDownloadRoot;
    private final boolean forceAlternativeNode;
    private final List<ProxyConfig.Proxy> proxies;
    private final String nodeFolder;

    NodeResolver(String alternativeDir, String nodeVersion, URI nodeDownloadRoot, boolean forceAlternativeNode, List<ProxyConfig.Proxy> proxies, String nodeFolder) {
        this.alternativeDir = Objects.requireNonNull(alternativeDir);
        this.nodeVersion = Objects.requireNonNull(nodeVersion);
        this.nodeDownloadRoot = Objects.requireNonNull(nodeDownloadRoot);
        this.forceAlternativeNode = forceAlternativeNode;
        this.proxies = Objects.requireNonNull(proxies);
        this.nodeFolder = nodeFolder;
    }

    ActiveNodeInstallation resolve() {
        ActiveNodeInstallation globalInstallation;
        if (this.nodeFolder != null && !this.nodeFolder.trim().isEmpty()) {
            return this.resolveFromSpecifiedFolder();
        }
        if (!this.forceAlternativeNode && (globalInstallation = this.tryUseGlobalNode()) != null) {
            return globalInstallation;
        }
        return this.resolveOrInstallAlternativeNode();
    }

    private ActiveNodeInstallation tryUseGlobalNode() {
        String nodeCommand = FrontendUtils.isWindows() ? "node.exe" : "node";
        File nodeExecutable = this.frontendToolsLocator.tryLocateTool(nodeCommand).orElse(null);
        if (nodeExecutable == null) {
            return null;
        }
        try {
            ActiveNodeInstallation installation = this.tryResolveFromExecutable(nodeExecutable, nodeExecutable.getParentFile(), null);
            if (installation != null) {
                FrontendVersion version = new FrontendVersion(installation.nodeVersion());
                if (version.isOlderThan(FrontendTools.SUPPORTED_NODE_VERSION)) {
                    this.getLogger().info("The globally installed Node.js version {} is older than the required minimum version {}. Using Node.js from {}.", new Object[]{installation.nodeVersion(), FrontendTools.SUPPORTED_NODE_VERSION.getFullVersion(), this.alternativeDir});
                    return null;
                }
                if (version.getMajorVersion() > 24) {
                    this.getLogger().info("The globally installed Node.js version {}.x is newer than the maximum supported version {}.x and may not be compatible. Using Node.js from {}.", new Object[]{version.getMajorVersion(), 24, this.alternativeDir});
                    return null;
                }
                this.getLogger().info("Using globally installed Node.js version {}", (Object)installation.nodeVersion());
            }
            return installation;
        }
        catch (FrontendUtils.UnknownVersionException e) {
            this.getLogger().error("Failed to get version for installed node.", (Throwable)e);
            return null;
        }
    }

    private ActiveNodeInstallation resolveFromSpecifiedFolder() {
        File nodeFolderFile = new File(this.nodeFolder);
        if (!nodeFolderFile.exists()) {
            throw new IllegalStateException(String.format("Configured node folder does not exist: %s", nodeFolderFile.getAbsolutePath()));
        }
        if (!nodeFolderFile.isDirectory()) {
            throw new IllegalStateException(String.format("Configured node folder is not a directory: %s", nodeFolderFile.getAbsolutePath()));
        }
        String nodeCommand = FrontendUtils.isWindows() ? "node.exe" : "node";
        File nodeExecutable = new File(nodeFolderFile, nodeCommand);
        if (!nodeExecutable.exists()) {
            throw new IllegalStateException(String.format("Node binary '%s' not found in configured folder: %s", nodeCommand, nodeFolderFile.getAbsolutePath()));
        }
        if (!nodeExecutable.canExecute()) {
            throw new IllegalStateException(String.format("Node binary is not executable: %s", nodeExecutable.getAbsolutePath()));
        }
        try {
            ActiveNodeInstallation installation = this.tryResolveFromExecutable(nodeExecutable, nodeFolderFile, null);
            if (installation == null) {
                throw new IllegalStateException(String.format("npm-cli.js not found in configured node folder: %s", nodeFolderFile.getAbsolutePath()));
            }
            this.getLogger().info("Using Node.js from configured folder: {} (version {})", (Object)nodeFolderFile.getAbsolutePath(), (Object)installation.nodeVersion());
            return installation;
        }
        catch (FrontendUtils.UnknownVersionException e) {
            throw new IllegalStateException(String.format("Failed to get version of node from configured folder: %s", nodeFolderFile.getAbsolutePath()), e);
        }
    }

    private ActiveNodeInstallation resolveOrInstallAlternativeNode() {
        String fallbackVersion;
        String versionToUse;
        File nodeExecutable;
        File alternativeDirFile = new File(this.alternativeDir);
        NodeInstaller nodeInstaller = new NodeInstaller(alternativeDirFile, this.proxies);
        if (this.nodeDownloadRoot != null) {
            nodeInstaller.setNodeDownloadRoot(this.nodeDownloadRoot);
        }
        if ((nodeExecutable = this.getNodeExecutableForVersion(alternativeDirFile, versionToUse = this.nodeVersion)).exists()) {
            try {
                String installedVersion = FrontendUtils.getVersion("node", List.of(nodeExecutable.getAbsolutePath(), "--version")).getFullVersion();
                String normalizedInstalled = NodeResolver.normalizeVersion(installedVersion);
                String normalizedRequested = NodeResolver.normalizeVersion(this.nodeVersion);
                if (normalizedInstalled.equals(normalizedRequested)) {
                    this.getLogger().info("Node {} is already installed in {}", (Object)this.nodeVersion, (Object)this.alternativeDir);
                    return this.createActiveInstallation(nodeExecutable, versionToUse, alternativeDirFile);
                }
            }
            catch (FrontendUtils.UnknownVersionException e) {
                this.getLogger().debug("Could not verify version of existing node installation", (Throwable)e);
            }
        }
        if ((fallbackVersion = this.findCompatibleInstalledVersion(alternativeDirFile)) != null) {
            this.getLogger().debug("Using existing Node {} instead of installing {}", (Object)fallbackVersion, (Object)this.nodeVersion);
            versionToUse = fallbackVersion;
            nodeExecutable = this.getNodeExecutableForVersion(alternativeDirFile, versionToUse);
            try {
                String installedVersion = FrontendUtils.getVersion("node", List.of(nodeExecutable.getAbsolutePath(), "--version")).getFullVersion();
                String normalizedInstalled = NodeResolver.normalizeVersion(installedVersion);
                String normalizedRequested = NodeResolver.normalizeVersion(this.nodeVersion);
                if (normalizedInstalled.equals(normalizedRequested)) {
                    return this.createActiveInstallation(nodeExecutable, versionToUse, alternativeDirFile);
                }
            }
            catch (FrontendUtils.UnknownVersionException e) {
                this.getLogger().debug("Could not verify version of existing node installation", (Throwable)e);
            }
        }
        this.getLogger().info("Installing Node {} to {}", (Object)this.nodeVersion, (Object)this.alternativeDir);
        try {
            nodeInstaller.setNodeVersion(this.nodeVersion);
            nodeInstaller.install();
            nodeExecutable = this.getNodeExecutableForVersion(alternativeDirFile, this.nodeVersion);
            return this.createActiveInstallation(nodeExecutable, this.nodeVersion, alternativeDirFile);
        }
        catch (InstallationException e) {
            throw new IllegalStateException("Failed to install Node", e);
        }
    }

    private ActiveNodeInstallation createActiveInstallation(File nodeExecutable, String version, File installDir) {
        if (!nodeExecutable.exists()) {
            throw new IllegalStateException("Node installation failed - executable not found at " + String.valueOf(nodeExecutable));
        }
        try {
            ActiveNodeInstallation installation = this.tryResolveFromExecutable(nodeExecutable, installDir, version);
            if (installation == null) {
                String versionedPath = "node-v" + NodeResolver.normalizeVersion(version);
                boolean isWindows = FrontendUtils.isWindows();
                String expectedNpmPath = isWindows ? versionedPath + "\\node_modules\\npm\\bin\\npm-cli.js" : versionedPath + "/lib/node_modules/npm/bin/npm-cli.js";
                File expectedNpmFile = new File(installDir, expectedNpmPath);
                throw new IllegalStateException("npm-cli.js not found at expected location: " + expectedNpmFile.getAbsolutePath());
            }
            return installation;
        }
        catch (FrontendUtils.UnknownVersionException e) {
            throw new IllegalStateException("Failed to verify node version", e);
        }
    }

    private ActiveNodeInstallation buildActiveInstallation(String nodePath, String nodeVersion, String npmCliScript) {
        String npmVersion;
        try {
            npmVersion = FrontendUtils.getVersion("npm", List.of(nodePath, npmCliScript, "--version")).getFullVersion();
        }
        catch (FrontendUtils.UnknownVersionException e) {
            this.getLogger().debug("Could not determine npm version", (Throwable)e);
            npmVersion = "unknown";
        }
        return new ActiveNodeInstallation(nodePath, nodeVersion, npmCliScript, npmVersion);
    }

    private String findCompatibleInstalledVersion(File installDir) {
        if (!installDir.exists() || !installDir.isDirectory()) {
            return null;
        }
        File[] nodeDirs = installDir.listFiles(file -> file.isDirectory() && file.getName().startsWith("node-v"));
        if (nodeDirs == null || nodeDirs.length == 0) {
            return null;
        }
        FrontendVersion bestVersion = null;
        String bestVersionString = null;
        for (File nodeDir : nodeDirs) {
            String dirName = nodeDir.getName();
            String versionString = dirName.substring("node-".length());
            try {
                FrontendVersion version = new FrontendVersion(versionString);
                if (version.isOlderThan(FrontendTools.MINIMUM_AUTO_INSTALLED_NODE)) {
                    this.getLogger().debug("Skipping {} - older than minimum auto-installed {}", (Object)versionString, (Object)FrontendTools.MINIMUM_AUTO_INSTALLED_NODE.getFullVersion());
                    continue;
                }
                if (version.getMajorVersion() > 24) {
                    this.getLogger().debug("Skipping {} - major version {} is newer than maximum supported {}", new Object[]{versionString, version.getMajorVersion(), 24});
                    continue;
                }
                File nodeExecutable = this.getNodeExecutableForVersion(installDir, versionString);
                if (!nodeExecutable.exists()) {
                    this.getLogger().debug("Skipping {} - executable not found at {}", (Object)versionString, (Object)nodeExecutable);
                    continue;
                }
                if (bestVersion != null && !version.isNewerThan(bestVersion)) continue;
                bestVersion = version;
                bestVersionString = versionString;
            }
            catch (NumberFormatException e) {
                this.getLogger().debug("Could not parse version from directory: {}", (Object)dirName);
            }
        }
        return bestVersionString;
    }

    private File getNodeExecutableForVersion(File installDir, String version) {
        String versionedPath = "node-v" + (version.startsWith("v") ? version.substring(1) : version);
        boolean isWindows = FrontendUtils.isWindows();
        String nodeExecutable = isWindows ? versionedPath + "\\node.exe" : versionedPath + "/bin/node";
        return new File(installDir, nodeExecutable);
    }

    private ActiveNodeInstallation tryResolveFromExecutable(File nodeExecutable, File installDir, String expectedVersion) throws FrontendUtils.UnknownVersionException {
        String normalizedExpected;
        String normalizedActual;
        FrontendVersion actualVersion = FrontendUtils.getVersion("node", List.of(nodeExecutable.getAbsolutePath(), "--version"));
        if (expectedVersion != null && !(normalizedActual = NodeResolver.normalizeVersion(actualVersion.getFullVersion())).equals(normalizedExpected = NodeResolver.normalizeVersion(expectedVersion))) {
            throw new IllegalStateException(String.format("Expected node version %s but found %s", normalizedExpected, normalizedActual));
        }
        String npmCliScript = this.findNpmCliScript(installDir, actualVersion.getFullVersion());
        if (npmCliScript == null) {
            npmCliScript = this.findNpmCliScript(installDir, null);
        }
        if (npmCliScript == null) {
            return null;
        }
        return this.buildActiveInstallation(nodeExecutable.getAbsolutePath(), actualVersion.getFullVersion(), npmCliScript);
    }

    private String findNpmCliScript(File nodeFolder, String nodeVersion) {
        boolean isWindows = FrontendUtils.isWindows();
        ArrayList<Object> searchPaths = new ArrayList<Object>();
        if (nodeVersion != null) {
            String normalizedVersion = NodeResolver.normalizeVersion(nodeVersion);
            String string = "node-v" + normalizedVersion;
            searchPaths.add(isWindows ? string + "\\node_modules\\npm\\bin\\npm-cli.js" : string + "/lib/node_modules/npm/bin/npm-cli.js");
        }
        if (isWindows) {
            searchPaths.add("node_modules\\npm\\bin\\npm-cli.js");
            searchPaths.add("npm\\bin\\npm-cli.js");
        } else {
            searchPaths.add("lib/node_modules/npm/bin/npm-cli.js");
            searchPaths.add("../lib/node_modules/npm/bin/npm-cli.js");
        }
        for (String string : searchPaths) {
            File npmCliScript = new File(nodeFolder, string);
            if (!npmCliScript.exists()) continue;
            return npmCliScript.getAbsolutePath();
        }
        return null;
    }

    private static String normalizeVersion(String version) {
        return version != null && version.startsWith("v") ? version.substring(1) : version;
    }

    private Logger getLogger() {
        return LoggerFactory.getLogger(NodeResolver.class);
    }

    record ActiveNodeInstallation(String nodeExecutable, String nodeVersion, String npmCliScript, String npmVersion) implements Serializable
    {
        ActiveNodeInstallation {
            Objects.requireNonNull(nodeExecutable);
            Objects.requireNonNull(nodeVersion);
            Objects.requireNonNull(npmCliScript);
            Objects.requireNonNull(npmVersion);
        }
    }
}

