/**
 * Copyright (C) 2024 Vaadin Ltd
 *
 * This program is available under Vaadin Commercial License and Service Terms.
 *
 * See {@literal <https://vaadin.com/commercial-license-and-service-terms>}  for the full
 * license.
 */
package com.vaadin.flow.server.startup;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.Set;
import java.util.stream.Stream;

import javax.servlet.ServletContainerInitializer;
import javax.servlet.ServletContext;
import javax.servlet.ServletException;

import com.vaadin.flow.di.Lookup;
import com.vaadin.flow.server.VaadinServletContext;

/**
 * Allows to load the implementation class by one classloader but accepts
 * classes in {@link #onStartup(Set, ServletContext)} method loaded by another
 * classloader.
 * <p>
 * Workaround for https://github.com/vaadin/flow/issues/7805.
 *
 * @author Vaadin Ltd
 *
 */
public interface ClassLoaderAwareServletContainerInitializer
        extends ServletContainerInitializer {

    /**
     * Overridden to use different classloaders if needed.
     * <p>
     * {@inheritDoc}
     */
    @Override
    default void onStartup(Set<Class<?>> set, ServletContext context)
            throws ServletException {
        // see DeferredServletContextIntializers
        DeferredServletContextInitializers.Initializer deferredInitializer = ctx -> {
            ClassLoader webClassLoader = ctx.getClassLoader();
            ClassLoader classLoader = getClass().getClassLoader();

            /*
             * Hack is needed to make a workaround for weird behavior of WildFly
             * with skinnywar See https://github.com/vaadin/flow/issues/7805
             */
            boolean noHack = false;
            while (classLoader != null) {
                if (classLoader.equals(webClassLoader)) {
                    noHack = true;
                    break;
                } else {
                    /*
                     * The classloader which has loaded this class ({@code
                     * classLoader}) should be either the {@code webClassLoader}
                     * or its child: in this case it knows how to handle the
                     * classes loaded by the {@code webClassLoader} : it either
                     * is able to load them itself or delegate to its parent
                     * (which is the {@code webClassLoader}): in this case hack
                     * is not needed and the {@link #process(Set,
                     * ServletContext)} method can be called directly.
                     */
                    classLoader = classLoader.getParent();
                }
            }

            if (noHack) {
                process(set, ctx);
                return;
            }

            try {
                Class<?> initializer = ctx.getClassLoader()
                        .loadClass(getClass().getName());

                String processMethodName = Stream
                        .of(ClassLoaderAwareServletContainerInitializer.class
                                .getDeclaredMethods())
                        .filter(method -> !method.isDefault()
                                && !method.isSynthetic())
                        .findFirst().get().getName();
                Method operation = Stream.of(initializer.getDeclaredMethods())
                        .filter(method -> method.getName()
                                .equals(processMethodName))
                        .findFirst().get();
                operation.invoke(initializer.newInstance(),
                        new Object[] { set, ctx });
            } catch (ClassNotFoundException | IllegalAccessException
                    | IllegalArgumentException | InvocationTargetException
                    | InstantiationException e) {
                throw new ServletException(e);
            }
        };

        if (requiresLookup()) {
            VaadinServletContext vaadinContext = new VaadinServletContext(
                    context);
            synchronized (context) {
                if (vaadinContext.getAttribute(Lookup.class) == null) {
                    DeferredServletContextInitializers initializers = vaadinContext
                            .getAttribute(
                                    DeferredServletContextInitializers.class,
                                    () -> new DeferredServletContextInitializers());
                    initializers.addInitializer(
                            ctx -> deferredInitializer.init(ctx));
                    return;
                }
            }
        }
        deferredInitializer.init(context);
    }

    /**
     * Whether this initializer requires lookup or not.
     *
     * @return whether this initializer requires lookup
     */
    default boolean requiresLookup() {
        return true;
    }

    /**
     * Implement this method instead of {@link #onStartup(Set, ServletContext)}
     * to handle classes accessible by different classloaders.
     *
     * @param set
     *            the Set of application classes that extend, implement, or have
     *            been annotated with the class types specified by the
     *            {@link javax.servlet.annotation.HandlesTypes HandlesTypes}
     *            annotation, or <tt>null</tt> if there are no matches, or this
     *            <tt>ServletContainerInitializer</tt> has not been annotated
     *            with <tt>HandlesTypes</tt>
     *
     * @param ctx
     *            the <tt>ServletContext</tt> of the web application that is
     *            being started and in which the classes contained in <tt>c</tt>
     *            were found
     *
     * @throws ServletException
     *             if an error has occurred
     *
     * @see #onStartup(Set, ServletContext)
     */
    void process(Set<Class<?>> set, ServletContext ctx) throws ServletException;
}
