/**
 *    Copyright 2000-2026 Vaadin Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.vaadin.flow.gradle

import java.io.File
import com.vaadin.flow.plugin.base.BuildFrontendUtil
import com.vaadin.flow.server.Constants
import com.vaadin.flow.internal.FrontendUtils
import org.gradle.api.provider.Property
import org.gradle.api.tasks.OutputDirectory
import org.gradle.api.tasks.OutputFile

/**
 * Declaratively defines the outputs of the [VaadinPrepareFrontendTask]:
 * package.json, vite.config.ts, and other files generated by Vaadin, as well
 * as the files in the frontend directory. Being used for caching the results
 * of vaadinPrepareFrontend task to not run it again if inputs are the same.
 */
internal class PrepareFrontendOutputProperties(
    adapter: GradlePluginAdapter
) {

    private val config = adapter.config
    private val projectDir = config.projectDir
    private val generatedTsFolder =
        BuildFrontendUtil.getGeneratedFrontendDirectory(adapter)
    private val resourceOutputDirectory = config.resourceOutputDirectory

    @OutputFile
    fun getPackageJson(): File = File(projectDir, Constants.PACKAGE_JSON)

    @OutputFile
    fun getPackageLockJson(): File =
        File(projectDir, Constants.PACKAGE_LOCK_JSON)

    @OutputFile
    fun getPackageLockYaml(): File =
        File(projectDir, Constants.PACKAGE_LOCK_YAML)

    @OutputFile
    fun getViteConfig(): File =
        File(projectDir, FrontendUtils.VITE_CONFIG)

    @OutputFile
    fun getViteGeneratedConfig(): File =
        File(projectDir, FrontendUtils.VITE_GENERATED_CONFIG)

    @OutputFile
    fun getTsConfig(): File = File(projectDir, "tsconfig.json")

    @OutputFile
    fun getTsDefinition(): File = File(projectDir, "types.d.ts")

    @OutputDirectory
    fun getGeneratedTsFolder(): File = generatedTsFolder

    @OutputDirectory
    fun getResourceOutputDirectory(): Property<File> =
        resourceOutputDirectory
}
