/*
 * Copyright (C) 2000-2024 Vaadin Ltd
 *
 * This program is available under Vaadin Commercial License and Service Terms.
 *
 * See <https://vaadin.com/commercial-license-and-service-terms> for the full
 * license.
 */
package com.vaadin.flow.router.internal;

import java.io.Serializable;
import java.util.Optional;
import java.util.Set;

import com.vaadin.flow.router.RouteParameterFormatOption;
import com.vaadin.flow.router.RouteParameterRegex;

/**
 * Utility class which contains various methods for defining url parameter
 * template.
 * <p>
 * For internal use only. May be renamed or removed in a future release.
 */
class RouteFormat implements Serializable {

    /**
     * Returns whether the specified template contains route parameters.
     *
     * @param template
     *            a template.
     * @return true if the specified template contains route parameters,
     *         otherwise false.
     */
    static boolean hasParameters(String template) {
        return template.contains(":");
    }

    /**
     * Returns whether the specified template contains route parameters.
     *
     * @param template
     *            a template.
     * @return true if the specified template contains route parameters,
     *         otherwise false.
     */
    static boolean hasRequiredParameter(String template) {
        int index = -1;
        do {
            index = template.indexOf(':', index + 1);

            if (index >= 0) {
                final int regexIndex = template.indexOf('(', index);
                final int slashIndex = template.indexOf('/', index);

                int parameterNameEnding = Math.min(regexIndex, slashIndex);

                // ending -1 then check if it's regex or slash or both that is
                // missing
                if (parameterNameEnding < 0) {
                    parameterNameEnding = regexIndex == -1 ? slashIndex
                            : regexIndex;
                }
                // End of the string.
                if (parameterNameEnding < 0) {
                    parameterNameEnding = template.length();
                }

                int optional = template.indexOf('?', index);
                if (0 < optional && optional < parameterNameEnding) {
                    // This parameter is an optional, move on.
                    continue;
                }

                int wildcard = template.indexOf('*', index);
                if (0 < wildcard && wildcard < parameterNameEnding) {
                    // This parameter is a wildcard and should be the last.
                    return false;
                }

                // This parameter is required.
                return true;

            } else {
                // We reached the end of the search.
                return false;
            }

        } while (true);
    }

    static boolean isParameter(String segmentTemplate) {
        return segmentTemplate.startsWith(":");
    }

    static boolean isOptionalParameter(String segmentTemplate) {
        return isParameter(segmentTemplate) && (segmentTemplate.endsWith("?")
                || segmentTemplate.contains("?("));
    }

    static boolean isVarargsParameter(String segmentTemplate) {
        return isParameter(segmentTemplate) && (segmentTemplate.endsWith("*")
                || segmentTemplate.contains("*("));
    }

    static String getModifier(String segmentTemplate) {
        if (isOptionalParameter(segmentTemplate)) {
            return "?";
        } else if (isVarargsParameter(segmentTemplate)) {
            return "*";
        }

        return "";
    }

    static String formatSegment(RouteSegment segment,
            Set<RouteParameterFormatOption> format) {

        if (!segment.isParameter()) {
            return segment.getName();
        }

        StringBuilder result = new StringBuilder();

        result.append(":");

        final boolean formatRegex = format
                .contains(RouteParameterFormatOption.REGEX)
                || format.contains(RouteParameterFormatOption.REGEX_NAME);
        boolean wrapRegex = false;

        if (format.contains(RouteParameterFormatOption.NAME)) {
            result.append(segment.getName());
            wrapRegex = true;
        }

        if (format.contains(RouteParameterFormatOption.MODIFIER)) {
            result.append(getModifier(segment.getTemplate()));
            wrapRegex = true;
        }

        final Optional<String> regex = formatRegex
                ? formatSegmentRegex(segment, format)
                : Optional.empty();
        if (regex.isPresent()) {
            if (wrapRegex) {
                result.append("(");
            }

            result.append(regex.get());

            if (wrapRegex) {
                result.append(")");
            }
        }

        return result.toString();
    }

    static Optional<String> formatSegmentRegex(RouteSegment segment,
            Set<RouteParameterFormatOption> format) {
        final Optional<String> regex = segment.getRegex();
        if (format.contains(RouteParameterFormatOption.REGEX_NAME)) {
            return Optional.of(RouteParameterRegex.getName(regex.orElse("")));
        } else {
            return regex;
        }
    }

    /**
     * Define a route url parameter details.
     */
    static class ParameterInfo implements Serializable {

        private final String name;

        private final String template;

        private final boolean optional;

        private final boolean varargs;

        private final String regex;

        ParameterInfo(String template) {
            this.template = template;

            if (!isParameter(template)) {
                throw new IllegalArgumentException(
                        "The given string is not a parameter template.");
            }

            optional = isOptionalParameter(template);
            if (optional) {
                template = template.replaceFirst("\\?", "");
            }
            varargs = isVarargsParameter(template);
            if (varargs) {
                template = template.replaceFirst("\\*", "");
            }

            // Remove :
            template = template.substring(1);

            // Extract the template defining the value of the parameter.
            final int regexStartIndex = template.indexOf('(');
            if (regexStartIndex != -1) {

                name = template.substring(0, regexStartIndex);

                regex = template.substring(regexStartIndex + 1,
                        template.length() - 1);
            } else {
                name = template;
                regex = null;
            }
        }

        public String getName() {
            return name;
        }

        public String getTemplate() {
            return template;
        }

        public boolean isOptional() {
            return optional;
        }

        public boolean isVarargs() {
            return varargs;
        }

        public Optional<String> getRegex() {
            return Optional.ofNullable(regex);
        }
    }

}
