/**
 * Copyright (C) 2024 Vaadin Ltd
 *
 * This program is available under Vaadin Commercial License and Service Terms.
 *
 * See {@literal <https://vaadin.com/commercial-license-and-service-terms>}  for the full
 * license.
 */
package com.vaadin.flow.internal.nodefeature;

import java.util.Optional;

import com.vaadin.flow.component.Component;
import com.vaadin.flow.component.ComponentUtil;
import com.vaadin.flow.component.Composite;
import com.vaadin.flow.component.polymertemplate.PolymerTemplate;
import com.vaadin.flow.internal.StateNode;

/**
 * A server side only node feature for mapping a node to a component.
 * <p>
 * For internal use only. May be renamed or removed in a future release.
 *
 * @author Vaadin Ltd
 * @since 1.0
 */
public class ComponentMapping extends ServerSideFeature {

    private Component component = null;

    /**
     * Creates an instance of this node feature.
     *
     * @param node
     *            the node that the feature belongs to
     */
    protected ComponentMapping(StateNode node) {
        super(node);
    }

    /**
     * Assigns the given component to this node.
     * <p>
     * When assigning a component to the node, there must be no previously
     * assigned component.
     *
     * @param component
     *            the component to assign to this node, not {@code null}
     */
    public void setComponent(Component component) {
        assert component != null : "Component must not be null";
        assert this.component == null || component instanceof Composite
                : "Only a Composite is allowed to remap a component";
        this.component = component;

        if (getNode().hasFeature(ClientCallableHandlers.class)) {
            getNode().getFeature(ClientCallableHandlers.class)
                    .componentSet(component);
        }
        if (component instanceof PolymerTemplate<?>
                && getNode().hasFeature(PolymerServerEventHandlers.class)) {
            getNode().getFeature(PolymerServerEventHandlers.class)
                    .componentSet((PolymerTemplate<?>) component);
        }
    }

    /**
     * Gets the component this node has been mapped to, if any.
     *
     * @return an optional component, or an empty optional if no component has
     *         been mapped to this node
     */
    public Optional<Component> getComponent() {
        return Optional.ofNullable(component);
    }

    /**
     * Gets the component mapped to the given state node.
     *
     * @param node
     *            the state node for which to find a component, not
     *            <code>null</code>
     * @return the mapped component, or an empty optional if no component is
     *         mapped
     */
    public static Optional<Component> getComponent(StateNode node) {
        assert node != null;
        assert node.hasFeature(ComponentMapping.class);

        return node.getFeatureIfInitialized(ComponentMapping.class)
                .flatMap(ComponentMapping::getComponent);
    }

    @Override
    public void onAttach(boolean initialAttach) {
        getComponent().ifPresent(
                c -> ComponentUtil.onComponentAttach(c, initialAttach));
    }

    @Override
    public void onDetach() {
        getComponent().ifPresent(ComponentUtil::onComponentDetach);
    }

}
