package com.vaadin.copilot.theme;

/**
 * Utility class for checking the current application theme in use.
 * <p>
 * This class provides helper methods to determine whether the active
 * application theme is {@link ApplicationTheme#LUMO} or
 * {@link ApplicationTheme#AURA}.
 * </p>
 *
 * <p>
 * The methods rely on the {@link ApplicationDetectedThemeHolder} singleton to
 * retrieve the currently detected {@link ApplicationTheme}.
 * </p>
 *
 * <p>
 * <strong>Usage Example:</strong>
 * </p>
 * 
 * <pre>{@code
 * if (AppThemeUtils.isLumo()) {
 *     // Apply Lumo-specific styling or logic
 * }
 * }</pre>
 *
 * <p>
 * This class cannot be instantiated.
 * </p>
 */
public class AppThemeUtils {
    private AppThemeUtils() {

    }

    /**
     * Checks if the currently active application theme is
     * {@link ApplicationTheme#LUMO}.
     *
     * @return {@code true} if the detected theme is Lumo, {@code false} otherwise
     */
    public static boolean isLumo() {
        return ApplicationTheme.LUMO == ApplicationDetectedThemeHolder.getInstance().getTheme();
    }

    /**
     * Checks if the currently active application theme is
     * {@link ApplicationTheme#AURA}.
     *
     * @return {@code true} if the detected theme is Aura, {@code false} otherwise
     */
    public static boolean isAura() {
        return ApplicationTheme.AURA == ApplicationDetectedThemeHolder.getInstance().getTheme();
    }

    /**
     * Checks if there is no currently active application theme.
     *
     * @return {@code true} if none, {@code false} otherwise
     */
    public static boolean isNone() {
        return ApplicationTheme.NONE == ApplicationDetectedThemeHolder.getInstance().getTheme();
    }
}
