package com.vaadin.copilot;

import java.lang.reflect.Method;
import java.util.List;

/**
 * Utility class for connecting to a service method.
 */
public class ConnectToService {

    public static final String PAGEABLE_TYPE = "org.springframework.data.domain.Pageable";
    public static final String HILLA_FILTER_TYPE = "com.vaadin.hilla.crud.filter.Filter";
    public static final String STRING_TYPE = String.class.getName();

    /**
     * The type of binding that can be used for a Flow Grid.
     */
    public enum BindingType {
        EAGER, LAZY, NONE
    }

    /**
     * Checks if a Flow Grid can be bound to the given service method.
     *
     * @param serviceMethod
     *            the service method to check
     * @return the binding type that can be used
     */
    public static BindingType canBindFlowGridToService(SpringBridge.ServiceMethodInfo serviceMethod) {

        Method method = serviceMethod.serviceMethod();
        if (method.getReturnType() != List.class) {
            return BindingType.NONE;
        }

        Class<?>[] parameterTypes = method.getParameterTypes();
        if (parameterTypes.length == 0) {
            // Eager binding is OK
            return BindingType.EAGER;
        }

        if (parameterTypes.length == 1 && parameterTypes[0].getName().equals(PAGEABLE_TYPE)) {
            // Any method with Pageable as parameter is OK
            return BindingType.LAZY;
        }

        return BindingType.NONE;
    }

    /**
     * Checks if a Hilla Grid can be bound to the given service method.
     *
     * @param serviceMethod
     *            the service method to check
     * @return the binding type that can be used
     */
    public static BindingType canBindHillaGridToService(SpringBridge.ServiceMethodInfo serviceMethod) {
        BindingType result = canBindFlowGridToService(serviceMethod);
        if (result == BindingType.NONE) {
            Method method = serviceMethod.serviceMethod();
            Class<?>[] parameterTypes = method.getParameterTypes();
            if (parameterTypes.length == 2 && parameterTypes[0].getName().equals(PAGEABLE_TYPE)
                    && (parameterTypes[1].getName().equals(HILLA_FILTER_TYPE))) {
                // Using CrudRepositoryService with Pageable and Filter is OK
                return BindingType.LAZY;
            }
        }
        return result;
    }

    /**
     * Checks if a Flow ComboBox can be bound to the given service method.
     *
     * @param serviceMethod
     *            the service method to check
     * @return the binding type that can be used
     */
    public static BindingType canBindFlowComboBoxToService(SpringBridge.ServiceMethodInfo serviceMethod) {
        Method method = serviceMethod.serviceMethod();
        if (method.getReturnType() != List.class) {
            return BindingType.NONE;
        }

        Class<?>[] parameterTypes = method.getParameterTypes();
        if (parameterTypes.length == 0) {
            // Eager binding with in-memory filtering
            return BindingType.EAGER;
        }

        if (parameterTypes.length == 2) {
            boolean pageableAndString = parameterTypes[0].getName().equals(PAGEABLE_TYPE)
                    && parameterTypes[1].getName().equals(STRING_TYPE);
            boolean stringAndPageable = parameterTypes[0].getName().equals(STRING_TYPE)
                    && parameterTypes[1].getName().equals(PAGEABLE_TYPE);
            boolean hillaFilter = parameterTypes[0].getName().equals(PAGEABLE_TYPE)
                    && parameterTypes[1].getName().equals(HILLA_FILTER_TYPE);

            if (pageableAndString || stringAndPageable || hillaFilter) {
                return BindingType.LAZY;
            }
        }

        return BindingType.NONE;
    }

    /**
     * Checks if a Hilla ComboBox can be bound to the given service method.
     *
     * @param serviceMethod
     *            the service method to check
     * @return the binding type that can be used
     */
    public static BindingType canBindHillaComboBoxToService(SpringBridge.ServiceMethodInfo serviceMethod) {
        return canBindFlowComboBoxToService(serviceMethod);
    }

}
