package com.vaadin.copilot;

import static com.vaadin.copilot.Util.getFilenameFromClassName;
import static com.vaadin.copilot.Util.getGetterName;
import static com.vaadin.copilot.Util.getPackageName;
import static com.vaadin.copilot.Util.getSetterName;
import static com.vaadin.copilot.Util.getSimpleName;

import java.io.File;
import java.io.IOException;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.time.temporal.TemporalAccessor;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.UUID;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.stream.Collectors;

import com.vaadin.copilot.plugins.info.JdkInfo;
import com.vaadin.flow.internal.JacksonUtils;
import com.vaadin.flow.server.auth.AnonymousAllowed;
import com.vaadin.flow.shared.util.SharedUtil;

import com.github.javaparser.StaticJavaParser;
import com.github.javaparser.ast.CompilationUnit;
import com.github.javaparser.ast.Modifier;
import com.github.javaparser.ast.NodeList;
import com.github.javaparser.ast.body.ClassOrInterfaceDeclaration;
import com.github.javaparser.ast.body.ConstructorDeclaration;
import com.github.javaparser.ast.body.FieldDeclaration;
import com.github.javaparser.ast.body.MethodDeclaration;
import com.github.javaparser.ast.body.VariableDeclarator;
import com.github.javaparser.ast.expr.AnnotationExpr;
import com.github.javaparser.ast.expr.FieldAccessExpr;
import com.github.javaparser.ast.expr.MethodCallExpr;
import com.github.javaparser.ast.expr.Name;
import com.github.javaparser.ast.expr.NameExpr;
import com.github.javaparser.ast.expr.ObjectCreationExpr;
import com.github.javaparser.ast.expr.SingleMemberAnnotationExpr;
import com.github.javaparser.ast.expr.VariableDeclarationExpr;
import com.github.javaparser.ast.stmt.BlockStmt;
import com.github.javaparser.ast.stmt.ExpressionStmt;
import com.github.javaparser.ast.type.ClassOrInterfaceType;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import tools.jackson.databind.JsonNode;
import tools.jackson.databind.node.ObjectNode;

/**
 * UIServiceCreator is responsible for generating Java service and bean classes
 * dynamically. It supports different data storage types and provides
 * functionality for creating Java classes with fields, getters, and setters.
 *
 * <p>
 * This class utilizes JavaParser for AST modifications and dynamically writes
 * Java files to the project structure.
 */
public class UIServiceCreator {

    private static DateTimeFormatter isoDateTimeFormatter = DateTimeFormatter.ISO_DATE_TIME;
    private static DateTimeFormatter isoDateFormatter = DateTimeFormatter.ISO_DATE;

    public enum DataStorage {
        JPA, IN_MEMORY
    }

    /**
     * Represents information about a field in a Java bean.
     *
     * @param name
     *            the name of the property
     * @param javaType
     *            the Java type of the property
     */
    public record FieldInfo(String name, String javaType) {
        public static FieldInfo fromJson(JsonNode obj) {
            return new UIServiceCreator.FieldInfo(obj.get("name").asString(), obj.get("javaType").asString());
        }

        public JsonNode toJson() {
            ObjectNode obj = JacksonUtils.createObjectNode();
            obj.put("name", name());
            obj.put("javaType", javaType());
            return obj;
        }
    }

    /**
     * Represents information about a Java bean.
     *
     * @param fullyQualifiedBeanName
     *            the fully qualified name of the bean class
     * @param fields
     *            the fields of the bean
     */
    public record BeanInfo(String fullyQualifiedBeanName, FieldInfo[] fields) {
    }

    /**
     * Represents information about a service and its associated bean.
     *
     * @param beanInfo
     *            the information about the bean
     * @param dataStorage
     *            the type of data storage
     * @param browserCallable
     *            whether the service can be called from the browser
     * @param exampleData
     *            example data to use or null to not generate example data
     */
    public record ServiceAndBeanInfo(BeanInfo beanInfo, DataStorage dataStorage, boolean browserCallable,
            List<Map<String, Object>> exampleData, String servicePackageName, String repositoryPackageName) {

        public String getBeanName() {
            return beanInfo().fullyQualifiedBeanName();
        }

        public String getRepositoryName() {
            String base;
            if (repositoryPackageName() != null) {
                base = repositoryPackageName() + "." + getSimpleName(getBeanName());
            } else {
                base = getBeanName();
            }
            return base + "Repository";
        }

        public String getServiceName() {
            String base;
            if (servicePackageName() != null) {
                base = servicePackageName() + "." + getSimpleName(getBeanName());
            } else {
                base = getBeanName();
            }
            return base + "Service";
        }
    }

    /**
     * Creates a service and bean based on the provided ServiceAndBeanInfo.
     *
     * @param serviceAndBeanInfo
     *            the information about the service and bean to create
     * @param moduleInfo
     *            the module to create the file in
     * @return true if a restart is needed, false otherwise
     * @throws IOException
     *             if an I/O error occurs
     */
    public boolean createServiceAndBean(ServiceAndBeanInfo serviceAndBeanInfo,
            JavaSourcePathDetector.ModuleInfo moduleInfo) throws IOException {
        JdkInfo jdkInfo = JdkInfo.get();
        boolean needsRestart = jdkInfo.runningWith() == JdkInfo.HotswapSolution.NONE;

        createBean(serviceAndBeanInfo.beanInfo(), serviceAndBeanInfo.dataStorage, moduleInfo);
        createRepository(serviceAndBeanInfo, moduleInfo);
        createService(serviceAndBeanInfo, moduleInfo);
        return needsRestart;

    }

    /**
     * Creates a Java bean class based on the provided BeanInfo.
     *
     * @param beanInfo
     *            the information about the bean to create
     * @param moduleInfo
     *            the module to create the file in
     * @throws IOException
     *             if an I/O error occurs
     */
    void createBean(BeanInfo beanInfo, DataStorage dataStorage, JavaSourcePathDetector.ModuleInfo moduleInfo)
            throws IOException {
        String beanName = beanInfo.fullyQualifiedBeanName();
        String filename = getFilenameFromClassName(beanName);
        File javaFile = new File(moduleInfo.javaSourcePaths().get(0).toFile(), filename);
        if (javaFile.exists()) {
            throw new IllegalArgumentException("File " + javaFile + " already exists");
        }
        ClassOrInterfaceDeclaration clazz = createClass(beanName);
        CompilationUnit compilationUnit = clazz.findCompilationUnit().orElseThrow();

        if (dataStorage == DataStorage.JPA) {
            // Add JPA entity annotations and imports
            compilationUnit.addImport("jakarta.persistence.Entity");
            clazz.addMarkerAnnotation("Entity");
        }

        // Add fields
        for (FieldInfo field : beanInfo.fields()) {
            FieldDeclaration fieldDeclaration = clazz.addField(getSimpleName(field.javaType()), field.name(),
                    Modifier.Keyword.PRIVATE);
            if (dataStorage == DataStorage.JPA) {
                if (field.name().equals("id")) {
                    compilationUnit.addImport("jakarta.persistence.Id");
                    fieldDeclaration.addMarkerAnnotation("Id");
                }
                if (field.javaType().equals(UUID.class.getName())) {
                    compilationUnit.addImport("org.hibernate.annotations.JdbcTypeCode");
                    compilationUnit.addImport("java.sql.Types");
                    AnnotationExpr jdbcTypeAnnotation = new SingleMemberAnnotationExpr(new Name("JdbcTypeCode"),
                            new FieldAccessExpr(new NameExpr("Types"), "VARCHAR"));
                    fieldDeclaration.addAnnotation(jdbcTypeAnnotation);
                }
            }
        }

        // Generate getters and setters
        for (FieldInfo field : beanInfo.fields()) {
            String simpleJavaType = getSimpleName(field.javaType());
            if (field.javaType().contains(".")) {
                compilationUnit.addImport(field.javaType());
            }

            // Getter method
            MethodDeclaration getter = clazz.addMethod(getGetterName(field.name, field.javaType),
                    com.github.javaparser.ast.Modifier.Keyword.PUBLIC);
            getter.setType(simpleJavaType);
            getter.setBody(new BlockStmt().addStatement("return " + field.name() + ";"));

            // Setter method
            MethodDeclaration setter = clazz.addMethod(getSetterName(field.name), Modifier.Keyword.PUBLIC);
            setter.addParameter(simpleJavaType, field.name());
            setter.setBody(new BlockStmt().addStatement("this." + field.name() + " = " + field.name() + ";"));
        }
        ProjectFileManager.get().writeFile(javaFile, "New bean", compilationUnit.toString());
    }

    /**
     * Creates a service class based on the provided ServiceAndBeanInfo.
     *
     * @param serviceAndBeanInfo
     *            the information about the service and bean to create
     * @param moduleInfo
     *            the module to create the file in
     * @throws IOException
     *             if an I/O error occurs
     */
    private void createService(ServiceAndBeanInfo serviceAndBeanInfo, JavaSourcePathDetector.ModuleInfo moduleInfo)
            throws IOException {
        String beanName = serviceAndBeanInfo.getBeanName();
        String serviceName = serviceAndBeanInfo.getServiceName();
        String repositoryName = serviceAndBeanInfo.getRepositoryName();

        String filename = getFilenameFromClassName(serviceName);
        File javaFile = new File(moduleInfo.javaSourcePaths().get(0).toFile(), filename);
        if (javaFile.exists()) {
            throw new IllegalArgumentException("File " + javaFile + " already exists");
        }

        ClassOrInterfaceDeclaration clazz = createClass(serviceName);
        CompilationUnit compilationUnit = clazz.findCompilationUnit().orElseThrow();

        String simpleBeanName = getSimpleName(beanName);
        String simpleRepositoryName = getSimpleName(repositoryName);

        compilationUnit.addImport("org.springframework.stereotype.Service");
        compilationUnit.addImport(beanName);
        compilationUnit.addImport("org.springframework.data.domain.Pageable");

        clazz.addMarkerAnnotation("Service");
        if (serviceAndBeanInfo.browserCallable()) {
            compilationUnit.addImport("com.vaadin.hilla.BrowserCallable");
            clazz.addMarkerAnnotation("BrowserCallable");
            compilationUnit.addImport(AnonymousAllowed.class);
            clazz.addMarkerAnnotation(AnonymousAllowed.class.getSimpleName());
        }
        if (serviceAndBeanInfo.dataStorage() == DataStorage.JPA) {
            compilationUnit.addImport(repositoryName);
            compilationUnit.addImport(List.class);

            // Add repository field
            clazz.addField(simpleRepositoryName, "repository", Modifier.Keyword.PRIVATE, Modifier.Keyword.FINAL);
            // Add constructor
            ConstructorDeclaration constructor = clazz.addConstructor(Modifier.Keyword.PUBLIC);
            constructor.addParameter(simpleRepositoryName, "repository");
            constructor.setBody(new BlockStmt().addStatement("this.repository = repository;"));

            // Add list method
            MethodDeclaration listMethod = clazz.addMethod("list", Modifier.Keyword.PUBLIC);
            listMethod.addParameter("Pageable", "pageable");
            listMethod.setType("List<" + simpleBeanName + ">");
            listMethod.setBody(new BlockStmt().addStatement("return repository.findAll(pageable).getContent();"));

            if (serviceAndBeanInfo.exampleData() != null) {
                List<Map<String, Object>> exampleData = serviceAndBeanInfo.exampleData();

                String propertyNames = Arrays.stream(serviceAndBeanInfo.beanInfo().fields()).map(FieldInfo::name)
                        .map(UIServiceCreator::getSqlIdentifier).collect(Collectors.joining(", "));

                List<String> statements = new ArrayList<>();
                for (Map<String, Object> data : exampleData) {
                    String propertyValues = Arrays.stream(serviceAndBeanInfo.beanInfo().fields())
                            .map(fieldInfo -> getDataSqlValueForField(fieldInfo, data.get(fieldInfo.name())))
                            .collect(Collectors.joining(", "));

                    String statement = "INSERT INTO " + getSqlIdentifier(simpleBeanName) + " (" + propertyNames
                            + ") VALUES (" + propertyValues + ");";
                    statements.add(statement);
                }

                File dataSqlFile = new File(moduleInfo.resourcePaths().get(0).toFile(), "data.sql");
                String dataSql = "";
                if (dataSqlFile.exists()) {
                    dataSql = ProjectFileManager.get().readFile(dataSqlFile);
                }
                if (!dataSql.endsWith("\n") && !dataSql.isEmpty()) {
                    dataSql += "\n";
                }
                dataSql += String.join("\n", statements);
                ProjectFileManager.get().writeFile(dataSqlFile, "data.sql update", dataSql);
            }

        } else if (serviceAndBeanInfo.dataStorage() == DataStorage.IN_MEMORY) {
            // Add in-memory service implementation
            compilationUnit.addImport(CopyOnWriteArrayList.class);
            compilationUnit.addImport(List.class);
            compilationUnit.addImport(Objects.class);
            compilationUnit.addImport(Optional.class);

            // Add static data list
            clazz.addField("List<" + simpleBeanName + ">", "data", Modifier.Keyword.PRIVATE, Modifier.Keyword.STATIC,
                    Modifier.Keyword.FINAL).getVariables().get(0).setInitializer("new CopyOnWriteArrayList<>()");

            if (serviceAndBeanInfo.exampleData() != null) {
                ClassOrInterfaceType itemType = StaticJavaParser.parseClassOrInterfaceType(simpleBeanName);
                ClassOrInterfaceType simpleItemType = itemType.clone().setScope(null);
                // Add method for creating an example instance
                // private static Car2 createExampleData(Long id, String make, String model,
                // Integer,...
                MethodDeclaration createExampleDataMethod = clazz.addMethod("createExampleData",
                        Modifier.Keyword.PRIVATE, Modifier.Keyword.STATIC);
                createExampleDataMethod.setType(simpleItemType);

                BlockStmt createExampleDataMethodBody = createExampleDataMethod.getBody().get();

                // Car2 car = new Car2();
                String itemVariable = simpleBeanName.toLowerCase(Locale.ENGLISH);

                ObjectCreationExpr createItemExpr = new ObjectCreationExpr(null, simpleItemType, NodeList.nodeList());
                VariableDeclarationExpr createItemVariable = new VariableDeclarationExpr(
                        new VariableDeclarator(simpleItemType, itemVariable, createItemExpr));
                createExampleDataMethodBody.addStatement(createItemVariable);

                Arrays.stream(serviceAndBeanInfo.beanInfo().fields()).forEach(fieldInfo -> {
                    String parameterName = fieldInfo.name();

                    String javaType = fieldInfo.javaType();
                    String simpleJavaType = getSimpleName(javaType);
                    if (javaType.contains(".")) {
                        compilationUnit.addImport(javaType);
                    }
                    createExampleDataMethod.addParameter(simpleJavaType, parameterName);
                    String setter = getSetterName(parameterName);
                    createExampleDataMethodBody
                            .addStatement(new ExpressionStmt(new MethodCallExpr(new NameExpr(itemVariable), setter,
                                    NodeList.nodeList(new NameExpr(parameterName)))));
                });
                createExampleDataMethodBody.addStatement("return " + itemVariable + ";");

                // Example data
                BlockStmt staticBlockBody = clazz.addStaticInitializer();
                serviceAndBeanInfo.exampleData().forEach(data -> {
                    String parameters = Arrays.stream(serviceAndBeanInfo.beanInfo().fields()).map(fieldInfo -> {
                        String parameterName = fieldInfo.name();
                        if (!data.containsKey(parameterName)) {
                            getLogger().warn("Missing parameter {} in example data", parameterName);
                            return getJavaSourceValueForField(fieldInfo, null);
                        }
                        return getJavaSourceValueForField(fieldInfo, data.get(parameterName));
                    }).collect(Collectors.joining(", "));
                    staticBlockBody.addStatement("data.add(createExampleData(" + parameters + "));");
                });
            }

            // Add list method
            MethodDeclaration listMethod = clazz.addMethod("list", Modifier.Keyword.PUBLIC);
            listMethod.addParameter("Pageable", "pageable");
            listMethod.setType("List<" + simpleBeanName + ">");
            listMethod.setBody(new BlockStmt().addStatement("int from = (int) pageable.getOffset();")
                    .addStatement(
                            "int to = (int) Math.min(pageable.getOffset() + pageable.getPageSize(), data.size());")
                    .addStatement("return data.subList(from, to);"));

            // Add save method
            MethodDeclaration saveMethod = clazz.addMethod("save", Modifier.Keyword.PUBLIC);
            saveMethod.addParameter(simpleBeanName, "value");
            saveMethod.setType(simpleBeanName);
            String saveMethodBody = """
                        {
                        Optional<%s> existingItem = data.stream().filter(item -> Objects.equals(item.getId(), value.getId())).findFirst();
                        existingItem.ifPresentOrElse(item -> {
                            int index = data.indexOf(item);
                            data.set(index, value);
                        }, () -> {
                            Long maxId = data.stream().map(%s::getId).max(Long::compareTo).orElse(0L);
                            value.setId(maxId + 1L);
                            data.add(value);
                        });
                        return value;
                        }
                    """
                    .formatted(simpleBeanName, simpleBeanName);

            saveMethod.setBody(StaticJavaParser.parseBlock(saveMethodBody));
            // Add delete method
            MethodDeclaration deleteMethod = clazz.addMethod("delete", Modifier.Keyword.PUBLIC);
            deleteMethod.addParameter("Long", "id");
            deleteMethod.setType("void");
            deleteMethod
                    .setBody(new BlockStmt().addStatement("data.removeIf(item -> Objects.equals(item.getId(), id));"));
        }

        ProjectFileManager.get().writeFile(javaFile, "New service", compilationUnit.toString());
    }

    private static String getSqlIdentifier(String camelCase) {
        return SharedUtil.camelCaseToDashSeparated(SharedUtil.firstToLower(camelCase)).replace("-", "_");
    }

    static String getJavaSourceValueForField(FieldInfo fieldInfo, Object value) {
        if (value == null) {
            if (fieldInfo.javaType().equals("boolean")) {
                return "false";
            }
            return "null";
        }

        return switch (fieldInfo.javaType()) {
        case "java.lang.String" -> "\"" + value + "\"";
        case "java.lang.Long" -> stripDecimals(value) + "L";
        case "java.lang.Integer" -> stripDecimals(value);
        case "java.lang.Double" -> String.valueOf(value) + 'd';
        case "java.lang.Boolean" -> String.valueOf(value);
        case "java.lang.Float" -> value + "f";
        case "java.math.BigDecimal" -> "new BigDecimal(\"" + value + "\")";
        case "java.util.UUID" -> "UUID.fromString(\"" + ensureUUID(value) + "\")";
        case "java.time.LocalDate" -> {
            LocalDate localDate = parseLocalDate(value);
            if (localDate != null) {
                yield "LocalDate.of(" + localDate.getYear() + ", " + localDate.getMonthValue() + ", "
                        + localDate.getDayOfMonth() + ")";
            } else {
                yield "null";
            }
        }
        case "java.time.LocalDateTime" -> {
            LocalDateTime localDateTime = parseLocalDateTime(value);
            if (localDateTime != null) {
                yield "LocalDateTime.of(" + localDateTime.getYear() + ", " + localDateTime.getMonthValue() + ", "
                        + localDateTime.getDayOfMonth() + ", " + localDateTime.getHour() + ", "
                        + localDateTime.getMinute() + ", " + localDateTime.getSecond() + ")";
            } else {
                yield "null";
            }
        }
        default -> {
            getLogger().debug("Using default toString for type " + fieldInfo.javaType());
            yield String.valueOf(value);
        }
        };
    }

    private static LocalDate parseLocalDate(Object value) {
        try {
            TemporalAccessor parsed = isoDateFormatter.parse(String.valueOf(value));
            return LocalDate.from(parsed);
        } catch (Exception e) {
            getLogger().warn("Unable to parse LocalDate from '{}'", value, e);
        }
        return null;
    }

    private static LocalDateTime parseLocalDateTime(Object value) {
        try {
            TemporalAccessor parsed = isoDateTimeFormatter.parse(String.valueOf(value));
            return LocalDateTime.from(parsed);
        } catch (Exception e) {
            getLogger().warn("Unable to parse LocalDateTime from '{}'", value, e);
        }
        return null;
    }

    static String getDataSqlValueForField(FieldInfo fieldInfo, Object value) {
        if (value == null) {
            if (fieldInfo.javaType().equals("boolean")) {
                return "false";
            }
            return "null";
        }

        return switch (fieldInfo.javaType()) {
        case "java.lang.String" -> "'" + Util.escapeSingleQuote(String.valueOf(value)) + "'";
        case "java.lang.Long", "java.lang.Integer" -> stripDecimals(value);
        case "java.lang.Double", "java.lang.Boolean", "java.lang.Float", "java.math.BigDecimal" ->
            String.valueOf(value);
        case "java.util.UUID" -> "'" + ensureUUID(value) + "'";
        case "java.time.LocalDate" -> {
            LocalDate localDate = parseLocalDate(value);
            if (localDate != null) {
                yield "'" + Util.escapeSingleQuote(isoDateFormatter.format(localDate)) + "'";
            } else {
                yield "null";
            }
        }
        case "java.time.LocalDateTime" -> {
            LocalDateTime localDateTime = parseLocalDateTime(value);
            if (localDateTime != null) {
                yield "'" + Util.escapeSingleQuote(isoDateTimeFormatter.format(localDateTime).replace('T', ' ')) + "'";
            }
            getLogger().debug("Unknown value for LocalDateTime of type {}: {}", value.getClass(), value);
            yield "null";
        }
        default -> {
            getLogger().debug("Using default toString for type " + fieldInfo.javaType());
            yield String.valueOf(value);
        }
        };
    }

    private static String ensureUUID(Object value) {
        // The AI isn't that smart so sometimes an UUID can contain a 'g'
        return String.valueOf(value).replaceAll("[^0-9a-fA-F-]", "f");
    }

    private static String stripDecimals(Object value) {
        return String.valueOf(value).replaceFirst("\\..*", "");
    }

    private static Logger getLogger() {
        return LoggerFactory.getLogger(UIServiceCreator.class);
    }

    private ClassOrInterfaceDeclaration createClass(String className) {
        CompilationUnit compilationUnit = createCompilationUnit(className);
        return compilationUnit.addClass(getSimpleName(className));
    }

    private ClassOrInterfaceDeclaration createInterface(String className) {
        CompilationUnit compilationUnit = createCompilationUnit(className);
        return compilationUnit.addInterface(getSimpleName(className));
    }

    private CompilationUnit createCompilationUnit(String className) {
        CompilationUnit compilationUnit = new CompilationUnit();

        // Set package name
        String packageName = getPackageName(className);
        if (!packageName.isEmpty()) {
            compilationUnit.setPackageDeclaration(packageName);
        }
        return compilationUnit;
    }

    /**
     * Creates a repository class based on the provided ServiceAndBeanInfo.
     *
     * @param serviceAndBeanInfo
     *            the information about the service and bean to create
     * @param moduleInfo
     *            the module to create the file in
     * @throws IOException
     *             if an I/O error occurs
     */
    private void createRepository(ServiceAndBeanInfo serviceAndBeanInfo, JavaSourcePathDetector.ModuleInfo moduleInfo)
            throws IOException {
        if (serviceAndBeanInfo.dataStorage != DataStorage.JPA) {
            return;
        }

        String beanName = serviceAndBeanInfo.getBeanName();
        String repositoryName = serviceAndBeanInfo.getRepositoryName();

        String filename = getFilenameFromClassName(repositoryName);
        File javaFile = new File(moduleInfo.javaSourcePaths().get(0).toFile(), filename);
        if (javaFile.exists()) {
            throw new IllegalArgumentException("File " + javaFile + " already exists");
        }

        ClassOrInterfaceDeclaration clazz = createInterface(repositoryName);
        CompilationUnit compilationUnit = clazz.findCompilationUnit().orElseThrow();

        String simpleBeanName = getSimpleName(beanName);

        // Add imports
        compilationUnit.addImport("org.springframework.data.jpa.repository.JpaRepository");
        compilationUnit.addImport("org.springframework.data.jpa.repository.JpaSpecificationExecutor");
        compilationUnit.addImport(beanName);

        // Extend JpaRepository and JpaSpecificationExecutor
        clazz.addExtendedType("JpaRepository<" + simpleBeanName + ", Long>");
        clazz.addExtendedType("JpaSpecificationExecutor<" + simpleBeanName + ">");

        ProjectFileManager.get().writeFile(javaFile, "New repository", compilationUnit.toString());
    }

}