package com.vaadin.copilot;

import java.util.Arrays;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.TreeSet;

import com.vaadin.flow.component.Component;
import com.vaadin.flow.component.Tag;
import com.vaadin.flow.server.VaadinContext;
import com.vaadin.flow.shared.Registration;
import com.vaadin.flow.shared.util.SharedUtil;

import io.github.classgraph.AnnotationParameterValueList;
import io.github.classgraph.ArrayTypeSignature;
import io.github.classgraph.ClassGraph;
import io.github.classgraph.ClassInfo;
import io.github.classgraph.ClassInfoList;
import io.github.classgraph.ClassRefTypeSignature;
import io.github.classgraph.MethodInfo;
import io.github.classgraph.MethodParameterInfo;
import io.github.classgraph.ScanResult;
import io.github.classgraph.TypeSignature;
import org.jetbrains.annotations.NotNull;

/**
 * Collects component event listener methods (methods returning
 * {@link Registration}) from Vaadin Flow components found on the classpath, and
 * optionally maps selected events to React-style attribute names.
 * <p>
 * The scan includes {@code com.vaadin.flow} and the user's application base
 * package (when Spring is available).
 *
 * @param vaadinContext
 *            Vaadin context used to detect Spring and resolve the application's
 *            base package.
 */
public record ComponentEventCollector(VaadinContext vaadinContext) {
    // Priority is used to sort events based on their method names.
    // 0 is the Highest priority
    private static final int DEFAULT_PRIORITY = 5;
    private static final Map<String, Integer> methodNamePriorityMap = new HashMap<>();
    // Some methods return Registration even though they are not listener
    private static final Set<String> methodNamesToIgnore = new HashSet<>();
    // static converter for java function name to react attributes
    private static final Map<String, String> commonJavaMethodToReactAttributeNameMap = new HashMap<>();
    static {
        methodNamePriorityMap.put("addValueListener", 0);
        methodNamePriorityMap.put("addSelectionListener", 0);
        methodNamePriorityMap.put("addItemClickListener", 1);
        methodNamePriorityMap.put("addItemDoubleClickListener", 2);
        methodNamePriorityMap.put("addValueChangeListener", 1);
        methodNamePriorityMap.put("addFocusListener", Integer.MAX_VALUE - 2);
        methodNamePriorityMap.put("addBlurListener", Integer.MAX_VALUE - 1);
        methodNamePriorityMap.put("addCompositionStartListener", Integer.MAX_VALUE - 10);
        methodNamePriorityMap.put("addCompositionUpdateListener", Integer.MAX_VALUE - 9);
        methodNamePriorityMap.put("addCompositionEndListener", Integer.MAX_VALUE - 8);
        methodNamePriorityMap.put("addAttachListener", Integer.MAX_VALUE);
        methodNamePriorityMap.put("addDetachListener", Integer.MAX_VALUE);
        methodNamePriorityMap.put("addValidationStatusChangeListener", Integer.MAX_VALUE);
        methodNamesToIgnore.add("addDataGenerator");
        methodNamesToIgnore.add("addValueProvider");
        commonJavaMethodToReactAttributeNameMap.put("addClickListener", "click");
        commonJavaMethodToReactAttributeNameMap.put("addValueChangeListener", "valueChanged");
        commonJavaMethodToReactAttributeNameMap.put("addFocusListener", "focus");
        commonJavaMethodToReactAttributeNameMap.put("addBlurListener", "blur");
        commonJavaMethodToReactAttributeNameMap.put("addCompositionStartListener", "compositionStart");
        commonJavaMethodToReactAttributeNameMap.put("addCompositionUpdateListener", "compositionUpdate");
        commonJavaMethodToReactAttributeNameMap.put("addCompositionEndListener", "compositionEnd");
        commonJavaMethodToReactAttributeNameMap.put("addKeyDownListener", "keyDown");
        commonJavaMethodToReactAttributeNameMap.put("addKeyPressListener", "keyPress");
        commonJavaMethodToReactAttributeNameMap.put("addKeyUpListener", "keyUp");
        commonJavaMethodToReactAttributeNameMap.put("addInputListener", "input");
        commonJavaMethodToReactAttributeNameMap.put("addSelectionListener", "selectedItemsChanged");
        commonJavaMethodToReactAttributeNameMap.put("addCellFocusListener", "cellFocus");
        commonJavaMethodToReactAttributeNameMap.put("addDragStartListener", "dragStart");
        commonJavaMethodToReactAttributeNameMap.put("addDragEndListener", "dragEnd");
        commonJavaMethodToReactAttributeNameMap.put("addDropListener", "drop");

    }

    /**
     * Holds the collected event listener information, grouped in two ways:
     * <ul>
     * <li>By Java fully-qualified component class name.</li>
     * <li>By Vaadin {@link Tag} name for components that have a mapped React
     * attribute name.</li>
     * </ul>
     *
     * @param javaQualifiedClassNameEventListMap
     *            map from component class FQN to its collected events
     * @param componentTagEventListMap
     *            map from tag name (e.g. {@code vaadin-button}) to its collected
     *            events
     */
    public record EventCollectionResult(Map<String, Set<ComponentEvent>> javaQualifiedClassNameEventListMap,
            Map<String, Set<ComponentEvent>> componentTagEventListMap) {

    }

    /**
     * Describes an event listener method discovered on a component type.
     */
    public static class ComponentEvent {
        private final String methodName;
        private final String humanReadableLabel;
        private final String parameterTemplate;
        private final String declaringTypeFullName;
        private final String declaringTypeBaseName;
        private String reactAttributeName;

        /**
         * Creates a new event description.
         *
         * @param methodName
         *            listener method name (e.g. {@code addClickListener})
         * @param humanReadableLabel
         *            human-friendly label derived from the method name
         * @param parameterTemplate
         *            a template snippet used when generating handler code
         * @param declaringTypeFullName
         *            fully qualified name of the declaring type
         * @param declaringTypeBaseName
         *            simple name of the declaring type
         */
        public ComponentEvent(String methodName, String humanReadableLabel, String parameterTemplate,
                String declaringTypeFullName, String declaringTypeBaseName) {
            this.methodName = methodName;
            this.humanReadableLabel = humanReadableLabel;
            this.parameterTemplate = parameterTemplate;
            this.declaringTypeFullName = declaringTypeFullName;
            this.declaringTypeBaseName = declaringTypeBaseName;
        }

        public String getMethodName() {
            return methodName;
        }

        public String getHumanReadableLabel() {
            return humanReadableLabel;
        }

        public String getParameterTemplate() {
            return parameterTemplate;
        }

        public String getDeclaringTypeFullName() {
            return declaringTypeFullName;
        }

        public String getDeclaringTypeBaseName() {
            return declaringTypeBaseName;
        }

        public String getReactAttributeName() {
            return reactAttributeName;
        }

        @Override
        public boolean equals(Object o) {
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            ComponentEvent that = (ComponentEvent) o;
            return Objects.equals(methodName, that.methodName)
                    && Objects.equals(parameterTemplate, that.parameterTemplate)
                    && Objects.equals(humanReadableLabel, that.humanReadableLabel)
                    && Objects.equals(reactAttributeName, that.reactAttributeName)
                    && Objects.equals(declaringTypeFullName, that.declaringTypeFullName)
                    && Objects.equals(declaringTypeBaseName, that.declaringTypeBaseName);
        }

        @Override
        public int hashCode() {
            return Objects.hash(methodName, humanReadableLabel, parameterTemplate, declaringTypeFullName,
                    declaringTypeBaseName, reactAttributeName);
        }

        @NotNull
        @Override
        public String toString() {
            return "ComponentEvent{" + "methodName='" + methodName + '\'' + ", humanReadableLabel='"
                    + humanReadableLabel + '\'' + ", declaringTypeFullName='" + declaringTypeFullName + '\''
                    + ", declaringTypeBaseName='" + declaringTypeBaseName + '\'' + ", reactAttributeName='"
                    + reactAttributeName + '\'' + '}';
        }
    }

    /**
     * Scans the classpath for Vaadin Flow {@link Component} subclasses and collects
     * listener-style methods.
     * <p>
     * A method is considered a candidate event listener registration when:
     * <ul>
     * <li>Its return type erases to {@link Registration}.</li>
     * <li>It is not a "signal" registration method (parameters referencing
     * {@code com.vaadin.signals}).</li>
     * <li>It is not listed in {@link #methodNamesToIgnore}.</li>
     * </ul>
     * The returned maps are sorted by a priority derived from the method name, then
     * by method name.
     *
     * @return the collected events grouped by component class and (when applicable)
     *         by {@link Tag} name
     */
    public EventCollectionResult getResult() {
        String basePackage = this.getBasePackage();
        String userProjectPackage = basePackage != null ? basePackage : "*";
        final Map<String, Set<ComponentEvent>> javaQualifiedNameEventListMap = new HashMap<>();
        final Map<String, Set<ComponentEvent>> vaadinTagNameEventListMap = new HashMap<>();
        try (ScanResult scan = new ClassGraph().acceptPackages("com.vaadin.flow", userProjectPackage).enableClassInfo()
                .enableMethodInfo().enableAnnotationInfo().scan()) {
            ClassInfoList componentClassInfoList = scan.getSubclasses(Component.class)
                    .filter(filter -> !filter.isInterface() && !filter.isAbstract());
            for (ClassInfo ci : componentClassInfoList) {
                for (MethodInfo mi : ci.getMethodInfo()) {
                    var sig = mi.getTypeSignatureOrTypeDescriptor();
                    var resultType = sig.getResultType();
                    String baseName = resolveErasedTypeName(resultType);
                    if (Registration.class.getName().equals(baseName) && !isSignalRegistrationMethod(mi)) {
                        ComponentEvent event = new ComponentEvent(mi.getName(),
                                SharedUtil.camelCaseToHumanFriendly(mi.getName()), """
                                        e -> {

                                        }
                                        """, mi.getClassInfo().getName(), mi.getClassInfo().getSimpleName());
                        setReactAttribute(event);
                        if (!methodNamesToIgnore.contains(mi.getName())) {
                            javaQualifiedNameEventListMap.computeIfAbsent(ci.getName(),
                                    k -> new TreeSet<>(getComparator()));
                            javaQualifiedNameEventListMap.get(ci.getName()).add(event);
                            putIfValidReactEvent(ci, event, vaadinTagNameEventListMap);
                        }
                    }
                }
            }
        }
        return new EventCollectionResult(javaQualifiedNameEventListMap, vaadinTagNameEventListMap);
    }

    private boolean isSignalRegistrationMethod(MethodInfo methodInfo) {
        MethodParameterInfo[] parameterInfos = methodInfo.getParameterInfo();
        return Arrays.stream(parameterInfos)
                .filter(parameterInfo -> parameterInfo.getTypeSignature() instanceof ClassRefTypeSignature)
                .map(methodParameterInfo -> ((ClassRefTypeSignature) methodParameterInfo.getTypeSignature()))
                .anyMatch(classRefTypeSignature -> classRefTypeSignature.getFullyQualifiedClassName()
                        .contains("com.vaadin.signals"));
    }

    private void setReactAttribute(ComponentEvent event) {
        String reactAttrName = null;
        if (commonJavaMethodToReactAttributeNameMap.containsKey(event.methodName)) {
            reactAttrName = commonJavaMethodToReactAttributeNameMap.get(event.methodName);
        }
        event.reactAttributeName = reactAttrName;
    }

    private void putIfValidReactEvent(ClassInfo ci, ComponentEvent event,
            Map<String, Set<ComponentEvent>> vaadinTagNameEventListMap) {
        if (event.reactAttributeName == null) {
            return;
        }
        if (ci.hasAnnotation(Tag.class)) {
            AnnotationParameterValueList parameterValues = ci.getAnnotationInfo(Tag.class).getParameterValues();
            Object value = parameterValues.getValue("value");
            if (value instanceof String tagStrValue) {
                vaadinTagNameEventListMap.computeIfAbsent(tagStrValue, k -> new TreeSet<>(getComparator())).add(event);
                vaadinTagNameEventListMap.get(tagStrValue).add(event);
            }
        }
    }

    private static Comparator<ComponentEvent> getComparator() {
        return (e1, e2) -> {
            int e1Priority = methodNamePriorityMap.getOrDefault(e1.methodName, DEFAULT_PRIORITY);
            int e2Priority = methodNamePriorityMap.getOrDefault(e2.methodName, DEFAULT_PRIORITY);
            int byPriority = Integer.compare(e1Priority, e2Priority);
            if (byPriority != 0) {
                return byPriority;
            }
            return e1.methodName.compareTo(e2.methodName);
        };
    }

    /**
     * Resolves an "erased" (raw) type name from a {@link TypeSignature}.
     * <p>
     * This is used to compare method return types against {@link Registration}
     * without requiring generic resolution.
     *
     * @param type
     *            the type signature to resolve
     * @return the fully qualified class name for class references, an array type
     *         name for array signatures, or a simple-name representation for other
     *         signature kinds
     */
    public static String resolveErasedTypeName(TypeSignature type) {
        if (type instanceof ClassRefTypeSignature cls) {
            return cls.getFullyQualifiedClassName();
        }
        if (type instanceof ArrayTypeSignature arr) {
            return resolveErasedTypeName(arr.getElementTypeSignature()) + "[]".repeat(arr.getNumDimensions());
        }
        return type.toStringWithSimpleNames();
    }

    private String getBasePackage() {
        if (!SpringBridge.isSpringAvailable(vaadinContext)) {
            return null;
        }
        Class<?> applicationClass = SpringBridge.getApplicationClass(vaadinContext);
        return applicationClass.getPackage().getName();
    }
}
