package com.vaadin.copilot;

import static com.vaadin.copilot.ConnectToService.STRING_TYPE;
import static com.vaadin.copilot.javarewriter.JavaFileSourceProvider.UNDO_LABEL;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.IdentityHashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Stream;

import com.vaadin.base.devserver.DevToolsInterface;
import com.vaadin.copilot.customcomponent.CustomComponentHelper;
import com.vaadin.copilot.customcomponent.CustomComponents;
import com.vaadin.copilot.exception.KotlinNotSupportedException;
import com.vaadin.copilot.exception.report.ExceptionReportCreator;
import com.vaadin.copilot.exception.report.ExceptionReportRelevantPairData;
import com.vaadin.copilot.ide.IdeUtils;
import com.vaadin.copilot.javarewriter.ComponentAttachInfo;
import com.vaadin.copilot.javarewriter.ComponentCreateInfo;
import com.vaadin.copilot.javarewriter.ComponentInfo;
import com.vaadin.copilot.javarewriter.ComponentInfoFinder;
import com.vaadin.copilot.javarewriter.ComponentTypeAndSourceLocation;
import com.vaadin.copilot.javarewriter.JavaBatchRewriter;
import com.vaadin.copilot.javarewriter.JavaComponent;
import com.vaadin.copilot.javarewriter.JavaDataProviderHandler;
import com.vaadin.copilot.javarewriter.JavaFileSourceProvider;
import com.vaadin.copilot.javarewriter.JavaRewriter;
import com.vaadin.copilot.javarewriter.JavaRewriterCopyPasteHandler;
import com.vaadin.copilot.javarewriter.JavaRewriterUtil;
import com.vaadin.copilot.javarewriter.JavaSource;
import com.vaadin.copilot.javarewriter.LumoRewriterUtil;
import com.vaadin.copilot.javarewriter.SourceSyncChecker;
import com.vaadin.copilot.javarewriter.exception.ComponentInfoNotFoundException;
import com.vaadin.copilot.plugins.propertypanel.ComponentPropertyHelperUtil;
import com.vaadin.copilot.theme.AppThemeUtils;
import com.vaadin.flow.component.internal.ComponentTracker;
import com.vaadin.flow.internal.JacksonUtils;

import com.github.javaparser.ast.expr.MethodCallExpr;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import tools.jackson.databind.JsonNode;
import tools.jackson.databind.node.ArrayNode;
import tools.jackson.databind.node.ObjectNode;

/**
 * Handles commands to rewrite Java source code.
 */
public class JavaRewriteHandler extends CopilotCommand {

    private static final String CAN_BE_EDITED = "canBeEdited";
    private static final String IS_TRANSLATION = "isTranslation";
    private static final String COMPONENT_ID = "componentId";
    private final SourceSyncChecker sourceSyncChecker;
    private ComponentSourceFinder sourceFinder;

    private static final String COMPONENT_PROPERTY = "component";
    private static final String PROPERTY_TO_CHECK_PROPERTY = "propertyToCheck";

    private static final String[] supportedEditableProperties = new String[] { "label", "helperText", "text", "title" };

    private interface Handler {
        void handle(JavaFileSourceProvider javaFileSourceProvider, JsonNode data, ObjectNode respData,
                ExceptionReportCreator exceptionReportCreator) throws IOException;
    }

    private static class RewriteHandler {
        private final String what;
        private final Handler handler;

        public RewriteHandler(String what, Handler handler) {
            this.what = what;
            this.handler = handler;
        }

        public void handle(JavaFileSourceProvider javaFileSourceProvider, JsonNode data, ObjectNode respData,
                ExceptionReportCreator exceptionReportCreator) throws IOException {
            this.handler.handle(javaFileSourceProvider, data, respData, exceptionReportCreator);
        }

        public String getWhat() {
            return what;
        }
    }

    private final Map<String, RewriteHandler> handlers = new HashMap<>();

    /**
     * Creates the one and only handler.
     *
     * @param sourceSyncChecker
     *            the source sync checker for detecting out of sync scenarios
     */
    public JavaRewriteHandler(SourceSyncChecker sourceSyncChecker) {
        this.sourceSyncChecker = sourceSyncChecker;
        this.sourceFinder = new ComponentSourceFinder(getVaadinSession());

        handlers.put("set-component-property",
                new RewriteHandler("set component property", this::handleSetComponentProperty));
        handlers.put("set-component-property-batch",
                new RewriteHandler("set component property batch", this::handleSetComponentPropertyBatch));
        handlers.put("add-call", new RewriteHandler("add call", this::handleAddCall));
        handlers.put("add-template", new RewriteHandler("add call", this::handleAddTemplate));
        handlers.put("delete-components", new RewriteHandler("delete components", this::handleDeleteComponents));
        handlers.put("duplicate-components",
                new RewriteHandler("duplicate components", this::handleDuplicateComponents));
        handlers.put("drag-and-drop", new RewriteHandler("drop component", this::handleDragAndDrop));
        handlers.put("set-alignment", new RewriteHandler("set alignment", this::handleAlignment));
        handlers.put("set-gap", new RewriteHandler("set gap", this::handleGap));
        handlers.put("wrap-with", new RewriteHandler("wrap with", this::handleWrapWith));
        handlers.put("set-styles", new RewriteHandler("set styles", this::handleSetStyles));
        handlers.put("set-padding", new RewriteHandler("set padding", this::handlePadding));
        handlers.put("copy", new RewriteHandler("copy", this::handleCopy));
        handlers.put("can-be-edited", new RewriteHandler("can be edited", this::handleCanBeEdited));
        handlers.put("set-sizing", new RewriteHandler("set sizing", this::handleSetSizing));
        handlers.put("connect-to-service", new RewriteHandler("connect to service", this::handleConnectToService));
        handlers.put("extract-component", new RewriteHandler("extract component", this::handleExtractComponent));
    }

    @Override
    public boolean handleMessage(String command, JsonNode data, DevToolsInterface devToolsInterface) {
        RewriteHandler handler = handlers.get(command);
        if (handler == null) {
            return false;
        }
        String reqId = data.get(KEY_REQ_ID).asString();
        ObjectNode respData = JacksonUtils.createObjectNode();
        respData.put(KEY_REQ_ID, reqId);

        JavaFileSourceProvider javaFileSourceProvider = new JavaFileSourceProvider();
        ExceptionReportCreator exceptionReportCreator = new ExceptionReportCreator();
        exceptionReportCreator.setTitle(getErrorMessage(handler));
        try {
            handler.handle(javaFileSourceProvider, data, respData, exceptionReportCreator);
            devToolsInterface.send(command + "-response", respData);
        } catch (ComponentInfoNotFoundException e) {
            if (sourceSyncChecker.maybeOutOfSync(e)) {
                ErrorHandler.sendErrorResponse(devToolsInterface, command, respData,
                        e.getComponentTypeAndSourceLocation().javaFile().map(File::getName).orElse("?")
                                + " may be out of sync. Please recompile and deploy the file",
                        e);
            } else {
                getLogger().debug("Failed to {} for input {}", handler.getWhat(), data, e);
                ErrorHandler.sendErrorResponse(devToolsInterface, command, respData, getErrorMessage(handler), e,
                        exceptionReportCreator);
            }
        } catch (Exception e) {
            getLogger().debug("Failed to {} for input {}", handler.getWhat(), data, e);
            ErrorHandler.sendErrorResponse(devToolsInterface, command, respData, getErrorMessage(handler), e,
                    exceptionReportCreator);
        }

        return true;
    }

    private String getErrorMessage(RewriteHandler handler) {
        return "Failed to " + handler.getWhat();
    }

    private void handleAddCall(JavaFileSourceProvider javaFileSourceProvider, JsonNode data, JsonNode respDat,
            ExceptionReportCreator exceptionReportCreator) throws IOException {
        String func = data.get("func").asString();
        String parameter = data.get("parameter").asString();
        Integer lineToShowInIde = data.has("lineToShowInIde") ? data.get("lineToShowInIde").asInt() : null;
        var component = data.get(COMPONENT_PROPERTY);
        exceptionReportCreator.addRelevantComponentNode("Component to Modify", component);
        exceptionReportCreator.addRelevantPair(new ExceptionReportRelevantPairData("Function", func));
        exceptionReportCreator.addRelevantPair(new ExceptionReportRelevantPairData("Parameter", parameter));

        ComponentTypeAndSourceLocation typeAndSourceLocation = sourceFinder.findTypeAndSourceLocation(component);

        ComponentInfoFinder finder = new ComponentInfoFinder(javaFileSourceProvider, typeAndSourceLocation,
                sourceFinder::getSiblingsTypeAndSourceLocations);
        ComponentInfo info = finder.find();
        exceptionReportCreator.addComponentInfo(info);
        if (!JavaRewriterUtil.hasSingleParameterMethod(info.type(), func)) {
            throw new IllegalArgumentException(
                    "Component " + info.type().getName() + " does not support the given method " + func);
        }
        var rewriter = new JavaRewriter();
        rewriter.addCall(info, func, new JavaRewriter.Code(parameter));
        javaFileSourceProvider.saveComponentInfoSourceFiles(info);

        if (lineToShowInIde != null) {
            showComponentSourceFileInIde(info, lineToShowInIde);
        }
    }

    private void showComponentSourceFileInIde(ComponentInfo info, int lineToShowInIde) {
        if (info.componentCreateInfoOptional().isPresent()) {
            ComponentCreateInfo componentCreateInfo = info.componentCreateInfoOptional().get();
            if (componentCreateInfo.getJavaSource().isChanged()) {
                showFileInIde(componentCreateInfo.getJavaSource(), lineToShowInIde);
                return;
            }
        }
        if (info.componentAttachInfoOptional().isPresent()) {
            ComponentAttachInfo componentAttachInfo = info.componentAttachInfoOptional().get();
            if (componentAttachInfo.getJavaSource().isChanged()) {
                showFileInIde(componentAttachInfo.getJavaSource(), lineToShowInIde);
            }
        }
    }

    private void showFileInIde(JavaSource source, int lineToShowInIde) {
        int firstModifiedRow = source.getFirstModifiedRow();
        int lineNumber = firstModifiedRow + lineToShowInIde;
        IdeUtils.openFile(source.getFile(), lineNumber);
        CompletableFuture.runAsync(() -> {
            try {
                // Workaround for
                // https://youtrack.jetbrains.com/issue/IDEA-342750
                Thread.sleep(1000);
                IdeUtils.openFile(source.getFile(), lineNumber);
            } catch (InterruptedException e) {
                getLogger().error("Failed to show file in IDE", e);
                Thread.currentThread().interrupt();
            }
        });
    }

    private void handleDeleteComponents(JavaFileSourceProvider javaFileSourceProvider, JsonNode data,
            ObjectNode respData, ExceptionReportCreator exceptionReportCreator) {
        ArrayNode componentsJson = data.withArray("components");
        String activeJavaClassName = safeGet(data, "activeJavaClassName");
        boolean drilledDownComponent;
        if (activeJavaClassName != null) {
            // for components that are not a descendant of a non custom components e.g.
            // components from main layout.
            drilledDownComponent = CustomComponents.isCustomComponent(activeJavaClassName);
        } else {
            drilledDownComponent = false;
        }

        List<ComponentTypeAndSourceLocation> components = new ArrayList<>();
        for (int i = 0; i < componentsJson.size(); i++) {
            List<ComponentTypeAndSourceLocation> found = findTypeAndSourceLocationIncludingChildren(
                    componentsJson.get(i));
            exceptionReportCreator.addRelevantComponentNode("Delete requested", componentsJson.get(i));
            components.addAll(found);
        }
        List<ComponentInfo> list = components.stream().filter(typeAndSourceLocation -> Stream
                .of(typeAndSourceLocation.createLocationInProject(), typeAndSourceLocation.attachLocationInProject())
                .flatMap(Optional::stream).map(ComponentTracker.Location::className).anyMatch(locationClass -> {
                    if (drilledDownComponent) {
                        if (locationClass.contains("$")) {
                            return locationClass.substring(0, locationClass.indexOf("$")).equals(activeJavaClassName);
                        }
                        return locationClass.equals(activeJavaClassName);
                    }
                    // the component might be in a layout related class
                    boolean customComponent = CustomComponents.isCustomComponent(locationClass);
                    return !customComponent;
                })).map(typeAndSourceLocation -> {
                    try {
                        return new ComponentInfoFinder(javaFileSourceProvider, typeAndSourceLocation,
                                sourceFinder::getSiblingsTypeAndSourceLocations).find();
                    } catch (IOException e) {
                        throw new IllegalArgumentException("Unable to find component info");
                    }
                }).toList();
        exceptionReportCreator.addComponentInfo(list);
        JavaRewriter javaRewriter = new JavaRewriter();
        list.forEach(javaRewriter::delete);
        list.forEach(javaFileSourceProvider::saveComponentInfoSourceFiles);
    }

    private List<ComponentTypeAndSourceLocation> findTypeAndSourceLocationIncludingChildren(JsonNode object) {
        ArrayList<ComponentTypeAndSourceLocation> all = new ArrayList<>();
        ComponentTypeAndSourceLocation root = sourceFinder.findTypeAndSourceLocation(object, true);

        addRecursively(all, root);
        return all;
    }

    /**
     * Finds the type and source location of the given component and its first level
     * children. The target of this method is to identify a selected component in
     * the UI that is why is not necessary to find all the children but just the
     * first level ones to make equals method to work.
     *
     * @param object
     *            the component
     * @return the type and source location of the component and its first level
     *         children
     */
    private ComponentTypeAndSourceLocation findTypeAndSourceLocationIncludingFirstLevelChildren(JsonNode object) {
        ComponentTypeAndSourceLocation root = sourceFinder.findTypeAndSourceLocation(object, true);
        return root;
    }

    private void addRecursively(ArrayList<ComponentTypeAndSourceLocation> all, ComponentTypeAndSourceLocation root) {
        all.add(root);
        addChildrenRecursively(all, root, root.javaFile().orElse(null));
    }

    private void addChildrenRecursively(ArrayList<ComponentTypeAndSourceLocation> all,
            ComponentTypeAndSourceLocation parent, File rootFile) {
        for (ComponentTypeAndSourceLocation child : parent.children()) {
            File childFile = child.javaFile().orElse(null);
            boolean childSourcesInProject = childFile != null && childFile.exists();
            boolean childInRootFile = rootFile != null && rootFile.equals(childFile);

            if (childSourcesInProject && childInRootFile) {
                // The child is in the root file - so e.g. if duplicating a component in
                // HelloWorldView,
                // this child in also in HelloWorldView.java and we include it.
                all.add(child);
            }
            // Regardless of where the child is, we need to continue down to see if we find
            // more children that are in the same file. Either an intermediate child can be
            // in an external component or inside a project custom component.
            addChildrenRecursively(all, child, rootFile);
        }
    }

    private void handleCopy(JavaFileSourceProvider javaFileSourceProvider, JsonNode data, ObjectNode response,
            ExceptionReportCreator exceptionReportCreator) throws IOException {
        int componentId = data.get(COMPONENT_ID).asInt();

        int uiId = data.get("uiId").asInt();
        ComponentTypeAndSourceLocation copiedComponent = sourceFinder.findTypeAndSourceLocation(uiId, componentId,
                true);

        JavaRewriterCopyPasteHandler handler = new JavaRewriterCopyPasteHandler();
        JavaComponent copiedJavaComponent = handler.getCopiedJavaComponent(javaFileSourceProvider, copiedComponent,
                exceptionReportCreator, sourceFinder);
        String s = CopilotJacksonUtils.writeValueAsString(copiedJavaComponent);
        response.put(COMPONENT_PROPERTY, s);
    }

    private void handleDuplicateComponents(JavaFileSourceProvider javaFileSourceProvider, JsonNode data,
            ObjectNode respData, ExceptionReportCreator exceptionReportCreator) {
        ArrayNode componentsJson = data.withArray("components");
        ArrayList<ComponentTypeAndSourceLocation> components = new ArrayList<>();
        List<ComponentTypeAndSourceLocation> selectedComponents = new ArrayList<>();
        for (int i = 0; i < componentsJson.size(); i++) {
            ComponentTypeAndSourceLocation root = sourceFinder.findTypeAndSourceLocation(componentsJson.get(i), true);
            exceptionReportCreator.addRelevantComponentNode("Duplicate", componentsJson.get(i));
            selectedComponents.add(root);
            addRecursively(components, root);
        }

        JavaBatchRewriter batchRewriter = new JavaBatchRewriter(javaFileSourceProvider, components,
                exceptionReportCreator, sourceFinder);
        selectedComponents.forEach(batchRewriter::duplicate);
        batchRewriter.writeResult();
    }

    private void handleWrapWith(JavaFileSourceProvider javaFileSourceProvider, JsonNode data, ObjectNode respData,
            ExceptionReportCreator exceptionReportCreator) throws IOException {
        ArrayNode componentsJson = data.withArray("components");
        JavaComponent wrapperComponent = JavaComponent.componentFromJson(data.get("wrapperComponent"));
        List<ComponentTypeAndSourceLocation> components = new ArrayList<>();
        for (int i = 0; i < componentsJson.size(); i++) {
            components.add(sourceFinder.findTypeAndSourceLocation(componentsJson.get(i)));
        }
        File javaFile = getProjectFileManager().getSourceFile(components.get(0).getCreateLocationOrThrow());
        KotlinUtil.throwIfKotlin(javaFile);

        JavaRewriter rewriter = new JavaRewriter();

        List<ComponentInfo> componentInfos = new ArrayList<>();
        for (ComponentTypeAndSourceLocation typeAndSourceLocation : components) {
            ComponentInfo componentInfo = new ComponentInfoFinder(javaFileSourceProvider, typeAndSourceLocation,
                    sourceFinder::getSiblingsTypeAndSourceLocations).find();
            componentInfos.add(componentInfo);
        }
        exceptionReportCreator.addComponentInfo(componentInfos);
        rewriter.mergeAndReplace(componentInfos, wrapperComponent);
        for (ComponentInfo componentInfo : componentInfos) {
            javaFileSourceProvider.saveComponentInfoSourceFiles(componentInfo);
        }
    }

    private void handleSetComponentPropertyBatch(JavaFileSourceProvider javaFileSourceProvider, JsonNode data,
            ObjectNode respData, ExceptionReportCreator exceptionReportCreator) throws IOException {
        ArrayNode properties = data.withArrayProperty("properties");
        int propertiesCount = properties.size();

        // Set write mode only for the last property to optimize file writes
        javaFileSourceProvider.setWriteMode(false);
        boolean propertyFailed = false;

        for (int i = 0; i < propertiesCount; i++) {
            JsonNode property = properties.get(i);
            boolean isLast = (i == propertiesCount - 1);
            if (isLast) {
                javaFileSourceProvider.setWriteMode(true);
            }
            try {
                handleSetComponentProperty(javaFileSourceProvider, property, respData, exceptionReportCreator);
            } catch (KotlinNotSupportedException ke) {
                getLogger().error("Error handling multi set property Kotlin not supported {}", property, ke);
                throw ke;
            } catch (IOException e) {
                propertyFailed = true;
                getLogger().error("Error handling multi set property {}", property, e);
            }
        }
        if (propertyFailed) {
            throw new IOException("One or more properties failed to be set");
        }
    }

    private void handleSetComponentProperty(JavaFileSourceProvider javaFileSourceProvider, JsonNode data,
            ObjectNode respData, ExceptionReportCreator exceptionReportCreator) throws IOException {
        String property = data.get("property").asString();
        Object value;
        try {
            value = ComponentPropertyHelperUtil.extractPropertyValueFromJson(data);
        } catch (Exception e) {
            exceptionReportCreator.addRelevantPair(new ExceptionReportRelevantPairData("Request", data.toString()));
            throw e;
        }

        var component = data.get("component");

        ComponentTypeAndSourceLocation typeAndSourceLocation = sourceFinder.findTypeAndSourceLocation(component);
        if (JavaDataProviderHandler.isDataProviderItemChange(typeAndSourceLocation)
                && value instanceof String valueStr) {
            JavaDataProviderHandler dataProviderHandler = new JavaDataProviderHandler(javaFileSourceProvider,
                    typeAndSourceLocation, exceptionReportCreator, sourceFinder);
            JavaSource javaSource = dataProviderHandler.handleSetComponentProperty(javaFileSourceProvider, property,
                    valueStr);
            getProjectFileManager().writeFile(javaSource.getFile(), UNDO_LABEL, javaSource.getResult());
            return;
        }
        ComponentInfo componentInfo = new ComponentInfoFinder(javaFileSourceProvider, typeAndSourceLocation,
                sourceFinder::getSiblingsTypeAndSourceLocations).find();
        exceptionReportCreator.addComponentInfo(componentInfo);
        var rewriter = new JavaRewriter();
        String setter = JavaRewriterUtil.getSetterName(property, componentInfo.type(), true);
        if (value != null) {
            JavaRewriter.ReplaceResult result = rewriter.replaceFunctionCall(componentInfo, setter, value);
            if (result.variableRenamedTo() != null) {
                respData.put("variableRenamedTo", result.variableRenamedTo());
            }
        } else {
            // removing calls for null values
            rewriter.removePropertySetter(componentInfo, setter);
        }

        javaFileSourceProvider.saveComponentInfoSourceFiles(componentInfo);
    }

    private void handleAddTemplate(JavaFileSourceProvider javaFileSourceProvider, JsonNode data, ObjectNode respData,
            ExceptionReportCreator exceptionReportCreator) throws IOException {
        final String whereJsonKey = "where";
        final String refNodeJsonKey = "refNode";
        List<JavaComponent> template = JavaComponent.componentsFromJson(data.withArray("template"));
        JavaRewriter.Where where = null;
        if (data.hasNonNull(whereJsonKey)) {
            where = JavaRewriter.Where.valueOf(data.get(whereJsonKey).asString().toUpperCase(Locale.ENGLISH));
        }
        ComponentTypeAndSourceLocation refSource = getRealOrAnalyzedComponentTypeAndSourceLocation(data,
                refNodeJsonKey);
        exceptionReportCreator.addRelevantPair(new ExceptionReportRelevantPairData("Where", where));
        exceptionReportCreator.addRelevantComponentNode("Reference Node", data.get(refNodeJsonKey));

        var componentInfo = new ComponentInfoFinder(javaFileSourceProvider, refSource,
                sourceFinder::getSiblingsTypeAndSourceLocations).find();
        exceptionReportCreator.addComponentInfo(componentInfo);
        var rewriter = new JavaRewriter();
        var customComponentApiSelection = CustomComponentHelper.extractFromRequest(data);

        JavaRewriter.AddTemplateOptions options = new JavaRewriter.AddTemplateOptions(
                data.get("javaFieldsForLeafComponents").asBoolean(), false,
                customComponentApiSelection != null ? customComponentApiSelection.methodName() : null,
                customComponentApiSelection != null ? customComponentApiSelection.action() : null);
        exceptionReportCreator.addRelevantPair(new ExceptionReportRelevantPairData("Selected API",
                CopilotJacksonUtils.writeValueAsString(customComponentApiSelection)));
        rewriter.addComponentUsingTemplate(componentInfo, where, template, options);
        javaFileSourceProvider.saveComponentInfoSourceFiles(componentInfo);
    }

    private void handleDragAndDrop(JavaFileSourceProvider javaFileSourceProvider, JsonNode data, ObjectNode respData,
            ExceptionReportCreator exceptionReportCreator) throws IOException {
        JavaRewriter.Where where = JavaRewriter.Where.valueOf(data.get("where").asString().toUpperCase(Locale.ENGLISH));

        ComponentTypeAndSourceLocation dragged = getRealOrAnalyzedComponentTypeAndSourceLocation(data, "dragged");
        ComponentTypeAndSourceLocation container = getRealOrAnalyzedComponentTypeAndSourceLocation(data, "container");

        ComponentTypeAndSourceLocation insertBefore = where == JavaRewriter.Where.BEFORE
                ? sourceFinder.findTypeAndSourceLocation(data.get("insertBefore"))
                : null;

        var componentInfo = new ComponentInfoFinder(javaFileSourceProvider, container,
                sourceFinder::getSiblingsTypeAndSourceLocations).find();
        var rewriter = new JavaRewriter();

        if (!dragged.javaFile().equals(container.javaFile())) {
            throw new IllegalArgumentException("Cannot move a component in one file (" + dragged.javaFile()
                    + ") to another file (" + container.javaFile() + ")");
        }

        var customComponentApiSelection = CustomComponentHelper.extractFromRequest(data);
        if (customComponentApiSelection != null) {
            exceptionReportCreator.addRelevantPair(new ExceptionReportRelevantPairData("Selected API",
                    CopilotJacksonUtils.writeValueAsString(customComponentApiSelection)));
        }

        JavaRewriter.MoveTemplateOptions options = new JavaRewriter.MoveTemplateOptions(
                customComponentApiSelection != null ? customComponentApiSelection.methodName() : null,
                customComponentApiSelection != null ? customComponentApiSelection.action() : null);

        ComponentInfo draggedRef = new ComponentInfoFinder(javaFileSourceProvider, dragged,
                sourceFinder::getSiblingsTypeAndSourceLocations).find();
        ComponentInfo containerRef = new ComponentInfoFinder(javaFileSourceProvider, container,
                sourceFinder::getSiblingsTypeAndSourceLocations).find();
        ComponentInfo insertBeforeRef = insertBefore == null ? null
                : new ComponentInfoFinder(javaFileSourceProvider, insertBefore,
                        sourceFinder::getSiblingsTypeAndSourceLocations).find();
        exceptionReportCreator.addComponentInfo(draggedRef);

        rewriter.moveComponent(draggedRef, containerRef, insertBeforeRef, where, options);
        javaFileSourceProvider.saveComponentInfoSourceFiles(componentInfo);
    }

    /**
     * Gets the component type and source location by considering if given node is
     * the drilled down component.
     *
     * @param data
     *            request data
     * @param nodeKey
     *            key to get component nodeId and uiId info
     * @return source from sourceFinder if not drilled down.
     * @throws IOException
     *             is thrown when file operation fails.
     */
    private ComponentTypeAndSourceLocation getRealOrAnalyzedComponentTypeAndSourceLocation(JsonNode data,
            String nodeKey) throws IOException {
        if (CustomComponentHelper.isDrilledDownComponent(getVaadinSession(), data, data.get(nodeKey))) {
            JsonNode component = data.get(nodeKey);
            return sourceFinder
                    .analyzeSourceFileAndGetComponentTypeAndSourceLocation(component.get("nodeId").asInt(),
                            component.get("uiId").asInt())
                    .orElseThrow(() -> new RuntimeException("Could not find ref source file"));
        } else {
            return sourceFinder.findTypeAndSourceLocation(data.get(nodeKey));
        }
    }

    private void handleAlignment(JavaFileSourceProvider javaFileSourceProvider, JsonNode data, ObjectNode respData,
            ExceptionReportCreator exceptionReportCreator) throws IOException {
        ComponentTypeAndSourceLocation componentTypeAndSourceLocation = sourceFinder
                .findTypeAndSourceLocation(data.get(COMPONENT_ID));
        JavaRewriter.AlignmentValues alignmentValues = JacksonUtils.readToObject(data.get("values"),
                JavaRewriter.AlignmentValues.class);
        ComponentInfo componentInfo = new ComponentInfoFinder(javaFileSourceProvider, componentTypeAndSourceLocation,
                sourceFinder::getSiblingsTypeAndSourceLocations).find();
        exceptionReportCreator.addComponentInfo(componentInfo);
        var rewriter = new JavaRewriter();
        rewriter.setAlignment(componentInfo, alignmentValues);
        javaFileSourceProvider.saveComponentInfoSourceFiles(componentInfo);
        addLumoUtilityClassesIfLumoTheme();
    }

    private void handleSetStyles(JavaFileSourceProvider javaFileSourceProvider, JsonNode data, ObjectNode respData,
            ExceptionReportCreator exceptionReportCreator) throws IOException {
        ComponentTypeAndSourceLocation componentTypeAndSourceLocation = sourceFinder
                .findTypeAndSourceLocation(data.get(COMPONENT_ID));
        ArrayNode added = data.withArray("added");
        ArrayNode removed = data.withArray("removed");
        Set<String> toRemove = new HashSet<>();
        for (int i = 0; i < removed.size(); i++) {
            toRemove.add(removed.get(i).get("key").asString());
        }
        ComponentInfo componentInfo = new ComponentInfoFinder(javaFileSourceProvider, componentTypeAndSourceLocation,
                sourceFinder::getSiblingsTypeAndSourceLocations).find();
        exceptionReportCreator.addComponentInfo(componentInfo);
        var rewriter = new JavaRewriter();

        for (int i = 0; i < added.size(); i++) {
            JsonNode rule = added.get(i);
            String key = rule.get("key").asString();
            String value = rule.get("value").asString();
            rewriter.setStyle(componentInfo, key, value);
            toRemove.remove(key);
        }

        for (String key : toRemove) {
            rewriter.setStyle(componentInfo, key, null);
        }
        javaFileSourceProvider.saveComponentInfoSourceFiles(componentInfo);
    }

    private void handleConnectToService(JavaFileSourceProvider javaFileSourceProvider, JsonNode data,
            ObjectNode respData, ExceptionReportCreator exceptionReportCreator) throws IOException {
        if (!SpringBridge.isSpringAvailable(getVaadinContext())) {
            throw new IllegalStateException("Connecting to a service is only supported for Spring applications");
        }

        ComponentTypeAndSourceLocation componentTypeAndSourceLocation = sourceFinder
                .findTypeAndSourceLocation(data.get(COMPONENT_ID));
        ComponentInfoFinder componentInfoFinder = new ComponentInfoFinder(javaFileSourceProvider,
                componentTypeAndSourceLocation, sourceFinder::getSiblingsTypeAndSourceLocations);
        ComponentInfo componentInfo = componentInfoFinder.find();
        exceptionReportCreator.addComponentInfo(componentInfo);
        var rewriter = new JavaRewriter();
        JsonNode service = data.get("service");

        String serviceClassName = service.get("className").asString();
        String serviceMethodName = service.get("methodName").asString();
        JavaReflectionUtil.TypeInfo methodReturnType = JacksonUtils.readToObject(service.get("returnType"),
                JavaReflectionUtil.TypeInfo.class);
        List<JavaReflectionUtil.ParameterTypeInfo> parameters = JacksonUtils.stream(service.withArray("parameters"))
                .map(parameter -> JacksonUtils.readToObject(parameter, JavaReflectionUtil.ParameterTypeInfo.class))
                .toList();

        if (methodReturnType.typeParameters().size() != 1) {
            throw new IllegalArgumentException("Method return type " + methodReturnType + " is not a List of items");
        }
        JavaReflectionUtil.TypeInfo itemTypeInfo = methodReturnType.typeParameters().get(0);
        // Grid<Something> -> Grid<Product>

        String itemType = itemTypeInfo.typeName();
        List<UIServiceCreator.FieldInfo> propertiesInSourceOrder = JavaRewriterUtil
                .getPropertiesInSourceOrder(itemType);

        // This is very common to have and you don't want to show it
        propertiesInSourceOrder.removeIf(fieldInfo -> fieldInfo.name().equals("id"));

        if (isGrid(componentInfo)) {
            rewriter.setGridDataSource(componentInfo, serviceClassName, serviceMethodName, parameters, itemType,
                    propertiesInSourceOrder);
        } else if (isComboBox(componentInfo)) {
            UIServiceCreator.FieldInfo firstStringProperty = propertiesInSourceOrder.stream()
                    .filter(f -> f.javaType().equals(STRING_TYPE)).findFirst().orElse(null);
            rewriter.setComboBoxDataSource(componentInfo, serviceClassName, serviceMethodName, parameters, itemType,
                    firstStringProperty);
        } else {
            throw new IllegalArgumentException(
                    "Connecting to a service is not supported for " + componentInfo.type().getName());
        }
        javaFileSourceProvider.saveComponentInfoSourceFiles(componentInfo);
    }

    private boolean isGrid(ComponentInfo componentInfo) {
        return isComponent(componentInfo, "com.vaadin.flow.component.grid.Grid");
    }

    private boolean isComboBox(ComponentInfo componentInfo) {
        return isComponent(componentInfo, "com.vaadin.flow.component.combobox.ComboBox");
    }

    private boolean isComponent(ComponentInfo componentInfo, String componentClass) {
        try {
            return Class.forName(componentClass).isAssignableFrom(componentInfo.type());
        } catch (Exception e) {
            return false;
        }

    }

    private void handleExtractComponent(JavaFileSourceProvider javaFileSourceProvider, JsonNode data,
            ObjectNode respData, ExceptionReportCreator exceptionReportCreator) throws IOException {
        ComponentTypeAndSourceLocation rootComponent = sourceFinder.findTypeAndSourceLocation(data.get("component"),
                true);
        ComponentInfo rootComponentInfo = new ComponentInfoFinder(javaFileSourceProvider, rootComponent,
                sourceFinder::getSiblingsTypeAndSourceLocations).find();
        exceptionReportCreator.addComponentInfo(rootComponentInfo);
        JavaRewriter rewriter = new JavaRewriter();

        IdentityHashMap<ComponentInfo, List<ComponentInfo>> childrenMap = new IdentityHashMap<>();
        populateChildren(rootComponent, rootComponentInfo, childrenMap, javaFileSourceProvider);
        String generatedClassName = rewriter.extractComponent(rootComponentInfo, childrenMap, "MyComponent");
        javaFileSourceProvider.saveComponentInfoSourceFiles(rootComponentInfo);
        // send data to UI so a notification can be linked to the generated component
        respData.put("generatedClassName", generatedClassName);
        ComponentTracker.Location createLocationOrThrow = rootComponent.getCreateLocationOrThrow();
        respData.put("fileName", createLocationOrThrow.filename());
        respData.put("filePath", rootComponentInfo.getCreateLocationFileOrThrowIfNull().getPath());
        JavaRewriterUtil.findLineContaining(rootComponentInfo.getCreateLocationCompilationUnitOrThrowIfNull(),
                generatedClassName).ifPresent(lineNumber -> respData.put("lineNumber", lineNumber + 1));
    }

    private void populateChildren(ComponentTypeAndSourceLocation component, ComponentInfo componentInfo,
            IdentityHashMap<ComponentInfo, List<ComponentInfo>> childrenMap,
            JavaFileSourceProvider javaFileSourceProvider) throws IOException {
        List<ComponentTypeAndSourceLocation> children = component.children();
        List<ComponentInfo> childComponentInfo = new ArrayList<>();
        for (ComponentTypeAndSourceLocation child : children) {
            if (child.createLocationInProject().isEmpty()) {
                continue;
            }
            ComponentInfo childInfo = new ComponentInfoFinder(javaFileSourceProvider, child,
                    sourceFinder::getSiblingsTypeAndSourceLocations).find();
            populateChildren(child, childInfo, childrenMap, javaFileSourceProvider);
            childComponentInfo.add(childInfo);
        }
        childrenMap.put(componentInfo, childComponentInfo);
    }

    private void handleSetSizing(JavaFileSourceProvider javaFileSourceProvider, JsonNode data, ObjectNode respData,
            ExceptionReportCreator exceptionReportCreator) throws IOException {
        ComponentTypeAndSourceLocation componentTypeAndSourceLocation = sourceFinder
                .findTypeAndSourceLocation(data.get(COMPONENT_ID));

        var componentInfo = new ComponentInfoFinder(javaFileSourceProvider, componentTypeAndSourceLocation,
                sourceFinder::getSiblingsTypeAndSourceLocations).find();
        exceptionReportCreator.addComponentInfo(componentInfo);
        JsonNode changesJson = data.get("changes");
        HashMap<String, String> changes = CopilotJacksonUtils.toMap(changesJson);
        JavaRewriter rewriter = new JavaRewriter();
        rewriter.setSizing(componentInfo, changes);
        javaFileSourceProvider.saveComponentInfoSourceFiles(componentInfo);
    }

    private void handleGap(JavaFileSourceProvider javaFileSourceProvider, JsonNode data, ObjectNode respData,
            ExceptionReportCreator exceptionReportCreator) throws IOException {
        ComponentTypeAndSourceLocation componentTypeAndSourceLocation = sourceFinder
                .findTypeAndSourceLocation(data.get(COMPONENT_ID));
        String lumoClassAll = safeGet(data, "all");
        String lumoClassRow = safeGet(data, "row");
        String lumoClassColumn = safeGet(data, "column");
        ComponentInfo componentInfo = new ComponentInfoFinder(javaFileSourceProvider, componentTypeAndSourceLocation,
                sourceFinder::getSiblingsTypeAndSourceLocations).find();
        exceptionReportCreator.addComponentInfo(componentInfo);
        exceptionReportCreator.addRelevantPair(new ExceptionReportRelevantPairData("All", lumoClassAll));
        exceptionReportCreator.addRelevantPair(new ExceptionReportRelevantPairData("Row", lumoClassRow));
        exceptionReportCreator.addRelevantPair(new ExceptionReportRelevantPairData("Column", lumoClassColumn));
        JavaRewriter rewriter = new JavaRewriter();
        rewriter.setGap(componentInfo, lumoClassAll, lumoClassColumn, lumoClassRow);
        javaFileSourceProvider.saveComponentInfoSourceFiles(componentInfo);
        javaFileSourceProvider.throwExceptionIfNoChanges(componentInfo);
        addLumoUtilityClassesIfLumoTheme();
    }

    private void handlePadding(JavaFileSourceProvider javaFileSourceProvider, JsonNode data, ObjectNode respData,
            ExceptionReportCreator exceptionReportCreator) throws IOException {
        ComponentTypeAndSourceLocation componentTypeAndSourceLocation = sourceFinder
                .findTypeAndSourceLocation(data.get(COMPONENT_ID));
        String allPaddingLumoClass = safeGet(data, "all");
        String topPaddingLumoClass = safeGet(data, "top");
        String rightPaddingLumoClass = safeGet(data, "right");
        String bottomPaddingLumoClass = safeGet(data, "bottom");
        String leftPaddingLumoClass = safeGet(data, "left");
        var componentInfo = new ComponentInfoFinder(javaFileSourceProvider, componentTypeAndSourceLocation,
                sourceFinder::getSiblingsTypeAndSourceLocations).find();
        exceptionReportCreator.addComponentInfo(componentInfo);
        JavaRewriter rewriter = new JavaRewriter();
        rewriter.setPadding(componentInfo, allPaddingLumoClass, topPaddingLumoClass, rightPaddingLumoClass,
                bottomPaddingLumoClass, leftPaddingLumoClass);
        javaFileSourceProvider.saveComponentInfoSourceFiles(componentInfo);
        addLumoUtilityClassesIfLumoTheme();
    }

    // TODO remove this and use a proper way.
    private String safeGet(JsonNode data, String key) {
        return data.hasNonNull(key) ? data.get(key).asString() : null;
    }

    private void handleCanBeEdited(JavaFileSourceProvider javaFileSourceProvider, JsonNode data, ObjectNode response,
            ExceptionReportCreator exceptionReportCreator) throws IOException {
        var component = data.get(COMPONENT_PROPERTY);
        String propertyToCheck = data.get(PROPERTY_TO_CHECK_PROPERTY).asString();

        if (Arrays.stream(supportedEditableProperties)
                .noneMatch(property -> property.equalsIgnoreCase(propertyToCheck))) {
            response.put(CAN_BE_EDITED, false);
            response.put(IS_TRANSLATION, false);
            return;
        }
        var typeAndSourceLocation = sourceFinder.findTypeAndSourceLocation(component);
        ComponentInfoFinder finder = new ComponentInfoFinder(javaFileSourceProvider, typeAndSourceLocation,
                sourceFinder::getSiblingsTypeAndSourceLocations);
        ComponentInfo info = finder.find();
        exceptionReportCreator.addComponentInfo(info);
        var rewriter = new JavaRewriter();

        try {
            var result = rewriter.getPropertyValue(info, propertyToCheck);
            response.put(CAN_BE_EDITED, result instanceof String);
            if (result instanceof MethodCallExpr methodCallExpr) {
                response.put(IS_TRANSLATION, methodCallExpr.getNameAsString().equals("translate"));
            } else {
                response.put(IS_TRANSLATION, false);
            }
        } catch (Exception e) {
            getLogger().error("Failed to check if property can be edited", e);
            response.put(CAN_BE_EDITED, false);
            response.put(IS_TRANSLATION, false);
        }
    }

    private void addLumoUtilityClassesIfLumoTheme() throws IOException {
        if (AppThemeUtils.isLumo()) {
            LumoRewriterUtil.addLumoUtilityStylesheetIfNotExists(getVaadinContext());
        }
    }

    private Logger getLogger() {
        return LoggerFactory.getLogger(getClass());
    }
}
