package com.vaadin.copilot;

import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.TreeSet;

import com.vaadin.flow.component.Component;
import com.vaadin.flow.component.Tag;
import com.vaadin.flow.server.VaadinContext;
import com.vaadin.flow.shared.Registration;
import com.vaadin.flow.shared.util.SharedUtil;

import io.github.classgraph.AnnotationParameterValueList;
import io.github.classgraph.ArrayTypeSignature;
import io.github.classgraph.ClassGraph;
import io.github.classgraph.ClassInfo;
import io.github.classgraph.ClassInfoList;
import io.github.classgraph.ClassRefTypeSignature;
import io.github.classgraph.MethodInfo;
import io.github.classgraph.ScanResult;
import io.github.classgraph.TypeSignature;
import org.jetbrains.annotations.NotNull;

public record ComponentEventCollector(VaadinContext vaadinContext) {
    // Priority is used to sort events based on their method names.
    // 0 is the Highest priority
    private static final int DEFAULT_PRIORITY = 5;
    private static final Map<String, Integer> methodNamePriorityMap = new HashMap<>();
    // Some methods return Registration even though they are not listener
    private static final Set<String> methodNamesToIgnore = new HashSet<>();
    // static converter for java function name to react attributes
    private static final Map<String, String> commonJavaMethodToReactAttributeNameMap = new HashMap<>();
    static {
        methodNamePriorityMap.put("addValueListener", 0);
        methodNamePriorityMap.put("addSelectionListener", 0);
        methodNamePriorityMap.put("addItemClickListener", 1);
        methodNamePriorityMap.put("addItemDoubleClickListener", 2);
        methodNamePriorityMap.put("addValueChangeListener", 1);
        methodNamePriorityMap.put("addFocusListener", Integer.MAX_VALUE - 2);
        methodNamePriorityMap.put("addBlurListener", Integer.MAX_VALUE - 1);
        methodNamePriorityMap.put("addCompositionStartListener", Integer.MAX_VALUE - 10);
        methodNamePriorityMap.put("addCompositionUpdateListener", Integer.MAX_VALUE - 9);
        methodNamePriorityMap.put("addCompositionEndListener", Integer.MAX_VALUE - 8);
        methodNamePriorityMap.put("addAttachListener", Integer.MAX_VALUE);
        methodNamePriorityMap.put("addDetachListener", Integer.MAX_VALUE);
        methodNamePriorityMap.put("addValidationStatusChangeListener", Integer.MAX_VALUE);
        methodNamesToIgnore.add("addDataGenerator");
        methodNamesToIgnore.add("addValueProvider");
        commonJavaMethodToReactAttributeNameMap.put("addClickListener", "click");
        commonJavaMethodToReactAttributeNameMap.put("addValueChangeListener", "valueChanged");
        commonJavaMethodToReactAttributeNameMap.put("addFocusListener", "focus");
        commonJavaMethodToReactAttributeNameMap.put("addBlurListener", "blur");
        commonJavaMethodToReactAttributeNameMap.put("addCompositionStartListener", "compositionStart");
        commonJavaMethodToReactAttributeNameMap.put("addCompositionUpdateListener", "compositionUpdate");
        commonJavaMethodToReactAttributeNameMap.put("addCompositionEndListener", "compositionEnd");
        commonJavaMethodToReactAttributeNameMap.put("addKeyDownListener", "keyDown");
        commonJavaMethodToReactAttributeNameMap.put("addKeyPressListener", "keyPress");
        commonJavaMethodToReactAttributeNameMap.put("addKeyUpListener", "keyUp");
        commonJavaMethodToReactAttributeNameMap.put("addInputListener", "input");
        commonJavaMethodToReactAttributeNameMap.put("addSelectionListener", "selectedItemsChanged");
        commonJavaMethodToReactAttributeNameMap.put("addCellFocusListener", "cellFocus");
        commonJavaMethodToReactAttributeNameMap.put("addDragStartListener", "dragStart");
        commonJavaMethodToReactAttributeNameMap.put("addDragEndListener", "dragEnd");
        commonJavaMethodToReactAttributeNameMap.put("addDropListener", "drop");

    }

    public record EventCollectionResult(Map<String, Set<ComponentEvent>> javaQualifiedClassNameEventListMap,
            Map<String, Set<ComponentEvent>> componentTagEventListMap) {

    }

    public static class ComponentEvent {
        private String methodName;
        private String humanReadableLabel;
        private String parameterTemplate;
        private String declaringTypeFullName;
        private String declaringTypeBaseName;
        private String reactAttributeName;

        public ComponentEvent(String methodName, String humanReadableLabel, String parameterTemplate,
                String declaringTypeFullName, String declaringTypeBaseName) {
            this.methodName = methodName;
            this.humanReadableLabel = humanReadableLabel;
            this.parameterTemplate = parameterTemplate;
            this.declaringTypeFullName = declaringTypeFullName;
            this.declaringTypeBaseName = declaringTypeBaseName;
        }

        public String getMethodName() {
            return methodName;
        }

        public String getHumanReadableLabel() {
            return humanReadableLabel;
        }

        public String getParameterTemplate() {
            return parameterTemplate;
        }

        public String getDeclaringTypeFullName() {
            return declaringTypeFullName;
        }

        public String getDeclaringTypeBaseName() {
            return declaringTypeBaseName;
        }

        public String getReactAttributeName() {
            return reactAttributeName;
        }

        @Override
        public boolean equals(Object o) {
            if (o == null || getClass() != o.getClass())
                return false;
            ComponentEvent that = (ComponentEvent) o;
            return Objects.equals(methodName, that.methodName)
                    && Objects.equals(parameterTemplate, that.parameterTemplate)
                    && Objects.equals(humanReadableLabel, that.humanReadableLabel)
                    && Objects.equals(reactAttributeName, that.reactAttributeName)
                    && Objects.equals(declaringTypeFullName, that.declaringTypeFullName)
                    && Objects.equals(declaringTypeBaseName, that.declaringTypeBaseName);
        }

        @Override
        public int hashCode() {
            return Objects.hash(methodName, humanReadableLabel, parameterTemplate, declaringTypeFullName,
                    declaringTypeBaseName, reactAttributeName);
        }

        @NotNull
        @Override
        public String toString() {
            return "ComponentEvent{" + "methodName='" + methodName + '\'' + ", humanReadableLabel='"
                    + humanReadableLabel + '\'' + ", declaringTypeFullName='" + declaringTypeFullName + '\''
                    + ", declaringTypeBaseName='" + declaringTypeBaseName + '\'' + ", reactAttributeName='"
                    + reactAttributeName + '\'' + '}';
        }
    }

    public EventCollectionResult getResult() {
        String basePackage = this.getBasePackage();
        String userProjectPackage = basePackage != null ? basePackage : "*";
        final Map<String, Set<ComponentEvent>> javaQualifiedNameEventListMap = new HashMap<>();
        final Map<String, Set<ComponentEvent>> vaadinTagNameEventListMap = new HashMap<>();
        try (ScanResult scan = new ClassGraph().acceptPackages("com.vaadin.flow", userProjectPackage).enableClassInfo()
                .enableMethodInfo().enableAnnotationInfo().scan()) {
            ClassInfoList componentClassInfoList = scan.getSubclasses(Component.class)
                    .filter(filter -> !filter.isInterface() && !filter.isAbstract());
            for (ClassInfo ci : componentClassInfoList) {
                for (MethodInfo mi : ci.getMethodInfo()) {
                    var sig = mi.getTypeSignatureOrTypeDescriptor();
                    var resultType = sig.getResultType();
                    String baseName = resolveErasedTypeName(resultType);
                    if (Registration.class.getName().equals(baseName)) {
                        ComponentEvent event = new ComponentEvent(mi.getName(),
                                SharedUtil.camelCaseToHumanFriendly(mi.getName()), """
                                        e -> {

                                        }
                                        """, mi.getClassInfo().getName(), mi.getClassInfo().getSimpleName());
                        setReactAttribute(event);
                        if (!methodNamesToIgnore.contains(mi.getName())) {
                            javaQualifiedNameEventListMap.computeIfAbsent(ci.getName(),
                                    k -> new TreeSet<>(getComparator()));
                            javaQualifiedNameEventListMap.get(ci.getName()).add(event);
                            putIfValidReactEvent(ci, event, vaadinTagNameEventListMap);
                        }
                    }
                }
            }
        }
        return new EventCollectionResult(javaQualifiedNameEventListMap, vaadinTagNameEventListMap);
    }

    private void setReactAttribute(ComponentEvent event) {
        String reactAttrName = null;
        if (commonJavaMethodToReactAttributeNameMap.containsKey(event.methodName)) {
            reactAttrName = commonJavaMethodToReactAttributeNameMap.get(event.methodName);
        }
        event.reactAttributeName = reactAttrName;
    }

    private void putIfValidReactEvent(ClassInfo ci, ComponentEvent event,
            Map<String, Set<ComponentEvent>> vaadinTagNameEventListMap) {
        if (event.reactAttributeName == null) {
            return;
        }
        if (ci.hasAnnotation(Tag.class)) {
            AnnotationParameterValueList parameterValues = ci.getAnnotationInfo(Tag.class).getParameterValues();
            Object value = parameterValues.getValue("value");
            if (value instanceof String tagStrValue) {
                vaadinTagNameEventListMap.computeIfAbsent(tagStrValue, k -> new TreeSet<>(getComparator())).add(event);
                vaadinTagNameEventListMap.get(tagStrValue).add(event);
            }
        }
    }

    private static Comparator<ComponentEvent> getComparator() {
        return (e1, e2) -> {
            int e1Priority = methodNamePriorityMap.getOrDefault(e1.methodName, DEFAULT_PRIORITY);
            int e2Priority = methodNamePriorityMap.getOrDefault(e2.methodName, DEFAULT_PRIORITY);
            int byPriority = Integer.compare(e1Priority, e2Priority);
            if (byPriority != 0) {
                return byPriority;
            }
            return e1.methodName.compareTo(e2.methodName);
        };
    }

    public static String resolveErasedTypeName(TypeSignature type) {
        if (type instanceof ClassRefTypeSignature cls) {
            return cls.getFullyQualifiedClassName();
        }
        if (type instanceof ArrayTypeSignature arr) {
            return resolveErasedTypeName(arr.getElementTypeSignature()) + "[]".repeat(arr.getNumDimensions());
        }
        return type.toStringWithSimpleNames();
    }

    private String getBasePackage() {
        if (!SpringBridge.isSpringAvailable(vaadinContext)) {
            return null;
        }
        Class<?> applicationClass = SpringBridge.getApplicationClass(vaadinContext);
        return applicationClass.getPackage().getName();
    }
}
