package com.vaadin.copilot.customcomponent;

import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.function.Supplier;

import com.vaadin.flow.component.Component;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Contains the map of detected custom component classes
 */
public final class CustomComponents {
    private static final Map<String, Supplier<CustomComponent>> customComponentsMap = new HashMap<>();
    private static final Logger log = LoggerFactory.getLogger(CustomComponents.class);

    private CustomComponents() {
    }

    /**
     * Returns the custom component information of a component if present. Returns
     * an optional if not registered or not found.
     *
     * @param component
     *            Component to check
     * @return Custom component information or {@link Optional#empty()} if not
     *         found.
     */
    public static Optional<CustomComponent> getCustomComponentInfo(Component component) {
        return getCustomComponentInfo(component.getClass().getName());
    }

    /**
     * Returns the custom component information of a component class name if
     * present. Returns an optional if not registered or not found.
     *
     * @param componentClassName
     *            Component class name to check
     * @return Custom component information or {@link Optional#empty()} if not
     *         found.
     */
    public static Optional<CustomComponent> getCustomComponentInfo(String componentClassName) {
        return Optional.ofNullable(customComponentsMap.get(componentClassName)).map(Supplier::get);
    }

    /**
     * Returns the custom component information of a component class if present.
     * Returns an optional if not registered or not found.
     *
     * @param clazz
     *            Component class to check
     * @return Custom component information or {@link Optional#empty()} if not
     *         found.
     */
    public static Optional<CustomComponent> getCustomComponentInfo(Class<?> clazz) {
        return getCustomComponentInfo(clazz.getName());
    }

    /**
     * Checks if the given component class is a custom component.
     *
     * @param componentClass
     *            Component class to check
     * @return true if custom component, false otherwise.
     */
    public static boolean isCustomComponent(Class<?> componentClass) {
        return isCustomComponent(componentClass.getName());
    }

    /**
     * Checks if the given component class name is a custom component.
     *
     * @param componentClassName
     *            Component class name to check
     * @return true if custom component, false otherwise.
     */
    public static boolean isCustomComponent(String componentClassName) {
        log.trace("Checking if custom component {} is defined", componentClassName);
        log.trace("Custom componentsMap keys: {}", customComponentsMap.keySet());

        try {
            Class<?> queriedClass = Class.forName(componentClassName);
            log.trace("Queried Class: {}, Identity HashCode: {}", queriedClass.getName(),
                    System.identityHashCode(queriedClass));
            boolean isPresent = getCustomComponentInfo(componentClassName).isPresent();
            log.trace("getCustomComponentInfo(componentClassName).isPresent() ----> {}", isPresent);

            return isPresent;
        } catch (ClassNotFoundException e) {
            log.warn("Class {} not found", componentClassName, e);
            return false;
        }
    }

    static void put(Class<?> componentClass, Supplier<CustomComponent> componentSupplier) {
        String className = componentClass.getName();
        log.trace("Registering custom component supplier for {} with identity code {}", className,
                System.identityHashCode(componentClass));
        customComponentsMap.put(className, componentSupplier);
    }
}
