package com.vaadin.copilot;

import java.io.File;
import java.io.IOException;
import java.nio.file.Path;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import com.vaadin.base.devserver.DevToolsInterface;
import com.vaadin.copilot.ide.IdeUtils;
import com.vaadin.copilot.javarewriter.JavaModifier;
import com.vaadin.copilot.theme.AppThemeUtils;
import com.vaadin.copilot.theme.ApplicationTheme;
import com.vaadin.flow.component.dependency.StyleSheet;
import com.vaadin.flow.component.page.AppShellConfigurator;
import com.vaadin.flow.internal.JacksonUtils;
import com.vaadin.flow.theme.Theme;

import com.helger.css.decl.CSSImportRule;
import com.helger.css.decl.CascadingStyleSheet;
import com.helger.css.reader.CSSReader;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.xml.sax.SAXException;
import tools.jackson.databind.JsonNode;
import tools.jackson.databind.node.ObjectNode;

/**
 * Helps to initialize a Spring Boot Vaadin application with no views by adding
 * either a Flow or Hilla view + a theme.
 */
public class ApplicationInitializer extends CopilotCommand {
    public static final String SUCCESS_KEY = "success";
    public static final String REASON_KEY = "reason";
    public static final String INIT_APP_RESULT = "copilot-init-app-result";
    public static final String REFRESH_KEY = "refresh";

    private static final String AURA_FQN = "com.vaadin.flow.theme.aura.Aura";
    private static final String LUMO_FQN = "com.vaadin.flow.theme.lumo.Lumo";

    @Override
    public boolean handleMessage(String command, JsonNode data, DevToolsInterface devToolsInterface) {
        if (command.equals("init-app")) {
            String framework = data.get("framework").asString();
            try {
                initApp(framework, devToolsInterface, data.get(KEY_REQ_ID).asString());
            } catch (IOException e) {
                getLogger().error("Unable to initialize project", e);
            }
            return true;
        } else if (command.equals("create-theme")) {
            ObjectNode responseData = JacksonUtils.createObjectNode();
            responseData.put(KEY_REQ_ID, data.get(KEY_REQ_ID).asString());
            try {
                String appTheme = data.get("theme").asString();
                createTheme(ApplicationTheme.fromString(appTheme));
                devToolsInterface.send(command + "-resp", responseData);
            } catch (IOException e) {
                ErrorHandler.sendErrorResponse(devToolsInterface, command + "-resp", responseData,
                        "Error creating theme", e);
            }
            return true;
        }
        return false;
    }

    private synchronized void createTheme(ApplicationTheme theme) throws IOException {

        if (!SpringBridge.isSpringAvailable(getVaadinContext())) {
            throw new IOException("This only works for Spring Boot projects");
        }
        Map<File, String> toWrite = new HashMap<>();

        Class<?> applicationClass = SpringBridge.getApplicationClass(getVaadinContext());
        File applicationClassFile = ProjectFileManager.get().getFileForClass(applicationClass);
        KotlinUtil.throwIfKotlin(applicationClassFile);
        applyStyles(applicationClass, toWrite, theme);
        for (Map.Entry<File, String> entry : toWrite.entrySet()) {
            getProjectFileManager().writeFile(entry.getKey(), "Create theme", entry.getValue());
        }
    }

    private synchronized void initApp(String framework, DevToolsInterface devToolsInterface, String reqId)
            throws IOException {
        ObjectNode responseData = JacksonUtils.createObjectNode();
        responseData.put(KEY_REQ_ID, reqId);

        if (!SpringBridge.isSpringAvailable(getVaadinContext())) {
            responseData.put(SUCCESS_KEY, false);
            responseData.put(REASON_KEY, "This only works for Spring Boot projects");
            devToolsInterface.send(INIT_APP_RESULT, responseData);
            return;
        }
        Map<File, String> toWrite = new HashMap<>();

        Class<?> applicationClass = SpringBridge.getApplicationClass(getVaadinContext());
        applyStyles(applicationClass, toWrite, ApplicationTheme.AURA);
        File toOpen;
        if (framework.equals("flow")) {
            Map<? extends File, String> flowView = addFlowView(applicationClass);
            toWrite.putAll(flowView);
            toOpen = flowView.keySet().iterator().next();
        } else if (framework.equals("hilla")) {
            Map<File, String> hillaView = addHillaView();
            toWrite.putAll(hillaView);
            toOpen = hillaView.keySet().iterator().next();
            addHillaDependency(toOpen);
        } else {
            responseData.put(SUCCESS_KEY, false);
            responseData.put(REASON_KEY, "Unknown framework " + framework);
            devToolsInterface.send(INIT_APP_RESULT, responseData);
            return;
        }

        for (Map.Entry<File, String> entry : toWrite.entrySet()) {
            getProjectFileManager().writeFile(entry.getKey(), "Project generation", entry.getValue());
        }

        if (toOpen != null) {
            try {
                Thread.sleep(1000);
                IdeUtils.openFile(toOpen, 1);
            } catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
        }
        responseData.put(SUCCESS_KEY, true);
        responseData.put(REFRESH_KEY, true);
        devToolsInterface.send(INIT_APP_RESULT, responseData);
    }

    private void applyStyles(Class<?> applicationClass, Map<File, String> toWrite, ApplicationTheme theme)
            throws IOException {
        // add import only when no theme or existing styles.css import
        Theme themeAnnotation = applicationClass.getAnnotation(Theme.class);
        if (themeAnnotation != null) {
            return;
        }

        toWrite.putAll(new JavaModifier().modify(applicationClass, operations -> {

            // Check if styles.css already contains aura.css or lumo.css imports
            boolean hasThemeCssImport = false;
            File stylesCss = getProjectFileManager().getStylesCss();
            if (stylesCss.exists()) {
                CascadingStyleSheet styleSheet = CSSReader.readFromFile(stylesCss);
                if (styleSheet != null && styleSheet.hasImportRules()) {
                    hasThemeCssImport = styleSheet.getAllImportRules().stream().map(CSSImportRule::getLocationString)
                            .anyMatch(location -> location.endsWith("/aura.css") || location.endsWith("/lumo.css"));
                }
            } else {
                toWrite.put(stylesCss, "");
            }

            if (hasThemeCssImport) {
                getLogger().info(
                        "styles.css already contains import of Aura or Lumo theme. User selection is not applied.");
            } else if (AppThemeUtils.isNone()) {
                if (ApplicationTheme.AURA.equals(theme)) {
                    operations.addClassAnnotationWithConstant(StyleSheet.class, AURA_FQN, "STYLESHEET");
                } else if (ApplicationTheme.LUMO.equals(theme)) {
                    operations.addClassAnnotationWithConstant(StyleSheet.class, LUMO_FQN, "STYLESHEET");
                    operations.addClassAnnotationWithConstant(StyleSheet.class, LUMO_FQN, "UTILITY_STYLESHEET");
                }
            }

            // Check for `@StyleSheet("styles.css")
            StyleSheet[] styleSheets = applicationClass.getAnnotationsByType(StyleSheet.class);
            if (Arrays.stream(styleSheets).map(StyleSheet::value).noneMatch(stylesCss.getName()::equals)) {
                operations.addClassAnnotation(StyleSheet.class, stylesCss.getName());
            }

            // Add required AppShellConfigurator
            operations.addInterface(AppShellConfigurator.class);
        }));

    }

    private Map<File, String> addHillaView() {
        File frontendFolder = getProjectFileManager().getFrontendFolder();
        File viewFile = new File(new File(frontendFolder, "views"), "@index.tsx");
        String viewTemplate = getViewTemplate(viewFile);

        return Collections.singletonMap(viewFile, viewTemplate);
    }

    private Map<? extends File, String> addFlowView(Class<?> applicationClass) {
        File applicationFile = getProjectFileManager().getFileForClass(applicationClass);
        String basePackage = applicationClass.getPackage().getName();
        File viewFolder = new File(applicationFile.getParentFile(), "views");
        File viewFile = new File(viewFolder, "HomeView.java");
        String relativeViewName = getProjectFileManager().getProjectRelativeName(viewFile);
        String viewPackage = basePackage + ".views";

        String content = String.format("""
                package %s;

                import com.vaadin.flow.component.html.H1;
                import com.vaadin.flow.component.html.Paragraph;
                import com.vaadin.flow.component.orderedlayout.VerticalLayout;
                import com.vaadin.flow.router.Route;

                @Route("")
                public class HomeView extends VerticalLayout {

                    public HomeView() {

                        add(new H1("Welcome to your new application"));
                        add(new Paragraph("This is the home view"));

                        add(new Paragraph("You can edit this view in %s"));

                    }
                }
                """, viewPackage, relativeViewName.replace("\\", "\\\\"));

        return Collections.singletonMap(viewFile, content);
    }

    private String getViewTemplate(File viewFile) {
        String relativeViewName = getProjectFileManager().getProjectRelativeName(viewFile);
        return String.format("""
                export default function HomeView() {
                  return (
                    <div>
                      <h1>Welcome to your new application</h1>
                      <p>This is the home view.</p>
                      <p>
                        You can edit this view in <code>%s</code> or by
                        activating Copilot by clicking the icon in the lower right corner
                      </p>
                    </div>
                  );
                }

                """, relativeViewName);
    }

    private void addHillaDependency(File viewFile) throws IOException {
        JavaSourcePathDetector.ModuleInfo module = ProjectFileManager.get().findModule(viewFile)
                .orElseThrow(() -> new IllegalArgumentException("No module found for file: " + viewFile));
        Path pomXml = module.rootPath().resolve("pom.xml");
        if (pomXml.toFile().exists()) {
            try {
                PomFileRewriter pomFileRewriter = new PomFileRewriter(pomXml);
                pomFileRewriter.addDependency("com.vaadin", "hilla-spring-boot-starter", null);
                pomFileRewriter.save();
            } catch (SAXException e) {
                throw new IOException(e);
            }
            return;
        }
        Path gradleFile = module.rootPath().resolve("build.gradle");
        if (gradleFile.toFile().exists()) {
            GradleFileRewriter gradleFileRewriter = new GradleFileRewriter(gradleFile);
            gradleFileRewriter.addDependency("com.vaadin", "hilla-spring-boot-starter", null);
            gradleFileRewriter.save();
        }
    }

    private static Logger getLogger() {
        return LoggerFactory.getLogger(ApplicationInitializer.class);
    }
}
