package com.vaadin.copilot.customcomponent;

import java.io.File;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import com.vaadin.copilot.KotlinUtil;
import com.vaadin.copilot.ProjectFileManager;
import com.vaadin.copilot.RouteHandler;
import com.vaadin.flow.component.Component;
import com.vaadin.flow.router.RouteData;
import com.vaadin.flow.server.RouteRegistry;
import com.vaadin.flow.server.VaadinSession;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Provides information about component is whether a custom component or not.
 * Once custom component is detected, it is stored in {@link CustomComponents}
 * where it can be accessed statically from anywhere
 */
public final class CustomComponentDetector {
    private CustomComponentDetector() {

    }

    /**
     * Detects if given class is a custom component class. If it is a custom
     * component, it is saved and can be accessible in {@link CustomComponents}
     *
     * @param component
     *            Component to put into the customComponentsMap
     * @param vaadinSession
     *            the VaadinSession
     */
    public static void detectAndPutIfPresent(Component component, VaadinSession vaadinSession) {
        detectAndPutIfPresent(vaadinSession, component.getClass());
    }

    /**
     * Detects if given class is a custom component class. If it is a custom
     * component, it is saved and can be accessible in {@link CustomComponents}
     *
     * @param vaadinSession
     *            the session to get the classes from
     * @param componentClass
     *            Component class
     */
    public static void detectAndPutIfPresent(VaadinSession vaadinSession, Class<?> componentClass) {
        getLogger().trace("Detecting class: {} and identity hashcode: {}", componentClass.getName(),
                System.identityHashCode(componentClass));
        if (CustomComponents.isCustomComponent(componentClass)) {
            return;
        }
        File fileForClass = ProjectFileManager.get().getFileForClass(componentClass);
        if (!fileForClass.exists()) {
            if (isExternalCustomComponent(componentClass)) {
                CustomComponents.put(componentClass, () -> new CustomComponentExternal(componentClass));
            }
            return;
        }
        // do not consider kotlin files as custom components as there is no parser for
        // kotlin to detect route and layout etc...
        if (KotlinUtil.isKotlin(fileForClass)) {
            return;
        }
        Set<Class<?>> routeLayoutClasses = getLayoutClasses(vaadinSession);
        if (routeLayoutClasses.contains(componentClass)) {
            return;
        }
        CustomComponents.put(componentClass, () -> new CustomComponentInProject(componentClass, fileForClass));
    }

    /**
     * Returns the distinct classes that are used for route creation in the given
     * session.
     *
     * @param vaadinSession
     *            the session to get the classes from
     * @return Set of classes which are used in {@link RouteRegistry}
     */
    private static Set<Class<?>> getLayoutClasses(VaadinSession vaadinSession) {
        Set<Class<?>> routeLayoutClasses = new HashSet<>();

        List<RouteData> serverRoutes = RouteHandler.getServerRoutes(vaadinSession);
        if (serverRoutes == null) {
            return routeLayoutClasses;
        }
        routeLayoutClasses.addAll(RouteHandler.getLayoutClasses(vaadinSession, serverRoutes));
        return routeLayoutClasses;
    }

    private static boolean isExternalCustomComponent(Class<?> componentClass) {
        return !componentClass.getPackageName().startsWith("com.vaadin");
    }

    private static Logger getLogger() {
        return LoggerFactory.getLogger(CustomComponentDetector.class);
    }
}
