package com.vaadin.copilot.plugins.docs;

import java.util.HashMap;
import java.util.Map;
import java.util.stream.Collectors;

import com.vaadin.base.devserver.DevToolsInterface;
import com.vaadin.base.devserver.ServerInfo;
import com.vaadin.copilot.CopilotCommand;
import com.vaadin.copilot.CopilotServerClient;
import com.vaadin.flow.internal.JacksonUtils;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import tools.jackson.databind.JsonNode;

public class DocsHandler extends CopilotCommand {

    private final DocsDataLoader docsDataLoader;
    private final CopilotServerClient copilotServerClient = new CopilotServerClient();
    private final Map<String, String> serverInfoVersions;

    public record AskDocsRequest(AskDocsData data, CopilotServerClient.AccessControlData accessControlData) {

    }

    public record AskDocsData(String question, String framework, Map<String, String> metadata) {

    }

    public record AskDocsResponse(String answer, String error) {
    }

    public DocsHandler() {
        this.docsDataLoader = new DocsDataLoader();
        ServerInfo serverInfo = new ServerInfo();
        this.serverInfoVersions = serverInfo.getVersions().stream()
                .collect(Collectors.toMap(ServerInfo.NameAndVersion::name, ServerInfo.NameAndVersion::version));
    }

    @Override
    public boolean handleMessage(String command, JsonNode data, DevToolsInterface devToolsInterface) {
        if (command.equals("get-docs")) {
            var reqId = data.get(KEY_REQ_ID).asString();
            var responseData = JacksonUtils.createObjectNode();
            responseData.put(KEY_REQ_ID, reqId);
            try {
                DocsRequest docsRequest = JacksonUtils.readValue(data, DocsRequest.class);
                DocsResponse response = docsDataLoader.get(docsRequest);
                responseData.put("content", JacksonUtils.getMapper().writeValueAsString(response));
            } catch (DocsException exception) {
                responseData.put("errorCode", exception.getCode());
            } catch (Exception ex) {
                responseData.put("errorCode", DocsException.UNKNOWN_ERROR);
                getLogger().error(ex.getMessage(), ex);
            }
            devToolsInterface.send("copilot-get-docs-response", responseData);

            return true;
        } else if (command.equals("ask-docs")) {
            var reqId = data.get(KEY_REQ_ID).asString();
            var responseData = JacksonUtils.createObjectNode();
            responseData.put(KEY_REQ_ID, reqId);
            AskDocsData askData = new AskDocsData(data.get("question").asString(), data.get("framework").asString(),
                    new HashMap<>(serverInfoVersions));
            copilotServerClient.sendCopilotRequest("ask-docs",
                    new AskDocsRequest(askData, CopilotServerClient.AccessControlData.create()), AskDocsResponse.class,
                    response -> {
                        responseData.put("answer", response.answer() != null ? response.answer() : "");
                        responseData.put("error", response.error() != null ? response.error() : "");
                        devToolsInterface.send(command + "-response", responseData);
                    }, devToolsInterface, command, responseData);
            return true;
        }
        return false;
    }

    private static Logger getLogger() {
        return LoggerFactory.getLogger(DocsHandler.class);
    }

    @Override
    public boolean canBeParallelCommand(String command) {
        return command.equalsIgnoreCase("get-docs");
    }
}
