package com.vaadin.copilot.userinfo;

import java.io.IOException;
import java.net.URI;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import com.vaadin.copilot.AIUsageAllowed;
import com.vaadin.copilot.CopilotJacksonUtils;
import com.vaadin.copilot.CopilotServerClient;
import com.vaadin.copilot.MachineConfiguration;
import com.vaadin.pro.licensechecker.LocalOfflineKey;
import com.vaadin.pro.licensechecker.LocalProKey;
import com.vaadin.pro.licensechecker.OfflineKey;
import com.vaadin.pro.licensechecker.ProKey;

public class UserInfoServerClient extends CopilotServerClient {

    static Map<String, UserInfo> userInfoCache = Collections.synchronizedMap(new HashMap<>());

    private UserInfo fetchUserInfo(String proKey, String offlineKey) {
        URI uri = getQueryURI("userInfo");
        UserInfoRequest requestData = new UserInfoRequest(proKey, offlineKey);
        HttpRequest request = buildRequest(uri, requestData);
        try {
            HttpResponse<String> response = getHttpClient().send(request, HttpResponse.BodyHandlers.ofString());
            if (response.statusCode() != 200) {
                return null;
            }
            return CopilotJacksonUtils.readValue(response.body(), UserInfo.class);
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        } catch (IOException e) {
            // no logging here as we don't want to bother developer that we
            // can't load his user info
        }

        return null;
    }

    /**
     * Requests to Copilot Server to get user information with local pro key if
     * present
     *
     * @return Returns null when pro key not exist or when request is not succeeded.
     */
    public static UserInfo getUserInfoWithLocalProKey() {
        OfflineKey offlineKey = LocalOfflineKey.get();
        ProKey proKey = LocalProKey.get();
        if (proKey == null && offlineKey == null) {
            return null;
        }
        if (proKey == null || proKey.getProKey() == null) {
            UserInfo cachedUserInfo = userInfoCache.get(offlineKey.getJwtData());
            if (cachedUserInfo == null) {
                cachedUserInfo = new UserInfoServerClient().fetchUserInfo(null, offlineKey.getJwtData());
                userInfoCache.put(offlineKey.getJwtData(), cachedUserInfo);
            }
            return cachedUserInfo;
        }
        UserInfo cachedUserInfo = userInfoCache.get(proKey.getProKey());
        if (cachedUserInfo == null && proKey.getProKey() != null) {
            cachedUserInfo = new UserInfoServerClient().fetchUserInfo(proKey.getProKey(), null);
            userInfoCache.put(proKey.getProKey(), cachedUserInfo);
        }

        return cachedUserInfo;
    }

    /**
     * Checks whether AI usage is allowed based on various runtime conditions, and
     * throws an {@link AIUsageDisabledException} if usage is not permitted.
     * <p>
     * This method validates the following conditions in order:
     * <ul>
     * <li>If the {@link UserInfo} indicates that the project must not leave
     * localhost (via {@code userInfo.copilotProjectCannotLeaveLocalhost()}), usage
     * is not allowed.</li>
     * <li>If the machine configuration explicitly disallows AI usage (via
     * {@code MachineConfiguration.get().isAIUsageAllowed()} returning
     * {@code AIUsageAllowed.NO}), usage is not allowed.</li>
     * </ul>
     * <p>
     * If none of the above conditions block access, then we assume AI usage is
     * allowed. In the case of {@code AIUsageAllowed.ASK}, it is assumed the user
     * has confirmed usage via a client-side popup.
     *
     * @throws AIUsageDisabledException
     *             if any of the above conditions disallow AI usage
     */
    public static void throwIfAIUsageDisabled() {
        UserInfo userInfo = getUserInfoWithLocalProKey();

        if (userInfo == null || userInfo.copilotProjectCannotLeaveLocalhost()) {
            throw new AIUsageDisabledException();
        }

        AIUsageAllowed aiUsageAllowed = MachineConfiguration.get().isAIUsageAllowed();
        if (aiUsageAllowed == AIUsageAllowed.NO) {
            throw new AIUsageDisabledException();
        }

        // If the setting is 'yes', then it is fine to execute. If the settings is
        // 'ask', we have to assume that there was a popup in the browser, and the user
        // confirmed to usage
    }

}
