package com.vaadin.copilot.testbenchgenerator;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import com.vaadin.copilot.ProjectFileManager;
import com.vaadin.copilot.Util;
import com.vaadin.copilot.exception.CopilotException;
import com.vaadin.copilot.ide.CopilotIDEPlugin;
import com.vaadin.copilot.javarewriter.JavaRewriterUtil;
import com.vaadin.copilot.testbenchgenerator.assertion.AssertionScenarioItem;
import com.vaadin.copilot.testbenchgenerator.assertion.AssertionType;
import com.vaadin.copilot.testbenchgenerator.assertion.PageTitleEqualsAssertion;
import com.vaadin.copilot.testbenchgenerator.assertion.TargetAssertionScenarioItem;
import com.vaadin.copilot.testbenchgenerator.assertion.TextEqualsAssertionItem;
import com.vaadin.copilot.testbenchgenerator.events.PageChangeEvent;
import com.vaadin.copilot.testbenchgenerator.events.TargetEvent;
import com.vaadin.copilot.testbenchgenerator.events.TargetEventData;
import com.vaadin.copilot.testbenchgenerator.events.TargetEventName;

import com.github.javaparser.Range;
import com.github.javaparser.StaticJavaParser;
import com.github.javaparser.ast.CompilationUnit;
import com.github.javaparser.ast.body.ClassOrInterfaceDeclaration;
import com.github.javaparser.ast.body.MethodDeclaration;
import com.github.javaparser.ast.body.VariableDeclarator;
import com.github.javaparser.ast.expr.Expression;
import com.github.javaparser.ast.expr.MethodCallExpr;
import com.github.javaparser.ast.expr.NameExpr;
import com.github.javaparser.ast.expr.StringLiteralExpr;
import com.github.javaparser.ast.expr.ThisExpr;
import com.github.javaparser.ast.expr.VariableDeclarationExpr;
import com.github.javaparser.ast.stmt.BlockStmt;
import com.github.javaparser.ast.stmt.ExpressionStmt;
import com.github.javaparser.ast.type.VarType;
import org.apache.commons.lang3.StringUtils;

/**
 * Generates a TestBench tests using the JavaParser with the given scenario.
 */
public class TestBenchJavaTestGenerator {
    private static final String UNDO_LABEL = CopilotIDEPlugin.undoLabel("Test Bench Java Test");
    private static final String JAVA_FILE_EXTENSION = "java";
    private static final String JUPITER_API_ASSERTIONS_IMPORT = "org.junit.jupiter.api.Assertions";
    private final TestBenchScenario scenario;
    private String testContainingClassName;
    private final String testMethodName;
    private CompilationUnit compilationUnit;

    /**
     * Constructs a TestBenchScenario
     *
     * @param scenario
     *            to create
     */
    public TestBenchJavaTestGenerator(TestBenchScenario scenario) {
        this.scenario = scenario;
        this.testMethodName = scenario.getMethodName();
    }

    /**
     * Generates the test case for the given scenario.
     *
     * @return Generated absolute file path
     * @throws IOException
     *             is thrown when IO exception happens.
     */
    public TestBenchGenerateResult generate() throws IOException {
        Path testFilePath = getTestFilePath(this.scenario.getQualifiedClassName());
        if (!testFilePath.toFile().exists()) {
            Files.createDirectories(testFilePath.getParent());
        }
        if (StringUtils.isEmpty(this.testMethodName)) {
            throw new IllegalArgumentException("Test method name cannot be empty");
        }
        parseOrCreateCompilationUnit(testFilePath.toFile(), this.scenario.getQualifiedClassName());
        addImports();
        ClassOrInterfaceDeclaration classOrInterfaceDeclaration = compilationUnit
                .getClassByName(testContainingClassName)
                .orElseThrow(() -> new IllegalStateException("Unable to find class " + testContainingClassName));
        boolean sameMethodExists = classOrInterfaceDeclaration.getMethods().stream()
                .anyMatch(f -> f.getNameAsString().equals(testMethodName));
        if (sameMethodExists) {
            throw new CopilotException("Method '" + testMethodName + "' already exists");
        }
        MethodDeclaration methodDeclaration = classOrInterfaceDeclaration.addMethod(this.testMethodName);
        methodDeclaration.addAnnotation("BrowserTest");
        this.generateMethodBody(methodDeclaration);
        File file = testFilePath.toFile();
        ProjectFileManager.get().writeFile(file, UNDO_LABEL, compilationUnit.toString());
        String resultAbsolutePath = file.getAbsolutePath();
        int lineNumber = 1;
        if (methodDeclaration.getRange().isPresent()) {
            Range range = methodDeclaration.getRange().get();
            lineNumber = range.begin.line;
        }

        return new TestBenchGenerateResult(resultAbsolutePath, Util.getFileName(resultAbsolutePath), lineNumber);
    }

    private void generateMethodBody(MethodDeclaration methodDeclaration) {
        BlockStmt body = new BlockStmt();
        methodDeclaration.setBody(body);
        for (TestBenchScenarioItem scenarioItem : this.scenario.getItems()) {
            if (scenarioItem.getType() == TestBenchScenarioItemType.pageChangeEvent) {
                body.addStatement(this.getPageChangeEventStatement((PageChangeEvent) scenarioItem));
            } else if (scenarioItem.getType() == TestBenchScenarioItemType.targetEvent) {
                List<ExpressionStmt> targetEventStatements = getTargetEventStatements((TargetEvent) scenarioItem, body);
                targetEventStatements.forEach(body::addStatement);
            } else if (scenarioItem.getType() == TestBenchScenarioItemType.assertion) {
                List<ExpressionStmt> assertExpressions = getAssertionStatements(scenarioItem, body);
                assertExpressions.forEach(body::addStatement);
            }
        }
    }

    private List<ExpressionStmt> getAssertionStatements(TestBenchScenarioItem scenario, BlockStmt body) {
        List<ExpressionStmt> statements = new ArrayList<>();
        if (scenario instanceof AssertionScenarioItem assertionScenarioItem) {
            if (assertionScenarioItem.getAssertionType() == AssertionType.noJsError) {
                String assertionUtilityMethod = "assertNoJsError";
                ClassOrInterfaceDeclaration classOrInterfaceDeclaration = compilationUnit
                        .getClassByName(testContainingClassName).orElseThrow(
                                () -> new IllegalStateException("Unable to find class " + testContainingClassName));
                List<MethodDeclaration> assertNoJsError = classOrInterfaceDeclaration
                        .getMethodsByName(assertionUtilityMethod);
                if (assertNoJsError.isEmpty()) {
                    MethodDeclaration methodDeclaration = classOrInterfaceDeclaration.addMethod(assertionUtilityMethod);
                    methodDeclaration.setBody(StaticJavaParser.parseBlock("""
                            {
                                LogEntries logs = getDriver().manage().logs().get(LogType.BROWSER);
                                for (LogEntry logEntry : logs) {
                                    if(logEntry.getLevel().equals(Level.SEVERE)) {
                                        Assertions.fail("JS Error: " + logEntry.getMessage());
                                    }
                                }
                            }
                            """));
                    compilationUnit.addImport("java.util.logging.Level");
                    compilationUnit.addImport(JUPITER_API_ASSERTIONS_IMPORT);
                    compilationUnit.addImport("org.openqa.selenium.logging.LogEntries");
                    compilationUnit.addImport("org.openqa.selenium.logging.LogEntry");
                    compilationUnit.addImport("org.openqa.selenium.logging.LogType");
                }
                MethodCallExpr assertionMethodCall = new MethodCallExpr(new ThisExpr(), "assertNoJsError");
                body.addStatement(assertionMethodCall);
            } else if (assertionScenarioItem.getAssertionType() == AssertionType.pageTitleEquals
                    && assertionScenarioItem instanceof PageTitleEqualsAssertion pageTitleEqualsAssertion) {
                String expectedTitle = pageTitleEqualsAssertion.getExpectedTitle();
                compilationUnit.addImport(JUPITER_API_ASSERTIONS_IMPORT);
                MethodCallExpr getDriverCall = new MethodCallExpr("getDriver");
                MethodCallExpr getTitleCall = new MethodCallExpr("getTitle");
                getTitleCall.setScope(getDriverCall);

                NameExpr assertionClassName = new NameExpr("Assertions");
                MethodCallExpr assertEqualsCall = new MethodCallExpr(assertionClassName, "assertEquals");
                assertEqualsCall.addArgument(new StringLiteralExpr(expectedTitle));
                assertEqualsCall.addArgument(getTitleCall);
                statements.add(new ExpressionStmt(assertEqualsCall));
            }
        } else if (scenario instanceof TargetAssertionScenarioItem targetAssertionScenarioItem) {
            if (targetAssertionScenarioItem.getAssertionType() == AssertionType.presence) {
                // waitForFirst is used
                addIfNeededAndGetVariableDeclarator(targetAssertionScenarioItem.getPossibleSelectors(),
                        targetAssertionScenarioItem.getClassInfo(), body,
                        targetAssertionScenarioItem.getTargetElementIdentifier(), statements, true);
            } else if (targetAssertionScenarioItem.getAssertionType() == AssertionType.textEquals) {
                VariableDeclarator variableDeclarator = addIfNeededAndGetVariableDeclarator(
                        targetAssertionScenarioItem.getPossibleSelectors(), targetAssertionScenarioItem.getClassInfo(),
                        body, targetAssertionScenarioItem.getTargetElementIdentifier(), statements, false);
                if (variableDeclarator == null) {
                    return statements;
                }
                compilationUnit.addImport(JUPITER_API_ASSERTIONS_IMPORT);
                TextEqualsAssertionItem textEqualsAssertionItem = (TextEqualsAssertionItem) targetAssertionScenarioItem;
                NameExpr assertionClassName = new NameExpr("Assertions");
                MethodCallExpr textEqualsMethodCallExpr = new MethodCallExpr();
                textEqualsMethodCallExpr.setScope(assertionClassName);
                textEqualsMethodCallExpr.setName("assertEquals");
                textEqualsMethodCallExpr.addArgument(new StringLiteralExpr(textEqualsAssertionItem.getExpectedText()));
                MethodCallExpr getTextMethodCallExpr = new MethodCallExpr();
                getTextMethodCallExpr.setScope(variableDeclarator.getNameAsExpression());
                getTextMethodCallExpr.setName("getText");
                textEqualsMethodCallExpr.addArgument(getTextMethodCallExpr);
                statements.add(new ExpressionStmt(textEqualsMethodCallExpr));
            }
        }
        return statements;
    }

    private VariableDeclarator addIfNeededAndGetVariableDeclarator(PossibleSelectors possibleSelectors,
            TestBenchElementClassInfo classInfo, BlockStmt body, String identifierToGenerateVarName,
            List<ExpressionStmt> statements, boolean waitForFirst) {
        VariableDeclarator variableDeclaration = getVariableDeclaration(possibleSelectors, classInfo, body,
                identifierToGenerateVarName, waitForFirst);
        if (variableDeclaration == null) {
            return null;
        }
        PossibleSelectorsWithClassInfo possibleSelectorsWithClassInfo = new PossibleSelectorsWithClassInfo(
                possibleSelectors, classInfo);
        if (!addedSelectors.contains(possibleSelectorsWithClassInfo)) {
            VariableDeclarationExpr variableDeclarationExpr = new VariableDeclarationExpr(
                    selectorsDeclaratorsMap.get(possibleSelectorsWithClassInfo));
            statements.add(new ExpressionStmt(variableDeclarationExpr));
            addedSelectors.add(possibleSelectorsWithClassInfo);
        }
        return variableDeclaration;
    }

    private List<ExpressionStmt> getTargetEventStatements(TargetEvent scenario, BlockStmt body) {
        List<TargetEventData> events = scenario.getEvents();
        List<ExpressionStmt> stmts = new ArrayList<>();
        for (TargetEventData event : events) {
            PossibleSelectors possibleSelectors = scenario.getPossibleSelectors();
            TestBenchElementClassInfo classInfo = scenario.getClassInfo();
            VariableDeclarator variableDeclaration = addIfNeededAndGetVariableDeclarator(possibleSelectors, classInfo,
                    body, scenario.getTargetElementIdentifier(), stmts, false);
            if (variableDeclaration == null) {
                continue;
            }
            String variableName = variableDeclaration.getNameAsString();
            if (event.getName() == TargetEventName.setValue && event.getValue() != null) {
                MethodCallExpr setValueExpr = new MethodCallExpr();
                setValueExpr.setScope(new NameExpr(variableName));
                setValueExpr.setName("setValue");
                setValueExpr.addArgument(new StringLiteralExpr(event.getValue()));
                stmts.add(new ExpressionStmt(setValueExpr));
            } else if (event.getName() == TargetEventName.click) {
                MethodCallExpr clickExp = new MethodCallExpr(variableName);
                clickExp.setScope(new NameExpr(variableName));
                clickExp.setName("click");
                stmts.add(new ExpressionStmt(clickExp));
            } else if (event.getName() == TargetEventName.contextmenu) {
                MethodCallExpr contextClickExp = new MethodCallExpr(variableName);
                contextClickExp.setScope(new NameExpr(variableName));
                contextClickExp.setName("contextClick");
                stmts.add(new ExpressionStmt(contextClickExp));
            } else if (event.getName() == TargetEventName.itemSelect) {
                MethodCallExpr itemSelectExp = new MethodCallExpr(variableName);
                itemSelectExp.setScope(new NameExpr(variableName));
                itemSelectExp.setName("selectByText");
                itemSelectExp.addArgument(new StringLiteralExpr(event.getSelectedItemText()));
                stmts.add(new ExpressionStmt(itemSelectExp));
            } else if (event.getName() == TargetEventName.deSelect) {
                MethodCallExpr deSelectExpr = new MethodCallExpr(variableName);
                deSelectExpr.setScope(new NameExpr(variableName));
                deSelectExpr.setName("deselectByText");
                deSelectExpr.addArgument(new StringLiteralExpr(event.getSelectedItemText()));
                stmts.add(new ExpressionStmt(deSelectExpr));
            }
        }

        return stmts;
    }

    private final Set<PossibleSelectorsWithClassInfo> addedSelectors = new HashSet<>();
    private final Map<PossibleSelectorsWithClassInfo, VariableDeclarator> selectorsDeclaratorsMap = new HashMap<>();

    private VariableDeclarator getVariableDeclaration(PossibleSelectors possibleSelectors,
            TestBenchElementClassInfo classInfo, BlockStmt methodBody, String identifierToGenerateVarName,
            boolean waitForFirst) {
        if (selectorsDeclaratorsMap.containsKey(new PossibleSelectorsWithClassInfo(possibleSelectors, classInfo))) {
            return selectorsDeclaratorsMap.get(new PossibleSelectorsWithClassInfo(possibleSelectors, classInfo));
        }
        if (classInfo == null) {
            // TODO body and other native elements that are not registered in the scenario
            // plugin have null values
            return null;
        }
        String firstExpressionCall = waitForFirst ? "waitForFirst" : "first";
        VariableDeclarator declarator = new VariableDeclarator();
        declarator.setType(new VarType());

        String selectedIdentifier;
        if (identifierToGenerateVarName != null) {
            selectedIdentifier = identifierToGenerateVarName;
        } else {
            selectedIdentifier = classInfo.baseName();
        }
        selectedIdentifier = JavaRewriterUtil.getJavaIdentifier(selectedIdentifier, 100);
        String freeVariableName = JavaRewriterUtil.findFreeVariableName(selectedIdentifier, methodBody);
        VariableDeclarator variableDeclarator = declarator.setName(freeVariableName);
        MethodCallExpr testBenchQuerySelectorCall = new MethodCallExpr();
        testBenchQuerySelectorCall.setName("$");

        String componentSimpleClassName = classInfo.qualifiedClassName()
                .substring(classInfo.qualifiedClassName().lastIndexOf('.') + 1);

        testBenchQuerySelectorCall.addArgument(new NameExpr(componentSimpleClassName + ".class"));

        MethodCallExpr firstExpression = null;
        if (possibleSelectors.selectedAttributePairByUser() != null) {
            PossibleSelectors.SelectedAttributePairByUser attrPair = possibleSelectors.selectedAttributePairByUser();
            MethodCallExpr withAttrExpression;
            if ("containing".equals(possibleSelectors.selectedAttributePairByUser().comparisonType())) {
                withAttrExpression = new MethodCallExpr("withAttributeContaining");
            } else {
                withAttrExpression = new MethodCallExpr("withAttribute");
            }
            withAttrExpression.setScope(testBenchQuerySelectorCall);
            withAttrExpression.addArgument(new StringLiteralExpr(attrPair.attrName()));
            withAttrExpression.addArgument(new StringLiteralExpr(attrPair.attrValue()));
            firstExpression = new MethodCallExpr(withAttrExpression, firstExpressionCall);
        } else if (possibleSelectors.label() != null) {
            MethodCallExpr withLabelExpression = new MethodCallExpr(testBenchQuerySelectorCall, "withLabel");
            withLabelExpression.addArgument(new StringLiteralExpr(possibleSelectors.label()));
            firstExpression = new MethodCallExpr(withLabelExpression, firstExpressionCall);
        } else if (possibleSelectors.text() != null) {
            MethodCallExpr withTextExpression = new MethodCallExpr(testBenchQuerySelectorCall, "withText");
            withTextExpression.addArgument(new StringLiteralExpr(possibleSelectors.text()));
            firstExpression = new MethodCallExpr(withTextExpression, firstExpressionCall);
        } else if (possibleSelectors.id() != null) {
            MethodCallExpr withIdExpression = new MethodCallExpr(testBenchQuerySelectorCall, "withId");
            withIdExpression.addArgument(new StringLiteralExpr(possibleSelectors.id()));
            firstExpression = new MethodCallExpr(withIdExpression, firstExpressionCall);
        } else if (possibleSelectors.dataTestId() != null) {
            MethodCallExpr withAttrExpression = new MethodCallExpr(testBenchQuerySelectorCall, "withAttribute");
            withAttrExpression.addArgument(new StringLiteralExpr("data-test-id"));
            withAttrExpression.addArgument(new StringLiteralExpr(possibleSelectors.dataTestId()));
            firstExpression = new MethodCallExpr(withAttrExpression, firstExpressionCall);
        } else if (Boolean.TRUE.equals(possibleSelectors.firstElementInDocument())) {
            firstExpression = new MethodCallExpr(testBenchQuerySelectorCall, firstExpressionCall);
        } else if (possibleSelectors.className() != null) {
            MethodCallExpr withClassNameExpression = new MethodCallExpr(testBenchQuerySelectorCall, "withClassName");
            withClassNameExpression.addArgument(new StringLiteralExpr(possibleSelectors.className()));
            firstExpression = new MethodCallExpr(withClassNameExpression, firstExpressionCall);
        } else if (possibleSelectors.cssPath() != null) {
            String expressionStmt = String.format(
                    "TestBench.wrap((TestBenchElement) getDriver().findElement(By.cssSelector(\"%s\")), %s.class)",
                    possibleSelectors.cssPath(), classInfo.baseName());
            compilationUnit.addImport("com.vaadin.testbench.TestBench");
            compilationUnit.addImport("com.vaadin.testbench.TestBenchElement");
            compilationUnit.addImport("org.openqa.selenium.By");
            Expression statement = StaticJavaParser.parseExpression(expressionStmt);
            variableDeclarator.setInitializer(statement);
        } else {
            throw new CopilotException("Unable to find selector");
        }
        if (firstExpression != null) {
            if (firstExpression.getScope().isEmpty()) {
                firstExpression.setScope(testBenchQuerySelectorCall);
            }

            variableDeclarator.setInitializer(firstExpression);
        }

        selectorsDeclaratorsMap.put(new PossibleSelectorsWithClassInfo(possibleSelectors, classInfo),
                variableDeclarator);

        return variableDeclarator;
    }

    private ExpressionStmt getPageChangeEventStatement(PageChangeEvent pageChangeEvent) {
        MethodCallExpr methodCallExpr = new MethodCallExpr("getDriver");
        MethodCallExpr navigateCall = new MethodCallExpr("navigate");
        MethodCallExpr toCall = new MethodCallExpr("to");
        toCall.addArgument(new StringLiteralExpr(pageChangeEvent.getHref()));
        toCall.setScope(navigateCall);
        navigateCall.setScope(methodCallExpr);
        ExpressionStmt expressionStmt = new ExpressionStmt();
        expressionStmt.setExpression(toCall);
        return expressionStmt;
    }

    private void parseOrCreateCompilationUnit(File file, String qualifiedClassName) throws FileNotFoundException {
        String qualifiedClassNameWithoutExtension = qualifiedClassName;
        if (qualifiedClassName.endsWith("." + JAVA_FILE_EXTENSION)) {
            qualifiedClassNameWithoutExtension = qualifiedClassName.substring(0, qualifiedClassName.length() - 5);
        }
        String classNameWithoutExtension = qualifiedClassNameWithoutExtension
                .substring(qualifiedClassNameWithoutExtension.lastIndexOf('.') + 1);
        String packageName = qualifiedClassNameWithoutExtension.substring(0,
                qualifiedClassNameWithoutExtension.lastIndexOf('.'));
        testContainingClassName = classNameWithoutExtension;

        if (file.exists()) {
            compilationUnit = StaticJavaParser.parse(file);
        } else {
            // TODO fix package declaration
            compilationUnit = new CompilationUnit(packageName);
            ClassOrInterfaceDeclaration classOrInterfaceDeclaration = compilationUnit.addClass(testContainingClassName);
            classOrInterfaceDeclaration.addExtendedType("BrowserTestBase");
        }
    }

    private void addImports() {
        compilationUnit.addImport("com.vaadin.testbench.BrowserTest");
        compilationUnit.addImport("com.vaadin.testbench.BrowserTestBase");
        scenario.getImports().forEach(compilationUnit::addImport);
    }

    private Path getTestFilePath(String qualifiedClassName) {
        if (qualifiedClassName.endsWith("." + JAVA_FILE_EXTENSION)) {
            qualifiedClassName = qualifiedClassName.substring(0, qualifiedClassName.length() - 5);
        }
        qualifiedClassName = qualifiedClassName.replace('.', '/');
        File javaSourceFolder = ProjectFileManager.get().getJavaSourceFolder();
        Path srcFolder = javaSourceFolder.toPath().getParent().getParent();
        Path testFolder = Path.of(srcFolder.toAbsolutePath().toString(), "/test/java");
        String qualifiedClassNameWithExtension = qualifiedClassName.endsWith("." + JAVA_FILE_EXTENSION)
                ? qualifiedClassName
                : (qualifiedClassName + "." + JAVA_FILE_EXTENSION);
        return Path.of(testFolder.toAbsolutePath().toString(), qualifiedClassNameWithExtension);
    }

    private record PossibleSelectorsWithClassInfo(PossibleSelectors possibleSelectors,
            TestBenchElementClassInfo classInfo) {
        @Override
        public boolean equals(Object o) {
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            PossibleSelectorsWithClassInfo that = (PossibleSelectorsWithClassInfo) o;
            return Objects.equals(possibleSelectors, that.possibleSelectors)
                    && Objects.equals(classInfo, that.classInfo);
        }

        @Override
        public int hashCode() {
            return Objects.hash(possibleSelectors, classInfo);
        }
    }
}
