package com.vaadin.copilot;

import java.io.IOException;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.concurrent.Executors;
import java.util.stream.Stream;

import com.vaadin.base.devserver.DevToolsInterface;
import com.vaadin.base.devserver.NamedDaemonThreadFactory;
import com.vaadin.copilot.ai.AICommandHandler;
import com.vaadin.copilot.ai.AICreateFieldsHandler;
import com.vaadin.copilot.ai.CreateLumoThemeVariablesHandler;
import com.vaadin.copilot.analytics.AnalyticsInterceptor;
import com.vaadin.copilot.feedback.FeedbackHandler;
import com.vaadin.copilot.ide.CopilotIDEPlugin;
import com.vaadin.copilot.ide.IdeHandler;
import com.vaadin.copilot.ide.IdePluginCommandHandler;
import com.vaadin.copilot.ide.IdeTooManyRequestException;
import com.vaadin.copilot.ide.OpenComponentInIDE;
import com.vaadin.copilot.javarewriter.SourceSyncChecker;
import com.vaadin.copilot.plugins.accessibilitychecker.AccessibilityCheckerMessageHandler;
import com.vaadin.copilot.plugins.devsetup.DevSetupHandler;
import com.vaadin.copilot.plugins.docs.DocsHandler;
import com.vaadin.copilot.plugins.i18n.I18nHandler;
import com.vaadin.copilot.plugins.testgeneration.GenerateTestsHandler;
import com.vaadin.copilot.plugins.themeeditor.ThemeEditorMessageHandler;
import com.vaadin.copilot.plugins.vaadinversionupdate.VaadinVersionUpdateHandler;
import com.vaadin.copilot.testbenchgenerator.TestBenchRecordHandler;
import com.vaadin.flow.di.Lookup;
import com.vaadin.flow.internal.DevModeHandlerManager;
import com.vaadin.flow.internal.JacksonUtils;
import com.vaadin.flow.server.VaadinContext;
import com.vaadin.flow.server.VaadinSession;
import com.vaadin.flow.server.startup.ApplicationConfiguration;

import tools.jackson.databind.JsonNode;
import tools.jackson.databind.node.ObjectNode;

/**
 * The main code for Copilot for a given VaadinSession instance.
 *
 * <p>
 * One instance of this class is created for each VaadinSession which mean it is
 * ok to use and store VaadinSession specific data in this class and command
 * classes it uses, and in project manager etc.
 */
public class CopilotSession {

    private final VaadinSession vaadinSession;

    private List<CopilotCommand> commands;

    private final CopilotIDEPlugin idePlugin;

    private SourceSyncChecker sourceSyncChecker;
    private CompilationStatusChecker compilationStatusChecker;

    private final Executor executor = Executors.newCachedThreadPool(new NamedDaemonThreadFactory("CopilotAsync"));

    /**
     * Create a new CopilotSession for the given VaadinSession.
     *
     * @param vaadinSession
     *            the VaadinSession
     * @param devToolsInterface
     *            used to send messages back to the browser
     * @throws IOException
     *             if an error occurs
     */
    public CopilotSession(VaadinSession vaadinSession, DevToolsInterface devToolsInterface) throws IOException {
        this.vaadinSession = vaadinSession;
        VaadinContext context = Copilot.getContext(vaadinSession);

        ApplicationConfiguration applicationConfiguration = ApplicationConfiguration.get(context);
        CopilotIDEPlugin.setFolderInLaunchedModule(applicationConfiguration.getProjectFolder().toPath());
        CopilotIDEPlugin.setDevToolsInterface(devToolsInterface);
        if (ProjectFileManager.get() == null
                || ProjectFileManager.get().getApplicationConfiguration() != applicationConfiguration) {
            ProjectFileManager.initialize(applicationConfiguration);
        }
        idePlugin = CopilotIDEPlugin.getInstance();

        Optional<DevModeHandlerManager> devModeHandlerManagerOptional = getDevModeHandlerManager(context);
        if (!applicationConfiguration.isProductionMode() && sourceSyncChecker == null
                && devModeHandlerManagerOptional.isPresent()) {
            sourceSyncChecker = new SourceSyncChecker(devModeHandlerManagerOptional.get());
            compilationStatusChecker = new CompilationStatusChecker();
        }
        setupCommands(applicationConfiguration);
    }

    public void handleConnect(DevToolsInterface devToolsInterface) {
        devToolsInterface.send(Copilot.PREFIX + "init", JacksonUtils.createObjectNode());
        for (CopilotCommand copilotCommand : commands) {
            copilotCommand.handleConnect(devToolsInterface);
        }
    }

    /**
     * Handle a message from the client.
     *
     * @param command
     *            the command
     * @param data
     *            the data, specific to the command
     * @param devToolsInterface
     *            used to send messages back to the browser
     */
    public void handleMessage(String command, JsonNode data, DevToolsInterface devToolsInterface) {
        if (command == null) {
            throw new NullPointerException("command is null");
        }
        compilationStatusChecker.subscribe(devToolsInterface);
        boolean canBeParallel = false;
        canBeParallel = commands.stream().anyMatch(c -> c.canBeParallelCommand(command));
        if (canBeParallel) {
            handleMessageAsync(command, data, devToolsInterface);
        } else {
            handleMessageSync(command, data, devToolsInterface);
        }
    }

    private void handleMessageAsync(String command, JsonNode data, DevToolsInterface devToolsInterface) {
        CompletableFuture<Boolean>[] futures = commands.stream()
                .filter(copilotCommand -> copilotCommand.canBeParallelCommand(command))
                .map(copilotCommand -> CompletableFuture.runAsync(() -> {
                    try {
                        copilotCommand.handleMessage(command, data, devToolsInterface);
                    } catch (Exception e) {
                        ErrorHandler.sendErrorResponse(devToolsInterface, command, data, "Error handling command", e);
                    }
                }, executor)).toArray(CompletableFuture[]::new);
        CompletableFuture.allOf(futures).thenRunAsync(() -> {
            boolean handled = Stream.of(futures).anyMatch(CompletableFuture::join);
            if (!handled) {
                ObjectNode respData = JacksonUtils.createObjectNode();
                if (data.has(CopilotCommand.KEY_REQ_ID)) {
                    respData.put(CopilotCommand.KEY_REQ_ID, data.get(CopilotCommand.KEY_REQ_ID).asString());
                }
                if (data.isObject()) {
                    ((ObjectNode) data).put("error", "Unknown command " + command);

                }
                devToolsInterface.send("unknown-command", data);
            }
        }, executor);
    }

    private void handleMessageSync(String command, JsonNode data, DevToolsInterface devToolsInterface) {
        for (CopilotCommand copilotCommand : commands) {
            try {
                if (copilotCommand.handleMessage(command, data, devToolsInterface)) {
                    return;
                }
            } catch (Exception e) {
                // Something that should maybe have been caught by the handler but was not
                String msg = "Error handling command";
                if (e instanceof IdeTooManyRequestException) {
                    msg = e.getMessage();
                }
                ErrorHandler.sendErrorResponse(devToolsInterface, command, data, msg, e);
                return;
            }
        }
        ObjectNode respData = JacksonUtils.createObjectNode();
        if (data.has(CopilotCommand.KEY_REQ_ID)) {
            respData.put(CopilotCommand.KEY_REQ_ID, data.get(CopilotCommand.KEY_REQ_ID).asString());
        }
        if (data.isObject()) {
            ((ObjectNode) data).put("error", "Unknown command " + command);
        }
        devToolsInterface.send("unknown-command", data);
    }

    private void setupCommands(ApplicationConfiguration applicationConfiguration) {
        CopilotCommand.currentSession.set(vaadinSession);
        commands = List.of(
                // This must be first as it is more of an interceptor than a
                // handler
                new AnalyticsInterceptor(), new IdeHandler(idePlugin), new ErrorHandler(), new OpenComponentInIDE(),
                new ProjectFileHandler(), new AICommandHandler(), new UserInfoHandler(), new I18nHandler(),
                new IdePluginCommandHandler(), new ApplicationInitializer(), new MachineConfigurationHandler(),
                new ProjectStateConfigurationHandler(), new ThemeEditorMessageHandler(),
                new RouteHandler(sourceSyncChecker), new UiServiceHandler(), new AccessibilityCheckerMessageHandler(),
                new FeedbackHandler(), new JavaRewriteHandler(sourceSyncChecker), new GenerateTestsHandler(),
                new JavaParserHandler(), new DocsHandler(), new DevSetupHandler(idePlugin),
                new HotswapDownloadHandler(), new ProjectInfoHandler(), new ApplicationUserSwitchHandler(),
                new VaadinVersionUpdateHandler(applicationConfiguration), new AICreateFieldsHandler(),
                new CreateLumoThemeVariablesHandler(), new ComponentInfoHandler(sourceSyncChecker),
                new TestBenchRecordHandler(), new LitTemplateHandler(), new ApplicationThemeHandler());
        CopilotCommand.currentSession.remove();
    }

    private Optional<DevModeHandlerManager> getDevModeHandlerManager(VaadinContext context) {
        return Optional.ofNullable(context).map(ctx -> ctx.getAttribute(Lookup.class))
                .map(lu -> lu.lookup(DevModeHandlerManager.class));
    }
}
