package com.vaadin.copilot.javarewriter;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

import com.vaadin.flow.internal.JacksonUtils;

import com.fasterxml.jackson.annotation.JsonIgnore;

import tools.jackson.databind.JsonNode;
import tools.jackson.databind.node.ArrayNode;

/**
 * Represents a Java component to be added to the source code.
 *
 * @param tag
 *            the tag of the component, used to determine the Java class
 * @param className
 *            the class name if known, or null to use the tag to look up the
 *            class
 * @param props
 *            the properties of the component
 * @param children
 *            the child components
 * @param metadata
 *            Metadata of the component. Filled when component is copied.
 */
public record JavaComponent(String tag, String className, Map<String, Object> props, List<JavaComponent> children,
        Metadata metadata) {

    public static final String TEXT_NODE = "_TEXT_NODE_";

    public JavaComponent(String text) {
        this(TEXT_NODE, null, Map.of("text", text), new ArrayList<>());
    }

    public JavaComponent(String tag, String className, Map<String, Object> props, List<JavaComponent> children) {
        this(tag, className, props, children, new Metadata());
    }

    public JavaComponent withTag(String tag) {
        return new JavaComponent(tag, className(), props(), children(), metadata());
    }

    public Optional<String> innerText() {
        String text = children().stream().filter(child -> child.tag().equals(JavaComponent.TEXT_NODE))
                .map(textNode -> (String) textNode.props().get("text")).collect(Collectors.joining()).trim();
        if (text.isEmpty()) {
            return Optional.empty();
        }
        return Optional.of(text);
    }

    private static Object propertyValueFromJson(JsonNode jsonValue) {
        if (jsonValue.isBoolean()) {
            return jsonValue.asBoolean();
        } else if (jsonValue.isString()) {
            return jsonValue.asString();
        } else if (jsonValue.isNull()) {
            return null;
        } else if (jsonValue.isIntegralNumber()) {
            return jsonValue.asInt();
        } else if (jsonValue.isFloatingPointNumber()) {
            return jsonValue.asDouble();
        } else if (jsonValue.isObject()) {
            // inner component
            if (jsonValue.has("tag")) {
                return JavaComponent.componentFromJson(jsonValue);
            }
            Map<String, Object> values = new LinkedHashMap<>();
            jsonValue.properties()
                    .forEach(field -> values.put(field.getKey(), propertyValueFromJson(field.getValue())));
            return values;
        } else if (jsonValue.isArray()) {
            List<Object> values = new ArrayList<>();
            jsonValue.forEach(node -> values.add(propertyValueFromJson(node)));
            return values;
        } else {
            throw new IllegalArgumentException("Unsupported JSON value: " + jsonValue);
        }
    }

    /**
     * Creates a new JavaComponent instance from a JSON object.
     *
     * @param json
     *            the JSON object
     * @return the JavaComponent instance
     */
    public static JavaComponent componentFromJson(JsonNode json) {
        String tag = json.has("tag") ? json.get("tag").asString() : null;
        if ("DashboardLayout".equals(tag)) {
            tag = "Dashboard";
        }
        String className = json.has("className") ? json.get("className").asString() : null;
        Map<String, Object> props = new HashMap<>();
        JsonNode jsonProps = json.get("props");
        if (jsonProps != null) {
            jsonProps.properties().forEach(entry -> {
                props.put(entry.getKey(), propertyValueFromJson(entry.getValue()));
            });
        }
        Metadata metadata = new Metadata();
        if (json.has("metadata")) {
            JsonNode jsonMetadata = json.get("metadata");
            if (jsonMetadata.has("javaClass")) {
                className = jsonMetadata.get("javaClass").asString();
            }
            if (jsonMetadata.has("itemType")) {
                JsonNode itemType = jsonMetadata.get("itemType");
                if (itemType.isString()) {
                    metadata.setItemType(itemType.asString());
                }
            }
        }
        List<JavaComponent> children = new ArrayList<>();
        ArrayNode jsonChildren = json.withArray("children");
        if (jsonChildren != null) {
            for (int i = 0; i < jsonChildren.size(); i++) {
                JsonNode jsonChild = jsonChildren.get(i);
                if (jsonChild.isString()) {
                    // Text node
                    children.add(new JavaComponent(jsonChild.asString()));
                } else {
                    JavaComponent child = JavaComponent.componentFromJson(jsonChildren.get(i));
                    Object slot = child.props.get("slot");

                    boolean remapped = false;
                    if (slot instanceof String slotString) {
                        String mappedProperty = mapSlotToProperty(tag, slotString);
                        if (mappedProperty != null) {
                            child.props.remove("slot");
                            addAsSingleOrArray(mappedProperty, child, props);
                            remapped = true;
                        }
                    }
                    if (!remapped) {
                        children.add(child);
                    }
                }
            }
        }

        return new JavaComponent(tag, className, props, children, metadata);
    }

    private static String mapSlotToProperty(String tag, String slot) {

        if ("Card".equals(tag)) {
            return switch (slot) {
            case "title", "subtitle", "media", "footer" -> slot;
            case "header-suffix" -> "headerSuffix";
            case "header-prefix" -> "headerPrefix";
            default -> null;
            };
        }

        // These should really be component specific..
        if ("prefix".equals(slot)) {
            return "prefixComponent";
        } else if ("suffix".equals(slot)) {
            return "suffixComponent";
        } else if ("add-button".equals(slot)) {
            return "uploadButton";
        } else if ("media".equals(slot)) {
            return "media";
        } else if ("header".equals(slot)) {
            return "header";
        } else if ("title".equals(slot)) {
            return "title";
        } else if ("footer".equals(slot)) {
            return "footer";
        }

        return null;
    }

    private static void addAsSingleOrArray(String prop, JavaComponent child, Map<String, Object> props) {
        if (!props.containsKey(prop)) {
            props.put(prop, child);
            return;
        }

        Object existing = props.get(prop);
        // If the existing property is not an array, convert it to an array
        if (existing instanceof JavaComponent[] existingArray) {
            JavaComponent[] newArray = new JavaComponent[existingArray.length + 1];
            System.arraycopy(existingArray, 0, newArray, 0, existingArray.length);
            newArray[existingArray.length] = child;
            props.put(prop, newArray);
        } else if (existing instanceof JavaComponent existingComponent) {
            JavaComponent[] newArray = new JavaComponent[] { existingComponent, child };
            props.put(prop, newArray);
        }
    }

    /**
     * Creates a new JavaComponent instance from a JSON array.
     *
     * @param template
     *            the JSON array
     * @return the JavaComponent instances
     */
    public static List<JavaComponent> componentsFromJson(ArrayNode template) {
        return JacksonUtils.stream(template).map(JavaComponent::componentFromJson).toList();
    }

    /**
     * return items of a component if they exist as a property
     *
     * @return the tag
     */
    @JsonIgnore
    public List<Map<String, Object>> getItemsFromProperty() {
        if (props.containsKey("items"))
            return (List<Map<String, Object>>) props.get("items");
        else if (props.containsKey("dataProvider"))
            return (List<Map<String, Object>>) props.get("dataProvider");
        else
            return new ArrayList<>();
    }

    /**
     * Metadata for the copied component. It will be used in
     * createComponentStatements to make pasted component statements look like the
     * original one.
     */
    public static class Metadata {
        private String localVariableName;
        private String fieldVariableName;
        private String originalClassName;
        private String itemType;

        public String getLocalVariableName() {
            return localVariableName;
        }

        public void setLocalVariableName(String localVariableName) {
            this.localVariableName = localVariableName;
        }

        public String getFieldVariableName() {
            return fieldVariableName;
        }

        public void setFieldVariableName(String fieldVariableName) {
            this.fieldVariableName = fieldVariableName;
        }

        public String getOriginalClassName() {
            return originalClassName;
        }

        public void setOriginalClassName(String originalClassName) {
            this.originalClassName = originalClassName;
        }

        public String getItemType() {
            return itemType;
        }

        public void setItemType(String itemType) {
            this.itemType = itemType;
        }
    }
}
