package com.vaadin.copilot;

import javax.sql.DataSource;

import java.lang.reflect.Method;
import java.sql.Connection;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;

import com.vaadin.flow.server.VaadinContext;
import com.vaadin.flow.server.VaadinServletContext;
import com.vaadin.flow.server.auth.AnnotatedViewAccessChecker;
import com.vaadin.flow.server.auth.NavigationAccessControl;
import com.vaadin.hilla.EndpointRegistry;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.aop.support.AopUtils;
import org.springframework.beans.factory.config.BeanDefinition;
import org.springframework.beans.factory.support.BeanDefinitionRegistry;
import org.springframework.boot.SpringBootVersion;
import org.springframework.boot.autoconfigure.SpringBootApplication;
import org.springframework.boot.h2console.autoconfigure.H2ConsoleProperties;
import org.springframework.cglib.proxy.Proxy;
import org.springframework.core.SpringVersion;
import org.springframework.core.env.Environment;
import org.springframework.stereotype.Service;
import org.springframework.util.ClassUtils;
import org.springframework.web.context.WebApplicationContext;
import org.springframework.web.context.support.WebApplicationContextUtils;

/**
 * Provides Spring related helpers for copilot. Depends on Spring classes and
 * cannot be directly imported
 */
public class SpringIntegration {

    /**
     * Returns the value of the given property from the Spring environment of the
     * given context. See {@link SpringBridge}
     *
     * @param context
     *            the Vaadin servlet context
     * @param property
     *            the property name
     * @return the property value or null if not found
     */
    public static String getPropertyValue(VaadinContext context, String property) {
        Environment env = getWebApplicationContext(context).getEnvironment();
        return env.getProperty(property);
    }

    public static WebApplicationContext getWebApplicationContext(VaadinContext context) {
        if (context instanceof VaadinServletContext vaadinServletContext) {
            WebApplicationContext webAppContext = WebApplicationContextUtils
                    .getWebApplicationContext(vaadinServletContext.getContext());
            if (webAppContext == null) {
                throw new IllegalStateException("No Spring web application context available");
            }
            return webAppContext;
        } else {
            throw new IllegalStateException("VaadinContext is not a VaadinServletContext");
        }
    }

    /**
     * Returns the Spring Boot application class of the given context. See
     * {@link SpringBridge}
     *
     * @param context
     *            the Vaadin servlet context
     * @return the Spring Boot application class or null if not found
     */
    public static Class<?> getApplicationClass(VaadinContext context) {
        Map<String, Object> beans = getWebApplicationContext(context)
                .getBeansWithAnnotation(SpringBootApplication.class);
        Class<?> appClass = beans.values().iterator().next().getClass();
        if (Proxy.isProxyClass(appClass)) {
            appClass = appClass.getSuperclass();
        }
        while (isCglibProxy(appClass)) {
            appClass = appClass.getSuperclass();
        }
        return appClass;
    }

    private static boolean isCglibProxy(Class<?> appClass) {
        return appClass.getName().contains("$$SpringCGLIB$$");
    }

    /**
     * Returns whether Flow view security is enabled in the given context.
     *
     * @param context
     *            the Vaadin servlet context
     * @return true if Flow view security is enabled, false otherwise
     */
    public static Boolean isViewSecurityEnabled(VaadinContext context) {
        WebApplicationContext webApplicationContext = getWebApplicationContext(context);
        String[] naviAccessControl = webApplicationContext.getBeanNamesForType(NavigationAccessControl.class);
        if (naviAccessControl.length != 1) {
            return false;
        }
        NavigationAccessControl accessControl = (NavigationAccessControl) webApplicationContext
                .getBean(naviAccessControl[0]);
        return accessControl.hasAccessChecker(AnnotatedViewAccessChecker.class);
    }

    public static List<SpringBridge.ServiceMethodInfo> getEndpoints(VaadinContext context) {
        EndpointRegistry endpointRegistry = getBean(EndpointRegistry.class, context);
        Map<String, EndpointRegistry.VaadinEndpointData> vaadinEndpoints = endpointRegistry.getEndpoints();

        return getEndpointInfos(vaadinEndpoints);
    }

    public static List<SpringBridge.ServiceMethodInfo> getFlowUIServices(VaadinContext context) {
        WebApplicationContext webApplicationContext = getWebApplicationContext(context);
        List<SpringBridge.ServiceMethodInfo> serviceMethods = new ArrayList<>();

        if (webApplicationContext instanceof BeanDefinitionRegistry registry) {
            List<String> springServiceClassnames = Arrays
                    .stream(getWebApplicationContext(context).getBeanNamesForAnnotation(Service.class))
                    .map(registry::getBeanDefinition).map(BeanDefinition::getBeanClassName).filter(Objects::nonNull)
                    .toList();
            for (String springServiceClass : springServiceClassnames) {
                try {
                    String targetServiceClassName = getTargetClassName(springServiceClass);
                    Class<?> cls = Class.forName(targetServiceClassName);
                    if (cls.getPackage().getName().startsWith("com.vaadin")) {
                        continue;
                    }
                    List<SpringBridge.ServiceMethodInfo> classMethods = Arrays.stream(cls.getMethods())
                            .filter(m -> !isObjectMethod(m))
                            .map(method -> new SpringBridge.ServiceMethodInfo(cls, method)).toList();
                    serviceMethods.addAll(classMethods);
                } catch (NoClassDefFoundError | ClassNotFoundException e) {
                    throw new RuntimeException(e);
                }
            }
        } else {
            getLogger().debug("WebApplicationContext is not a BeanDefinitionRegistry");
        }
        return serviceMethods;
    }

    /**
     * Extracts the original (user-defined) class name from a potentially proxied
     * class name.
     * <p>
     * This is particularly useful when dealing with Spring-generated CGLIB proxy
     * class names, which include a suffix such as {@code $$SpringCGLIB$$0}. For
     * example, if the input is: {@code com.example.MyService$$SpringCGLIB$$0}, this
     * method returns {@code com.example.MyService}.
     * </p>
     *
     * @param candidateClassName
     *            the fully qualified name of the candidate class, which may be a
     *            CGLIB proxy
     * @return the base class name, with any CGLIB proxy suffix removed if present
     */
    private static String getTargetClassName(String candidateClassName) {
        if (candidateClassName.contains(ClassUtils.CGLIB_CLASS_SEPARATOR)) {
            return candidateClassName.substring(0, candidateClassName.indexOf(ClassUtils.CGLIB_CLASS_SEPARATOR));
        }
        return candidateClassName;
    }

    private static Logger getLogger() {
        return LoggerFactory.getLogger(SpringIntegration.class);
    }

    private static boolean isObjectMethod(Method m) {
        return m.getDeclaringClass() == Object.class;
    }

    private static List<SpringBridge.ServiceMethodInfo> getEndpointInfos(
            Map<String, EndpointRegistry.VaadinEndpointData> vaadinEndpoints) {
        List<SpringBridge.ServiceMethodInfo> endpointInfos = new ArrayList<>();
        for (EndpointRegistry.VaadinEndpointData endpoint : vaadinEndpoints.values()) {
            if (isInternalEndpoint(endpoint)) {
                continue;
            }
            Class<?> endpointClass = AopUtils.getTargetClass(endpoint.getEndpointObject());
            Map<String, Method> methods = endpoint.getMethods();
            for (Method method : methods.values()) {
                endpointInfos.add(new SpringBridge.ServiceMethodInfo(endpointClass, method));
            }
        }
        return endpointInfos;
    }

    private static boolean isInternalEndpoint(EndpointRegistry.VaadinEndpointData endpointData) {
        String name = endpointData.getEndpointObject().getClass().getName();
        return name.startsWith("com.vaadin");
    }

    /**
     * Gets version information for Spring Boot and related libraries.
     *
     * @return version information
     */
    public static SpringBridge.VersionInfo getVersionInfo() {
        return new SpringBridge.VersionInfo(SpringBootVersion.getVersion(), SpringVersion.getVersion());
    }

    static <T> T getBean(Class<T> cls, VaadinContext context) {
        return getWebApplicationContext(context).getBean(cls);
    }

    public static Optional<SpringBridge.H2Info> getH2Info(VaadinContext context) {
        try {
            H2ConsoleProperties h2ConsoleProperties = getWebApplicationContext(context)
                    .getBean(H2ConsoleProperties.class);
            DataSource dataSource = getWebApplicationContext(context).getBean(DataSource.class);
            if (h2ConsoleProperties.isEnabled()) {
                String jdbcUrl = getConnectionUrl(dataSource);
                String path = h2ConsoleProperties.getPath();
                return Optional.of(new SpringBridge.H2Info(jdbcUrl, path));
            }
        } catch (Throwable e) {
            // No bean found
            getLogger().debug("No H2ConsoleProperties or DataSource bean found", e);
        }
        return Optional.empty();
    }

    private static String getConnectionUrl(DataSource dataSource) {
        try (Connection connection = dataSource.getConnection()) {
            return connection.getMetaData().getURL();
        } catch (Exception ex) {
            return null;
        }
    }

}
