package com.vaadin.copilot.javarewriter;

import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

import com.vaadin.copilot.ComponentSourceFinder;
import com.vaadin.copilot.JavaReflectionUtil;
import com.vaadin.copilot.customcomponent.CustomComponent;
import com.vaadin.copilot.customcomponent.CustomComponents;
import com.vaadin.copilot.exception.CopilotException;
import com.vaadin.copilot.exception.report.ExceptionReportCreator;
import com.vaadin.flow.component.Component;
import com.vaadin.flow.component.HasEnabled;
import com.vaadin.flow.component.HasSize;
import com.vaadin.flow.component.HasText;
import com.vaadin.flow.component.HasTheme;
import com.vaadin.flow.component.HasValueAndElement;
import com.vaadin.flow.dom.ElementStateProvider;

import com.github.javaparser.ast.body.VariableDeclarator;
import com.github.javaparser.ast.type.ClassOrInterfaceType;
import com.github.javaparser.ast.type.Type;
import com.github.javaparser.resolution.types.ResolvedType;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import tools.jackson.databind.node.JsonNodeFactory;
import tools.jackson.databind.node.ObjectNode;

/**
 * Handles the copy &amp; paste functionality for flow components.
 */
public class JavaRewriterCopyPasteHandler {
    private static final String COM_VAADIN_FLOW = "com.vaadin.flow";

    private static final String SPAN_CLASS_NAME = "com.vaadin.flow.component.html.Span";
    private static final String COMBOBOX_CLASS_NAME = "com.vaadin.flow.component.combobox.ComboBox";
    private static final String MENU_BAR_COMPONENT_CLASS_NAME = "com.vaadin.flow.component.menubar.MenuBar";
    private static final String MENU_BAR_ROOT_ITEM_CLASS_NAME = "com.vaadin.flow.component.menubar.MenuBarRootItem";
    private static final String ACCORDION_PANEL_CLASS_NAME = "com.vaadin.flow.component.accordion.AccordionPanel";
    private static final String DETAILS_CLASS_NAME = "com.vaadin.flow.component.details.Details";
    private static final String GRID_CLASS_NAME = "com.vaadin.flow.component.grid.Grid";
    private static final String TREE_GRID_CLASS_NAME = "com.vaadin.flow.component.treegrid.TreeGrid";
    private static final String CHECKBOX_GROUP_CLASS_NAME = "com.vaadin.flow.component.checkbox.CheckboxGroup";
    private static final String RADIO_GROUP_CLASS_NAME = "com.vaadin.flow.component.radiobutton.RadioButtonGroup";

    private static final String VALUE_PROPERTY_KEY = "value";
    private static final String INVALID_PROPERTY_KEY = "invalid";
    private static final String ERROR_MESSAGE_PROPERTY_KEY = "errorMessage";
    private static final String ENABLED_PROPERTY_KEY = "enabled";

    private static final String INDETERMINATE_PROPERTY_KEY = "indeterminate";
    private static final String CHECKED_PROPERTY_KEY = "checked";
    private static final String AUTO_FOCUS_PROPERTY_KEY = "autofocus";
    private static final String REQUIRED_PROPERTY_KEY = "required";
    private static final String OPENED_PROPERTY_KEY = "opened";

    private static final String ACCESSIBLE_NAME_REF = "accessibleNameRef";
    private static final String ACCESSIBLE_NAME = "accessibleName";
    private static final String REQUIRED_INDICATOR_VISIBLE = "requiredIndicatorVisible";

    private static final String READONLY = "readonly";
    private static final String READ_ONLY_CAMEL_CASE = "readOnly";
    private static final String MAX_ITEMS_VISIBLE = "maxItemsVisible";

    /**
     * Collects required data for copying and pasting a component
     *
     * @param componentTypeAndSourceLocation
     *            Component type and source location with children info included
     * @param sourceFinder
     *            Component source finder to access sibling sources of given
     *            component
     * @return JavaComponent that will be used for pasting.
     * @throws IOException
     *             exception happens when file read fails
     */
    public JavaComponent getCopiedJavaComponent(JavaFileSourceProvider javaFileSourceProvider,
            ComponentTypeAndSourceLocation componentTypeAndSourceLocation,
            ExceptionReportCreator exceptionReportCreator, ComponentSourceFinder sourceFinder) throws IOException {
        Component component = componentTypeAndSourceLocation.component();
        String className = getFirstComponentNameInheritanceChain(componentTypeAndSourceLocation.inheritanceChain());
        ElementStateProvider stateProvider = component.getElement().getStateProvider();
        Map<String, Object> properties = stateProvider.getPropertyNames(component.getElement().getNode())
                .filter(propertyName -> !propertyName.equals("manualValidation"))
                .collect(Collectors.toMap(k -> k, k -> stateProvider.getProperty(component.getElement().getNode(), k)));

        ComponentInfoFinder componentInfoFinder = new ComponentInfoFinder(javaFileSourceProvider,
                componentTypeAndSourceLocation, sourceFinder::getSiblingsTypeAndSourceLocations);
        ComponentInfo componentInfo = componentInfoFinder.find();
        exceptionReportCreator.addComponentInfo(componentInfo);
        addSuffixOrPrefixAttributes(component, properties);
        handleAccordionPanel(component, properties);
        if (isCustomComponent(className)) {
            Optional<CustomComponent> customComponentInfoOpt = CustomComponents
                    .getCustomComponentInfo(component.getClass());
            if (customComponentInfoOpt.isPresent()) {
                // not used for now, but might be useful in the future when we want to copy the
                // methods, children of custom components.
                CustomComponent customComponent = customComponentInfoOpt.get();
                customComponent.getChildAddableMethods();
            }
        } else {
            addExtraProperties(component, properties);
        }
        filterOutProperties(component, componentInfo, componentTypeAndSourceLocation.inheritanceChain(), properties);

        JavaComponent.Metadata copyMetadata = new JavaComponent.Metadata();
        try {
            copyMetadata = getCopyMetadata(componentTypeAndSourceLocation, componentInfo);
        } catch (Exception ex) {
            getLogger().error(ex.getMessage(), ex);
        }
        List<JavaComponent> childrenList = new ArrayList<>();
        if (!isCustomComponent(className)) {
            childrenList = childrenFilter(javaFileSourceProvider, componentTypeAndSourceLocation,
                    exceptionReportCreator, sourceFinder);
        }
        return new JavaComponent(getTag(componentTypeAndSourceLocation.inheritanceChain()), className, properties,
                childrenList, copyMetadata);
    }

    private void handleAccordionPanel(Component component, Map<String, Object> properties) {
        boolean isAccordionPanel = ACCORDION_PANEL_CLASS_NAME.equals(component.getClass().getName());
        if (!isAccordionPanel) {
            return;
        }
        // use reflection to reach summary field:
        Class<?> detailsClass = component.getClass();
        Method getSummaryTextMethod = null;
        try {
            getSummaryTextMethod = detailsClass.getMethod("getSummaryText");
            String summaryText = (String) getSummaryTextMethod.invoke(component);
            properties.put("summary", summaryText);
        } catch (IllegalAccessException e) {
            getLogger().debug("Could not access the method {}", getSummaryTextMethod.getName(), e);
        } catch (InvocationTargetException e) {
            getLogger().debug("Could not invoke the method {}", getSummaryTextMethod.getName(), e);
        } catch (NoSuchMethodException e) {
            getLogger().debug("Could not find the method getSummaryText", e);
        }
    }

    private List<JavaComponent> childrenFilter(JavaFileSourceProvider javaFileSourceProvider,
            ComponentTypeAndSourceLocation componentTypeAndSourceLocation,
            ExceptionReportCreator exceptionReportCreator, ComponentSourceFinder sourceFinder) throws IOException {
        List<JavaComponent> childrenList = new ArrayList<>();
        for (ComponentTypeAndSourceLocation typeAndSourceLocation : componentTypeAndSourceLocation.children()) {
            if (typeAndSourceLocation.createLocationInProject().isPresent()) {
                JavaComponent copiedJavaComponent = getCopiedJavaComponent(javaFileSourceProvider,
                        typeAndSourceLocation, exceptionReportCreator, sourceFinder);
                childrenList.add(copiedJavaComponent);
            } else {
                return childrenFilter(javaFileSourceProvider, typeAndSourceLocation, exceptionReportCreator,
                        sourceFinder);
            }
        }
        return childrenList;
    }

    /**
     * We can remove this method later, as it was created before we started
     * supporting Custom Components. This method goes through the inheritance chain
     * and tries to find the first class that is a Flow Component.
     * <p>
     * This logic is now outdated for custom components, as it prevents finding the
     * actual Custom Component class.
     * </p>
     *
     * @deprecated use {@code getCustomComponentClassName()} instead. This method
     *             will be removed in a future release.
     */
    @Deprecated(forRemoval = true)
    private String getFlowComponentClassName(List<Class<?>> inheritanceChain) {
        for (Class<?> aClass : inheritanceChain) {
            if (aClass.getName().startsWith(COM_VAADIN_FLOW)) {
                return aClass.getName();
            }
        }
        return inheritanceChain.get(0).getName();
    }

    private boolean isCustomComponent(String className) {
        return CustomComponents.isCustomComponent(className);
    }

    private String getFirstComponentNameInheritanceChain(List<Class<?>> inheritanceChain) {
        return inheritanceChain.get(0).getName();
    }

    /**
     * When a view is copied from the project, it should return the extended class
     * instead of view itself. //TODO in future this might change
     *
     * @param inheritanceChain
     *            inheritance chain of class
     * @return Tag name of the component e.g. class simple name
     */
    private String getTag(List<Class<?>> inheritanceChain) {
        for (Class<?> aClass : inheritanceChain) {
            if (aClass.getName().startsWith(COM_VAADIN_FLOW)) {
                return getTemplateTag(aClass.getSimpleName());
            }
        }
        return getTemplateTag(inheritanceChain.get(0).getSimpleName());
    }

    private String getTemplateTag(String originalTag) {
        if ("RadioButtonGroup".equals(originalTag)) {
            return "RadioGroup";
        }
        if ("CheckBoxItem".equals(originalTag)) {
            return "Checkbox";
        }
        return originalTag;
    }

    private JavaComponent.Metadata getCopyMetadata(ComponentTypeAndSourceLocation componentTypeAndSourceLocation,
            ComponentInfo componentInfo) {
        JavaComponent.Metadata metadata = new JavaComponent.Metadata();
        if (componentTypeAndSourceLocation.getCreateLocationOrThrow().className().startsWith("com.vaadin.flow.data")) {
            return metadata;
        }
        if (componentTypeAndSourceLocation.inheritanceChain().stream()
                .anyMatch(inheritance -> inheritance.getName().equals(GRID_CLASS_NAME))) {
            Class<?> beanTypeClass = getGridBeanType(componentInfo);
            String beanTypeClassName = beanTypeClass.getName();
            if (beanTypeClassName.contains("$")) {
                beanTypeClassName = beanTypeClassName.replace("$", ".");
            }
            metadata.setItemType(beanTypeClassName);
        }
        ComponentCreateInfo createInfo = componentInfo.getCreateInfoOrThrow();
        Optional.ofNullable(createInfo.getLocalVariableName()).ifPresent(metadata::setLocalVariableName);
        Optional.ofNullable(createInfo.getFieldName()).ifPresent(metadata::setFieldVariableName);
        metadata.setOriginalClassName(componentTypeAndSourceLocation.component().getClass().getName());
        return metadata;
    }

    private Class<?> getGridBeanType(ComponentInfo componentInfo) {
        VariableDeclarator variableDeclarator = null;

        if (componentInfo.getCreateInfoOrThrow().getFieldDeclaration() != null) {
            variableDeclarator = componentInfo.getCreateInfoOrThrow().getFieldDeclaration().getChildNodes().stream()
                    .filter(VariableDeclarator.class::isInstance).map(VariableDeclarator.class::cast).findFirst()
                    .orElseThrow(() -> new CopilotException("Unable to find grid declarator to resolve bean type"));
        } else if (componentInfo.getCreateInfoOrThrow().getLocalVariableDeclarator() != null) {
            variableDeclarator = componentInfo.getCreateInfoOrThrow().getLocalVariableDeclarator();
        }
        if (variableDeclarator == null) {
            throw new CopilotException("Unable to find variable declarator to resolve bean type");
        }
        Type type = variableDeclarator.getType();
        if (type != null && type.isClassOrInterfaceType()) {
            ClassOrInterfaceType classOrInterfaceType = type.asClassOrInterfaceType();
            if (classOrInterfaceType.getTypeArguments().isPresent()
                    && classOrInterfaceType.getTypeArguments().get().size() == 1) {
                Type genericArgType = classOrInterfaceType.getTypeArguments().get().get(0);
                ResolvedType resolve = genericArgType.resolve();
                return JavaReflectionUtil.getClass(resolve.describe());
            }
        }
        throw new CopilotException("Unable to resolve bean type");
    }

    private void addExtraProperties(Component component, Map<String, Object> properties) {

        if (component.getClassName() != null && !component.getClassName().isEmpty()) {
            properties.put("className", component.getClassName());
        }
        if (component.getElement().hasAttribute("disabled")) {
            properties.put(ENABLED_PROPERTY_KEY, false);
        }

        if (component instanceof HasText hasText && !"".equals(hasText.getText())) {
            properties.put("text", hasText.getText());
        }

        if (component instanceof HasTheme hasTheme && null != hasTheme.getThemeName()) {
            properties.put("themeName", hasTheme.getThemeName());
        }

        if (component instanceof HasEnabled hasEnabled) {
            properties.put(ENABLED_PROPERTY_KEY, hasEnabled.isEnabled());
        }

        if (!component.getElement().getThemeList().isEmpty()) {
            properties.put("theme", String.join(" ", component.getElement().getThemeList()));
        }
        // adding src and alt attributes as properties.
        if ("com.vaadin.flow.component.html.Image".equals(component.getClass().getName())) {
            if (!properties.containsKey("src") && component.getElement().hasAttribute("src")) {
                properties.put("src", component.getElement().getAttribute("src"));
            }
            if (!properties.containsKey("alt") && component.getElement().hasAttribute("alt")) {
                properties.put("alt", component.getElement().getAttribute("alt"));
            }
        }
        if ("com.vaadin.flow.component.radiobutton.RadioButton".equals(component.getClass().getName())
                && !properties.containsKey("label")) {
            component.getChildren()
                    .filter(child -> child.getClass().getName().equals("com.vaadin.flow.component.html.Label"))
                    .findFirst().ifPresent(label -> properties.put("label", label.getElement().getTextRecursively()));
        }
        if ("com.vaadin.flow.component.richtexteditor.RichTextEditor".equals(component.getClass().getName())) {
            JavaReflectionUtil.getFieldValue(component, "currentMode")
                    .ifPresent(currentModeValue -> properties.put("valueChangeMode", currentModeValue));
        }
        if ("com.vaadin.flow.component.avatar.Avatar".equals(component.getClass().getName())
                && component.getElement().hasAttribute("img")) {
            properties.put("img", component.getElement().getAttribute("img"));
        }
        if ("com.vaadin.flow.component.icon.Icon".equals(component.getClass().getName())
                && component.getElement().hasAttribute("icon")) {
            properties.put("icon", component.getElement().getAttribute("icon"));
        }
        if ("com.vaadin.flow.component.html.Anchor".equals(component.getClass().getName())
                && component.getElement().hasAttribute("href")) {
            properties.put("href", component.getElement().getAttribute("href"));
        }
        // Convert the Double value to Integer as the maxItemsVisible expects Integer
        // values,
        // but we got Double values from before parsing the element.
        if ("com.vaadin.flow.component.avatar.AvatarGroup".equals(component.getClass().getName())) {
            properties.computeIfPresent(MAX_ITEMS_VISIBLE, (key, value) -> ((Double) value).intValue());
        }

        if (DETAILS_CLASS_NAME.equals(component.getClass().getName())) {
            Optional<Object> summary = JavaReflectionUtil.getFieldValue(component, "summary");
            if (summary.isPresent()) {
                Object summaryField = summary.get();
                if (summaryField instanceof Component summaryComponent
                        && summaryField.getClass().getName().equals(SPAN_CLASS_NAME)) {
                    properties.put("summary", summaryComponent.getElement().getText());
                }
            }
        }
        addHasSizeProperties(component, properties);
        addMenuBarProperties(component, properties);
    }

    private void addHasSizeProperties(Component component, Map<String, Object> properties) {
        if (component instanceof HasSize hasSize) {
            if (hasSize.getMaxHeight() != null) {
                properties.put("maxHeight", hasSize.getMaxHeight());
            }
            if (hasSize.getMinHeight() != null) {
                properties.put("minHeight", hasSize.getMinHeight());
            }
            if (hasSize.getMinWidth() != null) {
                properties.put("minWidth", hasSize.getMinWidth());
            }
            if (hasSize.getMaxWidth() != null) {
                properties.put("maxWidth", hasSize.getMaxWidth());
            }
            if (hasSize.getWidth() != null) {
                properties.put("width", hasSize.getWidth());
            }
            if (hasSize.getHeight() != null) {
                properties.put("height", hasSize.getHeight());
            }
        }
    }

    private void filterOutProperties(Component component, ComponentInfo componentInfo, List<Class<?>> inheritanceChain,
            Map<String, Object> properties) {
        boolean textFieldBase = inheritanceChain.stream()
                .anyMatch(clazz -> clazz.getName().equals("com.vaadin.flow.component.textfield.TextFieldBase"));
        boolean select = "com.vaadin.flow.component.select.Select".equals(component.getClass().getName());
        boolean checkboxGroup = "com.vaadin.flow.component.checkbox.CheckboxGroup"
                .equals(component.getClass().getName());
        boolean radioButtonGroup = "com.vaadin.flow.component.radiobutton.RadioButtonGroup"
                .equals(component.getClass().getName());
        boolean richTextEditor = "com.vaadin.flow.component.richtexteditor.RichTextEditor"
                .equals(component.getClass().getName());
        boolean timePicker = "com.vaadin.flow.component.timepicker.TimePicker".equals(component.getClass().getName());
        boolean dateTimePicker = "com.vaadin.flow.component.datetimepicker.DateTimePicker"
                .equals(component.getClass().getName());
        boolean checkbox = "com.vaadin.flow.component.checkbox.Checkbox".equals(component.getClass().getName());
        boolean accordion = "com.vaadin.flow.component.accordion.Accordion".equals(component.getClass().getName());
        boolean accordionPanel = ACCORDION_PANEL_CLASS_NAME.equals(component.getClass().getName());
        boolean grid = GRID_CLASS_NAME.equals(component.getClass().getName());
        boolean treeGrid = TREE_GRID_CLASS_NAME.equals(component.getClass().getName());

        if (accordion || accordionPanel) {
            properties.remove(OPENED_PROPERTY_KEY);
        }

        if (grid || treeGrid) {
            properties.remove("size");
            properties.remove("pageSize");
        }

        if (textFieldBase || select || checkboxGroup || radioButtonGroup || timePicker || dateTimePicker) {
            // filtering out invalid properties that are generated in runtime for some
            // components
            properties.remove(VALUE_PROPERTY_KEY);
            properties.remove(INVALID_PROPERTY_KEY);
            properties.remove(ERROR_MESSAGE_PROPERTY_KEY);
            if (select) {
                properties.remove(OPENED_PROPERTY_KEY);
            }
        }
        if (richTextEditor) {
            properties.remove("htmlValue");
            properties.remove(VALUE_PROPERTY_KEY);
        }
        if ("com.vaadin.flow.component.checkbox.CheckboxGroup$CheckBoxItem".equals(component.getClass().getName())) {
            properties.remove(CHECKED_PROPERTY_KEY);
            properties.remove("disabled");
        }
        if ("com.vaadin.flow.component.progressbar.ProgressBar".equals(component.getClass().getName())) {
            removeSetterIfNotPresentInSourceCode(componentInfo, properties, "max");
            removeSetterIfNotPresentInSourceCode(componentInfo, properties, "min");
            removeSetterIfNotPresentInSourceCode(componentInfo, properties, VALUE_PROPERTY_KEY);
        }
        if (properties.containsKey(ACCESSIBLE_NAME_REF)) {
            properties.put("ariaLabelledBy", properties.get(ACCESSIBLE_NAME_REF));
            properties.remove(ACCESSIBLE_NAME_REF);
        }
        if (properties.containsKey(ACCESSIBLE_NAME)) {
            properties.put("ariaLabel", properties.get(ACCESSIBLE_NAME));
            properties.remove(ACCESSIBLE_NAME);
        }
        if (checkbox) {
            if (properties.containsKey(INDETERMINATE_PROPERTY_KEY)
                    && properties.get(INDETERMINATE_PROPERTY_KEY).equals(Boolean.FALSE)) {
                removeSetterIfNotPresentInSourceCode(componentInfo, properties, INDETERMINATE_PROPERTY_KEY);
            }
            if (properties.containsKey(CHECKED_PROPERTY_KEY)
                    && properties.get(CHECKED_PROPERTY_KEY).equals(Boolean.FALSE)) {
                removeSetterIfNotPresentInSourceCode(componentInfo, properties, CHECKED_PROPERTY_KEY);
            }
            if (properties.containsKey(AUTO_FOCUS_PROPERTY_KEY)
                    && properties.get(AUTO_FOCUS_PROPERTY_KEY).equals(Boolean.FALSE)) {
                removeSetterIfNotPresentInSourceCode(componentInfo, properties, AUTO_FOCUS_PROPERTY_KEY);
            }
            removeSetterIfNotPresentInSourceCode(componentInfo, properties, VALUE_PROPERTY_KEY);
        }
        if (component instanceof HasEnabled hasEnabled && hasEnabled.isEnabled()) {
            // check if source code is enabled
            removeSetterIfNotPresentInSourceCode(componentInfo, properties, ENABLED_PROPERTY_KEY);
        }
        if (component instanceof HasValueAndElement<?, ?>) {
            if (properties.containsKey(READONLY)) {
                properties.put(READ_ONLY_CAMEL_CASE, properties.get(READONLY));
                properties.remove(READONLY);
                if (properties.get(READ_ONLY_CAMEL_CASE).equals(Boolean.FALSE)) {
                    removeSetterIfNotPresentInSourceCode(componentInfo, properties, READ_ONLY_CAMEL_CASE);
                }
            }
            if (properties.containsKey(REQUIRED_PROPERTY_KEY)) {
                properties.put(REQUIRED_INDICATOR_VISIBLE, properties.get(REQUIRED_PROPERTY_KEY));
                properties.remove(REQUIRED_PROPERTY_KEY);
                if (properties.get(REQUIRED_INDICATOR_VISIBLE).equals(Boolean.FALSE)) {
                    removeSetterIfNotPresentInSourceCode(componentInfo, properties, REQUIRED_INDICATOR_VISIBLE);
                }
            }
        }
        filterOutComboBoxProperties(component, properties);
    }

    private void removeSetterIfNotPresentInSourceCode(ComponentInfo componentInfo, Map<String, Object> properties,
            String propertyKey) {
        String setterName = JavaRewriterUtil.getSetterName(propertyKey, componentInfo.type(), false);
        boolean setterFound = JavaRewriterUtil.findMethodCalls(componentInfo).stream()
                .anyMatch(f -> f.getNameAsString().equals(setterName));
        if (!setterFound) {
            properties.remove(propertyKey);
        }
    }

    /**
     * Sets slot property of the given component when it's defined as prefix or
     * suffix in its parent.
     *
     * @param possiblyPrefixOrSuffixComponent
     *            Component that may be a prefix or suffix component. e.g. Icon
     *            component in Button
     * @param properties
     *            properties of the component
     */
    private void addSuffixOrPrefixAttributes(Component possiblyPrefixOrSuffixComponent,
            Map<String, Object> properties) {
        Optional<Component> parentOptional = possiblyPrefixOrSuffixComponent.getParent();
        if (parentOptional.isEmpty()) {
            return;
        }
        Component parent = parentOptional.get();
        // check if element is a prefix element of its parent
        getPrefixComponent(parent).filter(prefix -> prefix.getElement().getNode()
                .getId() == possiblyPrefixOrSuffixComponent.getElement().getNode().getId())
                .ifPresent(prefix -> properties.put("slot", "prefix"));
        getSuffixComponent(parent)
                .filter(suffixComponent -> suffixComponent.getElement().getNode()
                        .getId() == possiblyPrefixOrSuffixComponent.getElement().getNode().getId())
                .ifPresent(suffixComponent -> properties.put("slot", "suffix"));
    }

    private Optional<Component> getPrefixComponent(Component component) {
        boolean hasPrefixComponent = Arrays.stream(component.getClass().getInterfaces())
                .anyMatch(interface0 -> interface0.getName().equals("com.vaadin.flow.component.shared.HasPrefix"));
        if (!hasPrefixComponent) {
            return Optional.empty();
        }
        return getMethodValue(component, "getPrefixComponent");
    }

    private Optional<Component> getSuffixComponent(Component component) {
        boolean hasPrefixComponent = Arrays.stream(component.getClass().getInterfaces())
                .anyMatch(interface0 -> interface0.getName().equals("com.vaadin.flow.component.shared.HasSuffix"));
        if (!hasPrefixComponent) {
            return Optional.empty();
        }
        return getMethodValue(component, "getSuffixComponent");
    }

    private <T> Optional<T> getMethodValue(Object target, String methodName) {
        try {
            Method declaredMethod = target.getClass().getMethod(methodName);
            Object result = declaredMethod.invoke(target);
            return Optional.ofNullable(result).map(o -> (T) o);
        } catch (NoSuchMethodException e) {
            getLogger().debug("Could not find method {} in {}", methodName, target.getClass());
        } catch (InvocationTargetException | IllegalAccessException e) {
            getLogger().debug("Could not invoke method {} in {}", methodName, target.getClass());
        }
        return Optional.empty();
    }

    private void addMenuBarProperties(Component component, Map<String, Object> properties) {
        boolean isMenuBar = MENU_BAR_COMPONENT_CLASS_NAME.equals(component.getClass().getName());
        if (!isMenuBar) {
            return;
        }
        List<String> texts = new ArrayList<>();
        List<Component> rootItems = component.getChildren()
                .filter(child -> MENU_BAR_ROOT_ITEM_CLASS_NAME.equals(child.getClass().getName())).toList();
        for (Component rootItem : rootItems) {
            if (rootItem instanceof HasText hasText) {
                texts.add(hasText.getText());
            }
        }
        List<ObjectNode> textObjectNodes = texts.stream()
                .map(k -> new ObjectNode(JsonNodeFactory.instance).put("text", k)).toList();
        properties.put("items", textObjectNodes);
    }

    private void filterOutComboBoxProperties(Component component, Map<String, Object> properties) {
        if (!COMBOBOX_CLASS_NAME.equals(component.getClass().getName())) {
            return;
        }
        properties.remove("_clientSideFilter");
        properties.remove("pageSize");
        properties.remove("itemIdPath");
        properties.remove("itemValuePath");
        properties.remove("selectedItem");
        properties.remove("filter");
        properties.remove("size");
        properties.remove(OPENED_PROPERTY_KEY);
        properties.remove(VALUE_PROPERTY_KEY);
    }

    private Logger getLogger() {
        return LoggerFactory.getLogger(getClass());
    }
}
