package com.vaadin.copilot;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import com.vaadin.copilot.exception.CopilotException;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Rewriter for {@code build.gradle} files that supports dependency, property
 * and repository updates while trying to preserve the original formatting.
 */
public class GradleFileRewriter {
    private static final Pattern DEP_NOTATION_PATTERN = Pattern
            .compile("['\"](?<group>[^:'\"]+):(?<artifact>[^:'\"]+)(?::(?<version>[^'\"\\s]+))?['\"]");
    private static final Pattern DEP_MAP_PATTERN = Pattern.compile(
            "group:\\s*['\"](?<group>[^'\"]+)['\"].*name:\\s*['\"](?<artifact>[^'\"]+)['\"](?:.*version:\\s*['\"](?<version>[^'\"]+)['\"])?.*");
    private static final Pattern URL_PATTERN = Pattern
            .compile("url\\s*(?:=)?\\s*(?:uri\\()?['\"]([^'\"\\)]+)['\"]\\)?");
    private static final Pattern SET_URL_PATTERN = Pattern.compile("setUrl\\(\\s*['\"]([^'\"\\)]+)['\"]\\s*\\)");
    private static final Pattern NAME_PATTERN = Pattern.compile("name\\s*(?:=)?\\s*['\"]([^'\"]+)['\"]");

    private final Path originalFile;
    private final List<String> lines;

    private final List<ReplacementChanges> replacementChanges = new ArrayList<>();
    private final List<AddChanges> addChanges = new ArrayList<>();

    /**
     * Creates a rewriter for the given Gradle build file.
     *
     * @param file
     *            the {@code build.gradle} file to be modified
     * @throws IOException
     *             if the file cannot be read
     */
    public GradleFileRewriter(Path file) throws IOException {
        this.originalFile = file;
        this.lines = Files.readAllLines(file);
    }

    /**
     * Finds dependency under dependencies and dependencyManagement blocks by
     * groupId and artifactId.
     *
     * @param groupId
     *            group id
     * @param artifactId
     *            artifact id
     * @return dependency if found, {@code null} otherwise
     */
    public Dependency findDependencyByGroupIdAndArtifactId(String groupId, String artifactId) {
        return findDependency(groupId, artifactId)
                .orElseGet(() -> findDependencyInDependencyManagement(groupId, artifactId));
    }

    /**
     * Finds the dependency with the given groupId and artifactId inside the
     * dependencies block.
     *
     * @param groupId
     *            group id
     * @param artifactId
     *            artifact id
     * @return optional dependency when found
     */
    public Optional<Dependency> findDependency(String groupId, String artifactId) {
        Block dependenciesBlock = findDependenciesBlock();
        if (dependenciesBlock == null) {
            return Optional.empty();
        }
        Dependency dependency = findDependencyInRange(dependenciesBlock, groupId, artifactId);
        return Optional.ofNullable(dependency);
    }

    /**
     * Checks whether repositories block exists.
     *
     * @return block if found, {@link Optional#empty()} otherwise
     */
    public Optional<Block> getRepositoriesBlock() {
        return Optional.ofNullable(findRepositoriesBlock());
    }

    /**
     * Checks whether buildscript repositories block exists.
     *
     * @return block if found, {@link Optional#empty()} otherwise
     */
    public Optional<Block> getPluginRepositoriesBlock() {
        return Optional.ofNullable(findPluginRepositoriesBlock());
    }

    /**
     * Checks repository exists in repositories block by id and url.
     *
     * @param id
     *            repository name
     * @param url
     *            repository url
     * @return {@code true} if found, {@code false} otherwise
     */
    public boolean hasRepositoryByIdAndUrl(String id, String url) {
        return hasRepositoryByIdAndUrl(id, url, findRepositoriesBlock());
    }

    /**
     * Checks repository exists in repositories block by id.
     *
     * @param id
     *            repository name
     * @return {@code true} if found, {@code false} otherwise
     */
    public boolean hasRepositoryById(String id) {
        return hasRepositoryById(id, findRepositoriesBlock());
    }

    /**
     * Checks repository exists in repositories block by url.
     *
     * @param url
     *            repository url
     * @return {@code true} if found, {@code false} otherwise
     */
    public boolean hasRepositoryByUrl(String url) {
        return hasRepositoryByUrl(url, findRepositoriesBlock());
    }

    /**
     * Adds repository to build.gradle if it does not exist.
     *
     * @param id
     *            repository name
     * @param url
     *            repository url
     */
    public void addRepository(String id, String url) {
        if (hasRepositoryByUrl(url) || hasRepositoryByUrl(toggleTrailingSlash(url))) {
            return;
        }
        Block repositoriesBlock = findRepositoriesBlock();
        if (repositoriesBlock == null) {
            String repositoryBlock = indentText(getRepositoryBlockText(id, url, ""), "    ");
            String textToAdd = """
                    repositories {
                    %s
                    }
                    """.formatted(repositoryBlock);
            addChanges.add(new AddChanges(textToAdd.stripTrailing()));
        } else {
            String indent = leadingWhitespace(lines.get(repositoriesBlock.startLine()));
            String formatted = indentText(getRepositoryBlockText(id, url, ""), indent + "    ");
            addChanges.add(new AddChanges(formatted.stripTrailing(), repositoriesBlock.startLine()));
        }
    }

    /**
     * Checks plugin repository exists in buildscript repositories block by id and
     * url.
     *
     * @param id
     *            repository name
     * @param url
     *            repository url
     * @return {@code true} if found, {@code false} otherwise
     */
    public boolean hasPluginRepositoryByIdAndUrl(String id, String url) {
        return hasRepositoryByIdAndUrl(id, url, findPluginRepositoriesBlock());
    }

    /**
     * Checks plugin repository exists in buildscript repositories block by id.
     *
     * @param id
     *            repository name
     * @return {@code true} if found, {@code false} otherwise
     */
    public boolean hasPluginRepositoryById(String id) {
        return hasRepositoryById(id, findPluginRepositoriesBlock());
    }

    /**
     * Checks plugin repository exists in buildscript repositories block by url.
     *
     * @param url
     *            repository url
     * @return {@code true} if found, {@code false} otherwise
     */
    public boolean hasPluginRepositoryByUrl(String url) {
        return hasRepositoryByUrl(url, findPluginRepositoriesBlock());
    }

    /**
     * Adds plugin repository to build.gradle if it does not exist.
     *
     * @param id
     *            repository name
     * @param url
     *            repository url
     */
    public void addPluginRepository(String id, String url) {
        if (hasPluginRepositoryByUrl(url) || hasPluginRepositoryByUrl(toggleTrailingSlash(url))) {
            return;
        }

        Block buildscriptBlock = findBuildscriptBlock();
        if (buildscriptBlock == null) {
            String repoBlock = indentText(getRepositoryBlockText(id, url, ""), "        ");
            String blockText = """
                    buildscript {
                        repositories {
                    %s
                        }
                    }
                                        """.formatted(repoBlock);
            addChanges.add(new AddChanges(blockText.stripTrailing()));
            return;
        }
        Block pluginRepositoriesBlock = findPluginRepositoriesBlock();
        if (pluginRepositoriesBlock == null) {
            String repoBlock = indentText(getRepositoryBlockText(id, url, ""), "        ");
            String repositoriesBlock = """
                    repositories {
                    %s
                    }
                                        """.formatted(repoBlock);
            String baseIndent = leadingWhitespace(lines.get(buildscriptBlock.startLine()));
            String indentedRepositories = indentText(repositoriesBlock, baseIndent + "    ");
            addChanges.add(new AddChanges(indentedRepositories.stripTrailing(), buildscriptBlock.startLine()));
            return;
        }
        String indent = leadingWhitespace(lines.get(pluginRepositoriesBlock.startLine()));
        String formatted = indentText(getRepositoryBlockText(id, url, ""), indent + "    ");
        addChanges.add(new AddChanges(formatted.stripTrailing(), pluginRepositoriesBlock.startLine()));
    }

    /**
     * Finds the property by key.
     *
     * @param propertyKey
     *            property key
     * @return property if found, {@code null} otherwise
     */
    public Property findPropertyByKey(String propertyKey) {
        Pattern propertyPattern = Pattern
                .compile("\\b" + Pattern.quote(propertyKey) + "\\b\\s*=\\s*['\"]?([^'\"\\s]+[^'\"\\s]*)['\"]?");
        for (int i = 0; i < lines.size(); i++) {
            String line = removeLineComment(lines.get(i)).trim();
            if (line.startsWith("//") || line.startsWith("/*")) {
                continue;
            }
            Matcher matcher = propertyPattern.matcher(line);
            if (matcher.find()) {
                return new Property(this, propertyKey, matcher.group(1), i);
            }
        }
        return null;
    }

    /**
     * Adds dependency to build.gradle if needed.
     *
     * @param groupId
     *            group id
     * @param artifactId
     *            artifact id
     * @param version
     *            dependency version, can be {@code null}
     * @return {@code true} if dependency was added, {@code false} when it already
     *         existed
     */
    public boolean addDependency(String groupId, String artifactId, String version) {
        if (findDependency(groupId, artifactId).isPresent()) {
            return false;
        }
        Block dependenciesBlock = findDependenciesBlock();
        String dependencyLine = version == null ? "implementation '" + groupId + ":" + artifactId + "'"
                : "implementation '" + groupId + ":" + artifactId + ":" + version + "'";

        if (dependenciesBlock == null) {
            String blockText = """
                    dependencies {
                        %s
                    }
                    """.formatted(dependencyLine);
            addChanges.add(new AddChanges(blockText.stripTrailing()));
        } else {
            String indent = leadingWhitespace(lines.get(dependenciesBlock.startLine()));
            String formatted = indent + "    " + dependencyLine;
            addChanges.add(new AddChanges(formatted, dependenciesBlock.startLine()));
        }
        return true;
    }

    /**
     * Updates the dependency version for the provided dependency.
     *
     * @param dependency
     *            dependency to update
     * @param newVersion
     *            new version
     */
    public void updateDependencyVersion(Dependency dependency, String newVersion) {
        if (dependency.version() == null) {
            throw new CopilotException("Dependency " + dependency.groupId() + ":" + dependency.artifactId()
                    + " does not define a version.");
        }
        String previous = dependency.groupId() + ":" + dependency.artifactId() + ":" + dependency.version();
        String replacement = dependency.groupId() + ":" + dependency.artifactId() + ":" + newVersion;
        replacementChanges.add(new ReplacementChanges(previous, replacement, dependency.versionLineNumber()));
    }

    /**
     * Updates the given property value.
     *
     * @param property
     *            property to update
     * @param newValue
     *            new value
     */
    public void updateProperty(Property property, String newValue) {
        replacementChanges.add(new ReplacementChanges(property.propertyValue, newValue, property.lineNumber));
    }

    /**
     * Apply all modifications to a temporary file and then replace the original.
     *
     * @throws IOException
     *             when a file operation fails
     */
    public void save() throws IOException {
        String newFileContent = applyModificationsAndNewFileContent();
        if (newFileContent == null) {
            return;
        }
        Path tempFilePath = null;
        try {
            tempFilePath = Files.createTempFile("build", ".gradle");
            Files.copy(this.originalFile, tempFilePath, StandardCopyOption.REPLACE_EXISTING);
            Files.writeString(tempFilePath, newFileContent);
            Files.copy(tempFilePath, this.originalFile, StandardCopyOption.REPLACE_EXISTING);
        } finally {
            if (tempFilePath != null) {
                try {
                    Files.deleteIfExists(tempFilePath);
                } catch (IOException e) {
                    getLogger().warn("Could not delete temporary file: {}", tempFilePath, e);
                }
            }
        }
    }

    private String applyModificationsAndNewFileContent() throws IOException {
        List<String> updatedLines = new ArrayList<>(lines);
        boolean hasChanges = false;

        for (ReplacementChanges replacementChange : replacementChanges) {
            String line = updatedLines.get(replacementChange.lineNumber);
            if (!line.contains(replacementChange.previousText)) {
                throw new CopilotException(
                        "Could not find " + replacementChange.previousText + " at " + replacementChange.lineNumber);
            }
            line = line.replace(replacementChange.previousText, replacementChange.newText);
            updatedLines.set(replacementChange.lineNumber, line);
            hasChanges = true;
        }
        for (AddChanges addChange : addChanges) {
            if (addChange.lineNumberAfter == -1) {
                updatedLines.add(addChange.newText);
            } else {
                updatedLines.add(addChange.lineNumberAfter + 1, addChange.newText);
            }
            hasChanges = true;
        }

        if (!hasChanges) {
            return null;
        }
        return String.join("\n", updatedLines);
    }

    private Dependency findDependencyInDependencyManagement(String groupId, String artifactId) {
        Block dependencyManagementBlock = findDependencyManagementBlock();
        if (dependencyManagementBlock == null) {
            return null;
        }
        Block importsBlock = findNestedBlock(dependencyManagementBlock, "imports");
        if (importsBlock == null) {
            return null;
        }
        return findDependencyInRange(importsBlock, groupId, artifactId);
    }

    private Dependency findDependencyInRange(Block range, String groupId, String artifactId) {
        for (int i = range.startLine(); i <= range.endLine(); i++) {
            String line = removeLineComment(lines.get(i));
            ParsedDependency parsed = parseDependencyFromLine(line);
            if (parsed == null) {
                continue;
            }
            if (groupId.equals(parsed.groupId()) && artifactId.equals(parsed.artifactId())) {
                return new Dependency(this, parsed.artifactId(), parsed.groupId(), parsed.version(), i);
            }
        }
        return null;
    }

    private ParsedDependency parseDependencyFromLine(String line) {
        Matcher notationMatcher = DEP_NOTATION_PATTERN.matcher(line);
        if (notationMatcher.find()) {
            return new ParsedDependency(notationMatcher.group("artifact"), notationMatcher.group("group"),
                    notationMatcher.group("version"));
        }
        Matcher mapMatcher = DEP_MAP_PATTERN.matcher(line);
        if (mapMatcher.find()) {
            return new ParsedDependency(mapMatcher.group("artifact"), mapMatcher.group("group"),
                    mapMatcher.group("version"));
        }
        return null;
    }

    private Block findRepositoriesBlock() {
        Block buildscriptBlock = findBuildscriptBlock();
        List<Block> excluded = new ArrayList<>();
        if (buildscriptBlock != null) {
            excluded.add(buildscriptBlock);
        }
        return findBlockOutside("repositories", excluded);
    }

    private Block findPluginRepositoriesBlock() {
        Block buildscriptBlock = findBuildscriptBlock();
        if (buildscriptBlock == null) {
            return null;
        }
        return findNestedBlock(buildscriptBlock, "repositories");
    }

    private Block findDependencyManagementBlock() {
        return findBlockOutside("dependencyManagement", List.of());
    }

    private Block findDependenciesBlock() {
        Block buildscriptBlock = findBuildscriptBlock();
        List<Block> excluded = new ArrayList<>();
        if (buildscriptBlock != null) {
            excluded.add(buildscriptBlock);
        }
        return findBlockOutside("dependencies", excluded);
    }

    private Block findBuildscriptBlock() {
        return findBlockOutside("buildscript", List.of());
    }

    private Block findBlockOutside(String blockName, List<Block> excludedRanges) {
        Pattern blockStartPattern = Pattern.compile("^\\s*" + Pattern.quote(blockName) + "\\b.*\\{");
        for (int i = 0; i < lines.size(); i++) {
            if (isInExcludedRange(i, excludedRanges)) {
                continue;
            }
            Matcher matcher = blockStartPattern.matcher(lines.get(i));
            if (matcher.find()) {
                Block block = resolveBlock(i);
                if (block != null) {
                    return block;
                }
            }
        }
        return null;
    }

    private Block findNestedBlock(Block parent, String blockName) {
        Pattern blockStartPattern = Pattern.compile("^\\s*" + Pattern.quote(blockName) + "\\b.*\\{");
        for (int i = parent.startLine() + 1; i <= parent.endLine(); i++) {
            Matcher matcher = blockStartPattern.matcher(lines.get(i));
            if (matcher.find()) {
                Block block = resolveBlock(i);
                if (block != null && block.endLine() <= parent.endLine()) {
                    return block;
                }
            }
        }
        return null;
    }

    private Block resolveBlock(int startLine) {
        int braceDelta = countBraces(lines.get(startLine));
        if (braceDelta == 0 && lines.get(startLine).contains("{")) {
            return new Block(startLine, startLine);
        }
        for (int i = startLine + 1; i < lines.size(); i++) {
            braceDelta += countBraces(lines.get(i));
            if (braceDelta == 0) {
                return new Block(startLine, i);
            }
        }
        return null;
    }

    private boolean isInExcludedRange(int lineNumber, List<Block> excludedRanges) {
        return excludedRanges.stream()
                .anyMatch(block -> lineNumber >= block.startLine() && lineNumber <= block.endLine());
    }

    private boolean hasRepositoryByIdAndUrl(String id, String url, Block repositoriesBlock) {
        if (repositoriesBlock == null) {
            return false;
        }
        for (Repository repository : findRepositories(repositoriesBlock)) {
            if (repository.url != null && normalizeUrl(repository.url).equals(normalizeUrl(url))
                    && id.equals(repository.name)) {
                return true;
            }
        }
        return false;
    }

    private boolean hasRepositoryById(String id, Block repositoriesBlock) {
        if (repositoriesBlock == null) {
            return false;
        }
        for (Repository repository : findRepositories(repositoriesBlock)) {
            if (id.equals(repository.name)) {
                return true;
            }
        }
        return false;
    }

    private boolean hasRepositoryByUrl(String url, Block repositoriesBlock) {
        if (repositoriesBlock == null) {
            return false;
        }
        String normalizedUrl = normalizeUrl(url);
        for (Repository repository : findRepositories(repositoriesBlock)) {
            if (repository.url != null && normalizeUrl(repository.url).equals(normalizedUrl)) {
                return true;
            }
        }
        return false;
    }

    private List<Repository> findRepositories(Block repositoriesBlock) {
        List<Repository> repositories = new ArrayList<>();
        List<Block> mavenBlocks = findMavenBlocks(repositoriesBlock);
        for (Block mavenBlock : mavenBlocks) {
            repositories.add(parseRepository(mavenBlock));
        }
        return repositories;
    }

    private Repository parseRepository(Block mavenBlock) {
        String name = null;
        String url = null;
        for (int i = mavenBlock.startLine(); i <= mavenBlock.endLine(); i++) {
            String line = removeLineComment(lines.get(i));
            if (url == null) {
                Matcher setUrlMatcher = SET_URL_PATTERN.matcher(line);
                if (setUrlMatcher.find()) {
                    url = setUrlMatcher.group(1);
                }
            }
            if (url == null) {
                Matcher urlMatcher = URL_PATTERN.matcher(line);
                if (urlMatcher.find()) {
                    url = urlMatcher.group(1);
                }
            }
            if (name == null) {
                Matcher nameMatcher = NAME_PATTERN.matcher(line);
                if (nameMatcher.find()) {
                    name = nameMatcher.group(1);
                }
            }
        }
        return new Repository(name, url);
    }

    private List<Block> findMavenBlocks(Block repositoriesBlock) {
        List<Block> result = new ArrayList<>();
        Pattern mavenPattern = Pattern.compile("^\\s*maven\\b.*\\{");
        for (int i = repositoriesBlock.startLine() + 1; i <= repositoriesBlock.endLine(); i++) {
            Matcher matcher = mavenPattern.matcher(lines.get(i));
            if (matcher.find()) {
                Block block = resolveBlock(i);
                if (block != null && block.endLine() <= repositoriesBlock.endLine()) {
                    result.add(block);
                    i = block.endLine();
                }
            }
        }
        return result;
    }

    private String removeLineComment(String line) {
        int index = line.indexOf("//");
        while (index != -1) {
            boolean partOfUrl = index > 0 && line.charAt(index - 1) == ':';
            if (!partOfUrl) {
                return line.substring(0, index);
            }
            index = line.indexOf("//", index + 2);
        }
        return line;
    }

    private int countBraces(String line) {
        int count = 0;
        boolean inSingleQuote = false;
        boolean inDoubleQuote = false;
        for (int i = 0; i < line.length(); i++) {
            char c = line.charAt(i);
            if (c == '\\') {
                i++;
                continue;
            }
            if (c == '\'' && !inDoubleQuote) {
                inSingleQuote = !inSingleQuote;
            } else if (c == '\"' && !inSingleQuote) {
                inDoubleQuote = !inDoubleQuote;
            } else if (!inSingleQuote && !inDoubleQuote) {
                if (c == '{') {
                    count++;
                } else if (c == '}') {
                    count--;
                }
            }
        }
        return count;
    }

    private String normalizeUrl(String url) {
        if (url.endsWith("/")) {
            return url.substring(0, url.length() - 1);
        }
        return url;
    }

    private String toggleTrailingSlash(String url) {
        return url.endsWith("/") ? url.substring(0, url.length() - 1) : url + "/";
    }

    private String leadingWhitespace(String line) {
        int i = 0;
        while (i < line.length() && Character.isWhitespace(line.charAt(i))) {
            i++;
        }
        return line.substring(0, i);
    }

    private String indentText(String text, String indent) {
        return text.lines().map(line -> indent + line).collect(Collectors.joining("\n"));
    }

    private String getRepositoryBlockText(String id, String url, String indent) {
        StringBuilder builder = new StringBuilder();
        builder.append(indent).append("maven {\n");
        if (id != null && !id.isBlank()) {
            builder.append(indent).append("    name = \"").append(id).append("\"\n");
        }
        builder.append(indent).append("    url \"").append(url).append("\"\n");
        builder.append(indent).append("}");
        return builder.toString();
    }

    private Logger getLogger() {
        return LoggerFactory.getLogger(getClass());
    }

    public record Property(GradleFileRewriter fileRewriter, String propertyKey, String propertyValue, int lineNumber) {
    }

    public record Dependency(GradleFileRewriter fileRewriter, String artifactId, String groupId, String version,
            int versionLineNumber) {
    }

    public record Block(int startLine, int endLine) {
    }

    private record Repository(String name, String url) {
    }

    private record ParsedDependency(String artifactId, String groupId, String version) {
    }

    private record ReplacementChanges(String previousText, String newText, int lineNumber) {
    }

    private record AddChanges(String newText, int lineNumberAfter) {
        public AddChanges(String newText) {
            this(newText, -1);
        }
    }
}
