package com.vaadin.copilot.plugins.propertypanel;

import java.beans.IntrospectionException;
import java.beans.PropertyDescriptor;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.List;

import com.vaadin.copilot.ComponentPropertyType;
import com.vaadin.copilot.ComponentSourceFinder;
import com.vaadin.copilot.exception.ComponentCreatedInLoopException;
import com.vaadin.copilot.exception.CopilotException;
import com.vaadin.copilot.exception.UnknownExpressionTypeException;
import com.vaadin.copilot.javarewriter.ComponentInfo;
import com.vaadin.copilot.javarewriter.ComponentInfoFinder;
import com.vaadin.copilot.javarewriter.JavaFileSourceProvider;
import com.vaadin.copilot.javarewriter.JavaRewriter;
import com.vaadin.flow.component.Component;
import com.vaadin.flow.internal.BeanUtil;
import com.vaadin.flow.shared.util.SharedUtil;

import com.github.javaparser.ast.Node;
import com.github.javaparser.ast.expr.NullLiteralExpr;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ComponentPropertyProvider {
    private final ComponentSourceFinder componentSourceFinder;
    private final JavaRewriter javaRewriter = new JavaRewriter();

    public ComponentPropertyProvider(ComponentSourceFinder componentSourceFinder) {
        this.componentSourceFinder = componentSourceFinder;
    }

    public List<ComponentProperty> getProperties(Component component) throws IOException {
        JavaFileSourceProvider javaFileSourceProvider = new JavaFileSourceProvider();
        ComponentInfoFinder finder = new ComponentInfoFinder(javaFileSourceProvider,
                componentSourceFinder.findTypeAndSourceLocation(component, false),
                componentSourceFinder::getSiblingsTypeAndSourceLocations);
        ComponentInfo componentInfo = finder.find();
        if (componentInfo.createdInLoop()) {
            throw new ComponentCreatedInLoopException();
        }
        Class<? extends Component> componentClass = component.getClass();
        List<PropertyDescriptor> beanPropertyDescriptors;
        try {
            beanPropertyDescriptors = BeanUtil.getBeanPropertyDescriptors(componentClass);
        } catch (IntrospectionException e) {
            throw new CopilotException("Could not get property descriptors for " + componentClass.getName(), e);
        }
        return beanPropertyDescriptors.stream()
                .filter(f -> ComponentPropertyHelperUtil.getPropertyType(f.getPropertyType()) != null)
                .filter(propertyDescriptor -> propertyDescriptor.getReadMethod() != null
                        && propertyDescriptor.getWriteMethod() != null)
                .map(propertyDescriptor -> this.map(componentInfo, component, propertyDescriptor))
                .sorted(Comparator.comparing(ComponentProperty::getLabel)).toList();
    }

    private ComponentProperty map(ComponentInfo componentInfo, Component component,
            PropertyDescriptor propertyDescriptor) {
        ComponentProperty componentProperty = new ComponentProperty(propertyDescriptor.getName());
        componentProperty.setLabel(SharedUtil.camelCaseToHumanFriendly(propertyDescriptor.getDisplayName()));
        ComponentPropertyHelperUtil.PropertyFieldInfo propertyFieldInfo = ComponentPropertyHelperUtil
                .getPropertyType(propertyDescriptor.getPropertyType());
        componentProperty.setType(propertyFieldInfo.type());
        componentProperty.setQualifiedClassName(propertyFieldInfo.qualifiedClassName());
        componentProperty.setMultiSelection(propertyFieldInfo.multipleSelection());
        componentProperty.setOptions(getOptions(propertyFieldInfo.type(), propertyDescriptor));

        try {
            FlowComponentDefaultValueCache.getInstance()
                    .getDefaultPropertyValue(component, propertyDescriptor.getName())
                    .ifPresent(componentProperty::setDefaultValue);
        } catch (Exception e) {
            componentProperty.setExceptionOnGettingDefaultValue(true);
            getLogger().trace("Unable to retrieve default value for {}", propertyDescriptor.getName());
        }
        try {
            Object propertyValue = javaRewriter.getPropertyValue(componentInfo, componentProperty.getPropertyName());
            if (propertyValue instanceof NullLiteralExpr) {
                componentProperty.setValue(null);
            } else if (propertyValue instanceof Node node) {
                componentProperty.setValue(node.toString());
            } else {
                componentProperty.setValue(propertyValue);
            }
        } catch (UnknownExpressionTypeException e) {
            getLogger().trace("Unable to retrieve value for {}", propertyDescriptor.getName());
            componentProperty.setExceptionOnGettingValue(true);
        }

        return componentProperty;
    }

    private List<ComponentProperty.ComponentPropertyOption> getOptions(ComponentPropertyType type,
            PropertyDescriptor propertyDescriptor) {
        if (!ComponentPropertyType.ENUM.equals(type)) {
            return new ArrayList<>();
        }
        Class<?> propertyType = propertyDescriptor.getPropertyType();
        Class<?> targetEnumClass = propertyType;
        if (propertyType.isArray()) {
            targetEnumClass = propertyType.getComponentType();
        }
        Object[] enumConstants = targetEnumClass.getEnumConstants();
        return Arrays.stream(enumConstants)
                .map(constant -> new ComponentProperty.ComponentPropertyOption(constant.toString(), constant)).toList();
    }

    private Logger getLogger() {
        return LoggerFactory.getLogger(getClass());
    }
}
