package com.vaadin.copilot.plugins.themeeditor;

import java.io.File;
import java.io.IOException;
import java.io.StringWriter;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.stream.Collectors;

import com.vaadin.copilot.ProjectFileManager;
import com.vaadin.copilot.exception.CopilotException;
import com.vaadin.copilot.ide.CopilotIDEPlugin;
import com.vaadin.copilot.plugins.themeeditor.utils.CssRule;
import com.vaadin.copilot.plugins.themeeditor.utils.ThemeEditorException;

import com.helger.css.decl.CSSDeclaration;
import com.helger.css.decl.CSSImportRule;
import com.helger.css.decl.CSSSelector;
import com.helger.css.decl.CSSSelectorSimpleMember;
import com.helger.css.decl.CSSStyleRule;
import com.helger.css.decl.CascadingStyleSheet;
import com.helger.css.decl.ICSSSelectorMember;
import com.helger.css.reader.CSSReader;
import com.helger.css.writer.CSSWriter;

public class ThemeModifier {

    private static final String COPILOT_THEME_MODIFY_UNDO_LABEL = CopilotIDEPlugin.undoLabel("Theme Modify");

    private static final String COPILOT_STYLES_MODIFY_UNDO_LABEL = CopilotIDEPlugin.undoLabel("Styles Modify");

    public static final String THEME_EDITOR_CSS = "theme-editor.css";

    private static final String HEADER_TEXT = "This file has been created by the Vaadin Theme Editor. Please note that\n"
            + "manual changes to individual CSS properties may be overwritten by the theme editor.";

    private static final String DEFAULT_FONT_NAME = "Default";

    private static final String FONT_CDN_URL = "https://cdn.jsdelivr.net/fontsource/css/%s@latest/index.css";

    private final File themeFolder;

    private final File themeEditorStyles;

    private boolean importPresent;

    /**
     * Constructor
     *
     * @throws ThemeEditorException
     *             in not present
     */
    public ThemeModifier() throws ThemeEditorException {
        this.themeFolder = getThemeFolder();
        this.themeEditorStyles = new File(themeFolder, getCssFileName());
    }

    /**
     * Performs update of CSS file setting (adding or updating) given
     * {@link CssRule}.
     *
     * @param rules
     *            list of {@link CssRule} to be added or updated
     */
    public void setThemeProperties(List<CssRule> rules) {
        CascadingStyleSheet styleSheet = getCascadingStyleSheet();
        for (CssRule rule : rules) {
            for (Map.Entry<String, String> property : rule.getProperties().entrySet()) {
                if (property.getValue() != null && !property.getValue().isBlank()) {
                    setCssProperty(styleSheet, rule.getSelector(), property.getKey(), property.getValue());
                } else {
                    removeCssProperty(styleSheet, rule.getSelector(), property.getKey());
                }
            }
        }
        sortStyleSheet(styleSheet);
        writeStyleSheet(styleSheet);
        insertImportIfNotExists();
    }

    /**
     * Adds an import rule for the specified font to the theme editor CSS file and
     * updates the Lumo font family theme property. Removes all previous import
     * rules.
     *
     * @param fontName
     *            the font to use
     */
    public void setFont(String fontName) {
        CascadingStyleSheet styleSheet = getCascadingStyleSheet();

        // Remove previous import rule
        styleSheet.removeAllImportRules();

        if (DEFAULT_FONT_NAME.equals(fontName)) {
            // Reset to Lumo default font
            // Only need to clear the font family property
            removeCssProperty(styleSheet, "html", "--lumo-font-family");
        } else {
            // Add import rule for custom font
            String fontPackageName = getFontPackageName(fontName);
            String fontCdnUrl = String.format(FONT_CDN_URL, fontPackageName);
            CSSImportRule importRule = new CSSImportRule(fontCdnUrl);
            styleSheet.addImportRule(importRule);

            // Update Lumo font family theme property
            setCssProperty(styleSheet, "html", "--lumo-font-family", fontName);
        }

        sortStyleSheet(styleSheet);
        writeStyleSheet(styleSheet);
    }

    private String getFontPackageName(String fontName) {
        if (fontName == null) {
            return null;
        }
        return fontName.replace(" ", "-").toLowerCase(Locale.ENGLISH);
    }

    /**
     * Returns the content of the theme editor CSS file.
     *
     * @return CSS string
     */
    public String getCss() {
        if (noStyleSheet()) {
            return "";
        }
        try {
            StringWriter stringWriter = new StringWriter();
            CascadingStyleSheet styleSheet = getCascadingStyleSheet();
            CSSWriter cssWriter = new CSSWriter().setWriteHeaderText(false);
            cssWriter.getSettings().setOptimizedOutput(true).setRemoveUnnecessaryCode(true);
            cssWriter.writeCSS(styleSheet, stringWriter);
            return stringWriter.toString();
        } catch (IOException e) {
            throw new ThemeEditorException("Cannot read stylesheet file.");
        }
    }

    /**
     * Retrieves list of {@link CssRule} for given selectors.
     *
     * @param selectors
     *            list of selectors
     * @return list of {@link CssRule}
     */
    public List<CssRule> getCssRules(List<String> selectors) {
        if (noStyleSheet()) {
            return Collections.emptyList();
        }
        List<CSSSelector> cssSelectors = selectors.stream().map(this::parseSelector).toList();
        CascadingStyleSheet styleSheet = getCascadingStyleSheet();
        return styleSheet.getAllStyleRules().stream().filter(rule -> rule.getSelectorCount() > 0)
                .filter(rule -> cssSelectors.contains(rule.getSelectorAtIndex(0))).map(this::toCssRule).toList();
    }

    /**
     * Replaces classname with new classname in all matching rules.
     *
     * @param oldClassName
     *            classname to be replaced
     * @param newClassName
     *            new classname
     */
    public void replaceClassName(String tagName, String oldClassName, String newClassName) {
        if (noStyleSheet()) {
            return;
        }
        CascadingStyleSheet styleSheet = getCascadingStyleSheet();
        replaceClassName(styleSheet, tagName, oldClassName, newClassName);
        writeStyleSheet(styleSheet);
    }

    /**
     * Gets location line of rule with given selector
     *
     * @param selectorString
     * @return line number when located, -1 otherwise
     */
    public int getRuleLocationLine(String selectorString) {
        if (noStyleSheet()) {
            return -1;
        }
        CascadingStyleSheet styleSheet = getCascadingStyleSheet();
        CSSSelector selector = parseSelector(selectorString);
        CSSStyleRule rule = findRuleBySelector(styleSheet, selector);
        if (rule == null) {
            return -1;
        }
        return rule.getSourceLocation().getFirstTokenBeginLineNumber();
    }

    /**
     * Creates empty rule with given selector
     *
     * @param selector
     *            Selector to add rule to stylesheet. Null means create the file if
     *            not exists
     */
    public void createEmptyStyleRule(String selector) {
        CascadingStyleSheet styleSheet = getCascadingStyleSheet();
        if (selector != null) {
            CSSSelector cssSelector = new CSSSelector().addMember(new CSSSelectorSimpleMember(selector));
            CSSStyleRule cssStyleRule = new CSSStyleRule().addSelector(cssSelector);
            styleSheet.addRule(cssStyleRule);
            sortStyleSheet(styleSheet);
        }
        writeStyleSheet(styleSheet);
        insertImportIfNotExists();
    }

    /**
     * Returns selectors in the theme editors
     *
     * @return list of class names
     */
    public List<String> getExistingClassNames() {
        if (noStyleSheet()) {
            return Collections.emptyList();
        }
        CascadingStyleSheet styleSheet = getCascadingStyleSheet();
        return styleSheet.getAllStyleRules().stream().map(this::toCssRule).map(CssRule::getSelector)
                .map(k -> k.split("::")[0]).toList();
    }

    protected boolean noStyleSheet() {
        return !themeEditorStyles.exists();
    }

    protected String getCssFileName() {
        return THEME_EDITOR_CSS;
    }

    protected String getHeaderText() {
        return HEADER_TEXT;
    }

    protected File getThemeFolder() {
        return getProjectFileManager().getThemeFolder();
    }

    public File getStyleSheetFile() {
        return themeEditorStyles;
    }

    protected CascadingStyleSheet getCascadingStyleSheet() {
        if (noStyleSheet()) {
            return new CascadingStyleSheet();
        }
        try {
            String content = getProjectFileManager().readFile(themeEditorStyles);
            CascadingStyleSheet styleSheet = CSSReader.readFromString(content, StandardCharsets.UTF_8);

            if (!importPresent) {
                insertImportIfNotExists();
                importPresent = true;
            }

            return styleSheet;
        } catch (IOException ex) {
            throw new CopilotException("Cannot read file " + themeEditorStyles.getPath());
        }
    }

    protected void setCssProperty(CascadingStyleSheet styleSheet, String selector, String property, String value) {
        CSSStyleRule newRule = createStyleRule(selector, property, value);
        CSSStyleRule existingRule = findRuleBySelector(styleSheet, newRule);
        if (existingRule == null) {
            styleSheet.addRule(newRule);
        } else {
            CSSDeclaration newDeclaration = newRule.getDeclarationAtIndex(0);
            CSSDeclaration existingDeclaration = existingRule.getDeclarationOfPropertyName(property);
            if (existingDeclaration == null) {
                existingRule.addDeclaration(newDeclaration);
            } else {
                // rule with given selector, property and value exists -> save
                // for undo
                existingDeclaration.setExpression(newDeclaration.getExpression());
            }
        }
    }

    protected void removeCssProperty(CascadingStyleSheet styleSheet, String selector, String property) {
        // value not considered
        CSSStyleRule newRule = createStyleRule(selector, property, "none");
        CSSStyleRule existingRule = findRuleBySelector(styleSheet, newRule);
        if (existingRule != null) {
            removeProperty(existingRule, newRule);
            if (existingRule.getDeclarationCount() == 0) {
                styleSheet.removeRule(existingRule);
            }
        }
    }

    protected void writeStyleSheet(CascadingStyleSheet styleSheet) {
        try {
            CSSWriter writer = new CSSWriter().setWriteHeaderText(true).setHeaderText(getHeaderText());
            writer.getSettings().setOptimizedOutput(false);
            StringWriter stringWriter = new StringWriter();
            writer.writeCSS(styleSheet, stringWriter);
            getProjectFileManager().writeFile(themeEditorStyles, COPILOT_THEME_MODIFY_UNDO_LABEL,
                    stringWriter.toString());
        } catch (IOException e) {
            throw new ThemeEditorException("Cannot write " + themeEditorStyles.getPath(), e);
        }
    }

    protected void sortStyleSheet(CascadingStyleSheet styleSheet) {
        List<CSSStyleRule> sortedRules = styleSheet.getAllStyleRules().stream()
                .sorted(Comparator.comparing(r -> r.getSelectorAtIndex(0).getAsCSSString()))
                .collect(Collectors.toList());
        for (CSSStyleRule rule : sortedRules) {
            List<CSSDeclaration> sortedDeclarations = rule.getAllDeclarations().stream()
                    .sorted(Comparator.comparing(CSSDeclaration::getAsCSSString)).collect(Collectors.toList());
            rule.removeAllDeclarations();
            sortedDeclarations.forEach(rule::addDeclaration);
        }
        sortedRules.forEach(styleSheet::removeRule);
        sortedRules.forEach(styleSheet::addRule);
    }

    protected CSSStyleRule createStyleRule(String selector, String property, String value) {
        return CSSReader.readFromString(selector + "{" + property + ": " + value + "}", StandardCharsets.UTF_8)
                .getStyleRuleAtIndex(0);
    }

    protected void removeProperty(CSSStyleRule existingRule, CSSStyleRule newRule) {
        CSSDeclaration newDeclaration = newRule.getDeclarationAtIndex(0);
        String property = newDeclaration.getProperty();
        CSSDeclaration declaration = existingRule.getDeclarationOfPropertyName(property);
        if (declaration != null) {
            existingRule.removeDeclaration(declaration);
        }
    }

    protected CSSStyleRule findRuleBySelector(CascadingStyleSheet styleSheet, CSSStyleRule rule) {
        return styleSheet.getAllStyleRules().stream()
                .filter(r -> r.getAllSelectors().containsAll(rule.getAllSelectors())).findFirst().orElse(null);
    }

    protected CSSStyleRule findRuleBySelector(CascadingStyleSheet styleSheet, CSSSelector selector) {
        return styleSheet.getAllStyleRules().stream().filter(r -> r.getAllSelectors().contains(selector)).findFirst()
                .orElse(null);
    }

    protected void replaceClassName(CascadingStyleSheet styleSheet, String tagName, String oldClassName,
            String newClassName) {
        String dotOldClassName = "." + oldClassName;
        String dotNewClassName = "." + newClassName;
        for (CSSStyleRule rule : styleSheet.getAllStyleRules()) {
            for (CSSSelector selector : rule.getAllSelectors()) {
                if (selector.getAllMembers().containsNone(m -> tagName.equals(m.getAsCSSString()))) {
                    continue;
                }
                List<ICSSSelectorMember> members = new ArrayList<>();
                selector.getAllMembers().findAll(m -> dotOldClassName.equals(m.getAsCSSString()), members::add);
                members.forEach(m -> {
                    int index = selector.getAllMembers().indexOf(m);
                    selector.removeMember(m);
                    selector.addMember(index, new CSSSelectorSimpleMember(dotNewClassName));
                });
            }
        }
    }

    protected void insertImportIfNotExists() {
        File themeStyles = getProjectFileManager().getStylesCss();
        CascadingStyleSheet styleSheet = CSSReader.readFromFile(themeStyles, StandardCharsets.UTF_8);

        CSSImportRule expectedRule = new CSSImportRule(getCssFileName());
        if (!styleSheet.getAllImportRules().contains(expectedRule)) {
            try {
                List<String> lines = getProjectFileManager().readLines(themeStyles);
                lines.add(0, "@import \"" + getCssFileName() + "\";");
                String content = lines.stream().collect(Collectors.joining(System.lineSeparator()));
                getProjectFileManager().writeFile(themeStyles, COPILOT_STYLES_MODIFY_UNDO_LABEL, content);
            } catch (IOException e) {
                throw new ThemeEditorException("Cannot insert theme-editor.css @import", e);
            }
        }
    }

    private ProjectFileManager getProjectFileManager() {
        return ProjectFileManager.get();
    }

    protected CssRule toCssRule(CSSStyleRule rule) {
        CSSSelector selector = rule.getSelectorAtIndex(0);
        Map<String, String> properties = new HashMap<>();
        rule.getAllDeclarations().forEach(cssDeclaration -> properties.put(cssDeclaration.getProperty(),
                cssDeclaration.getExpressionAsCSSString()));
        return new CssRule(selector.getAsCSSString(), properties);
    }

    protected CSSSelector parseSelector(String selector) {
        CascadingStyleSheet css = CSSReader.readFromString(selector + "{}");
        return css.getAllStyleRules().getFirst().getSelectorAtIndex(0);
    }
}
