package com.vaadin.copilot;

import java.nio.charset.StandardCharsets;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import com.vaadin.flow.component.Tag;

import com.helger.css.decl.CSSMediaRule;
import com.helger.css.decl.CSSSelector;
import com.helger.css.decl.CSSSelectorSimpleMember;
import com.helger.css.decl.CSSStyleRule;
import com.helger.css.decl.CSSSupportsRule;
import com.helger.css.decl.CascadingStyleSheet;
import com.helger.css.reader.CSSReader;
import com.helger.css.writer.CSSWriter;
import org.apache.commons.lang3.StringUtils;

public class LitTemplateCssUtil {

    /**
     * Fetches the tag name from the @Tag annotation on given class or its
     * superclasses.
     *
     * @param clazz
     *            class to check for @Tag annotation
     * @return tag name or null if not found
     */
    static String getTagName(Class<?> clazz) {
        while (clazz != null) {
            Tag tagAnnotation = clazz.getDeclaredAnnotation(Tag.class);
            if (tagAnnotation != null) {
                return tagAnnotation.value();
            }
            clazz = clazz.getSuperclass();
        }
        return null;
    }

    /**
     * Creates a css class name based on the component @Tag annotation value for use
     * in lit template style conversion. E.g. @Tag("my-component") ->
     * "my-component-converted"
     *
     * @param clazz
     *            class to check for @Tag annotation
     * @return css class name to use in style conversion
     */
    static String getConvertedCssClassName(Class<?> clazz) {
        String tagName = getTagName(clazz);
        return tagName == null ? null : tagName + "-converted";
    }

    static String transformHostToCssClass(String css, String cssClassName) {
        if (StringUtils.isBlank(css)) {
            return css;
        }

        String classSelector = "." + cssClassName;

        // Handle :host selectors
        String transformedCss = null;
        try {
            transformedCss = transformHostSelectors(css, classSelector);
        } catch (Exception e) {
            // Fallback to original css
            return css;
        }

        try {
            // Scope remaining selectors
            CascadingStyleSheet styleSheet = CSSReader.readFromString(transformedCss, StandardCharsets.UTF_8);

            for (CSSStyleRule rule : styleSheet.getAllStyleRules()) {
                scopeSelectors(rule, classSelector);
            }
            for (CSSMediaRule mediaRule : styleSheet.getAllMediaRules()) {
                for (CSSStyleRule rule : mediaRule.getAllStyleRules()) {
                    scopeSelectors(rule, classSelector);
                }
            }
            for (CSSSupportsRule supportsRule : styleSheet.getAllSupportsRules()) {
                for (CSSStyleRule rule : supportsRule.getAllStyleRules()) {
                    scopeSelectors(rule, classSelector);
                }
            }

            return new CSSWriter().setWriteHeaderText(false).getCSSAsString(styleSheet);
        } catch (Exception e) {
            // Fallback to :host transformed css
            return transformedCss;
        }
    }

    private static String transformHostSelectors(String css, String classSelector) {
        // Replace :host followed by rule in parentheses with class selector
        Pattern hostWithParens = Pattern.compile(":host\\(([^)]+)\\)");
        Matcher matcher = hostWithParens.matcher(css);
        StringBuilder sb = new StringBuilder();
        while (matcher.find()) {
            String innerSelector = matcher.group(1);
            matcher.appendReplacement(sb, Matcher.quoteReplacement(classSelector + innerSelector));
        }
        matcher.appendTail(sb);

        // Replace :host without parentheses with class selector
        return sb.toString().replace(":host", classSelector);
    }

    private static void scopeSelectors(CSSStyleRule rule, String className) {
        for (CSSSelector selector : rule.getAllSelectors()) {
            if (selector.hasMembers()) {
                if (!selector.getMemberAtIndex(0).getAsCSSString().startsWith(className)) {
                    CSSSelectorSimpleMember scopeMember = new CSSSelectorSimpleMember(className + " ");
                    selector.addMember(0, scopeMember);
                }
            }
        }
    }
}
