package com.vaadin.copilot;

import java.util.List;
import java.util.function.Consumer;

import com.vaadin.base.devserver.DevToolsInterface;
import com.vaadin.copilot.ide.IDEHeartbeatResponse;
import com.vaadin.copilot.ide.IDEHeartbeatScheduler;
import com.vaadin.flow.internal.JacksonUtils;

import tools.jackson.databind.node.ObjectNode;

/**
 * A consumer that listens to IDE heartbeat responses to detect and react to
 * changes in compilation status.
 * <p>
 * If the compilation status changes (i.e., from no errors to errors or vice
 * versa), it sends a message to the developer tools interface indicating the
 * new status and optionally includes the list of files with compilation errors.
 */

public class CompilationStatusChecker implements Consumer<IDEHeartbeatResponse> {
    private DevToolsInterface devToolsInterface;
    private boolean previousCompilationResultHadError = false;
    private boolean subscribed = false;

    /**
     * Subscribes this checker to the IDE heartbeat scheduler to start receiving
     * compilation status updates if not subscribed. Also, updates
     * {@link DevToolsInterface} reference so it stays up-to-date This method is
     * synchronized to ensure thread safety.
     *
     * @param devToolsInterface
     *            the interface to use for sending updates
     */
    public synchronized void subscribe(DevToolsInterface devToolsInterface) {
        this.devToolsInterface = devToolsInterface;
        if (!subscribed) {
            IDEHeartbeatScheduler.getInstance().addConsumer(this);
            subscribed = true;
        }
    }

    /**
     * Accepts an {@link IDEHeartbeatResponse} and checks for changes in compilation
     * error status. If a change is detected, sends a notification with the new
     * status and the list of files with errors (if available).
     *
     * @param ideHeartbeatResponse
     *            the heartbeat response containing compilation status
     */
    @Override
    public void accept(IDEHeartbeatResponse ideHeartbeatResponse) {
        Boolean compilationErrorFromIDEPlugin = ideHeartbeatResponse.hasCompilationError();
        if (compilationErrorFromIDEPlugin == null) {
            return;
        }
        if (previousCompilationResultHadError == compilationErrorFromIDEPlugin) {
            return;
        }
        previousCompilationResultHadError = compilationErrorFromIDEPlugin;

        ObjectNode resp = JacksonUtils.createObjectNode();
        resp.put("error", compilationErrorFromIDEPlugin);
        if (ideHeartbeatResponse.filesContainCompilationErrors() != null) {
            List<FilePathAndName> list = ideHeartbeatResponse.filesContainCompilationErrors().stream()
                    .map(filePath -> new FilePathAndName(filePath, Util.getFileName(filePath))).toList();
            resp.set("files", JacksonUtils.listToJson(list));
        }
        devToolsInterface.send(Copilot.PREFIX + "project-compilation-error", resp);

    }

    /**
     * A simple record to represent a file path and its corresponding file name.
     *
     * @param path
     *            the full file path
     * @param name
     *            the name extracted from the file path
     */
    public record FilePathAndName(String path, String name) {
    }
}
