package com.vaadin.copilot.exception.report;

import com.vaadin.copilot.FlowUtil;
import com.vaadin.flow.component.Component;

/**
 * Represents a reference to a relevant UI component node within the context of
 * an exception report.
 * <p>
 * Stores metadata about the component's relevance, its node ID, and the UI ID
 * it belongs to. This information helps identify and locate the component in
 * the UI hierarchy where the issue occurred.
 */

public class ExceptionReportRelevantComponentNode {
    private String relevance;
    private Integer nodeId;
    private Integer uiId;

    /**
     * Default constructor for {@code ExceptionReportRelevantComponentNode}.
     * <p>
     * Creates an empty instance, typically used for deserialization or manual field
     * population.
     */
    public ExceptionReportRelevantComponentNode() {

    }

    /**
     * Constructs a new {@code ExceptionReportRelevantComponentNode} with the given
     * relevance label and the target {@link Component}.
     * <p>
     * Extracts and stores the component's node ID and UI ID. Throws an exception if
     * the component is {@code null}.
     *
     * @param relevance
     *            a string describing the relevance of the component to the
     *            exception
     * @param component
     *            the UI component considered relevant; must not be {@code null}
     * @throws IllegalArgumentException
     *             if {@code component} is {@code null}
     */
    public ExceptionReportRelevantComponentNode(String relevance, Component component) {
        this.relevance = relevance;
        if (component == null) {
            throw new IllegalArgumentException("Component cannot be null");
        }
        this.nodeId = FlowUtil.getNodeId(component);
        component.getUI().ifPresent(ui -> this.uiId = ui.getUIId());
    }

    public String getRelevance() {
        return relevance;
    }

    public void setRelevance(String relevance) {
        this.relevance = relevance;
    }

    public Integer getNodeId() {
        return nodeId;
    }

    public void setNodeId(Integer nodeId) {
        this.nodeId = nodeId;
    }

    public Integer getUiId() {
        return uiId;
    }

    public void setUiId(Integer uiId) {
        this.uiId = uiId;
    }
}
