package com.vaadin.copilot;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.List;

import com.vaadin.copilot.exception.CopilotException;
import com.vaadin.flow.internal.StringUtil;
import com.vaadin.flow.server.VaadinContext;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.xml.sax.SAXException;

public class ProjectModifier {

    public record RouteAndPath(File file, String path) {

    }

    private static Logger getLogger() {
        return LoggerFactory.getLogger(ProjectModifier.class);
    }

    public static void addSpringSecurity(VaadinContext context, Path fileInModule,
            AccessRequirement defaultViewAndLayoutAccess, Framework loginViewType, List<RouteAndPath> serverRoutes,
            List<RouteAndPath> layouts) {
        ProjectFileManager projectFileManager = ProjectFileManager.get();
        if (!SpringBridge.isSpringAvailable(context)) {
            throw new CopilotException("The project needs to be Spring based to add Spring Security.");
        } else if (SpringBridge.isSpringSecurityEnabled(context)) {
            throw new CopilotException("Spring Security is already enabled for the project.");
        } else {
            try {
                JavaSourcePathDetector.ModuleInfo module = projectFileManager.findModule(fileInModule.toFile())
                        .orElseThrow(() -> new IllegalArgumentException("No module found for file: " + fileInModule));
                Path javaSourcePath = module.javaSourcePaths().get(0);
                Path frontendPath = module.frontendPath();

                // Add the dependency
                Path pomXml = module.rootPath().resolve("pom.xml");
                if (!Files.exists(pomXml)) {
                    throw new CopilotException(
                            "Only Maven projects are supported. There is no pom.xml at " + pomXml.toString());
                }
                PomFileRewriter pomFileRewriter = new PomFileRewriter(pomXml);
                if (pomFileRewriter.addDependency("org.springframework.boot", "spring-boot-starter-security", null)) {
                    pomFileRewriter.save();
                }

                // Add the Spring Security configuration classes

                String basePackage = Util.getPackageName(SpringBridge.getApplicationClassName(context));
                String securityPackage = basePackage + ".security";
                String loginViewPackage = basePackage + ".views";

                Path securityConfigurationJava = javaSourcePath.resolve(securityPackage.replace(".", File.separator))
                        .resolve("SecurityConfiguration.java");
                String securityConfiguration = getSecurityResource("SecurityConfiguration.java");
                securityConfiguration = addPackage(securityConfiguration, securityPackage);
                projectFileManager.writeFile(securityConfigurationJava, "Create security configuration",
                        securityConfiguration);

                // For now, always uses in memory credentials
                Path userDetailsServiceJava = javaSourcePath.resolve(securityPackage.replace(".", File.separator))
                        .resolve("AppUserDetailsService.java");
                String userDetailsService = getSecurityResource("UserDetailsServiceInMemory.java");
                userDetailsService = addPackage(userDetailsService, securityPackage);

                projectFileManager.writeFile(userDetailsServiceJava, "Create user details service", userDetailsService);

                // Add a Login view
                if (loginViewType == Framework.HILLA) {
                    Path loginViewTsx = frontendPath.resolve("views").resolve("login.tsx");
                    projectFileManager.writeFile(loginViewTsx, "", getSecurityResource("login.tsx"));
                } else {
                    Path loginViewJava = javaSourcePath.resolve(loginViewPackage.replace(".", File.separator))
                            .resolve("LoginView.java");

                    String loginViewSource = getSecurityResource("LoginView.java");
                    loginViewSource = addPackage(loginViewSource, loginViewPackage);

                    projectFileManager.writeFile(loginViewJava, "Create login view", loginViewSource);
                }

                // Modify access level for existing Java views and @Layout layouts
                serverRoutes.forEach(routeAndPath -> {
                    try {
                        if (RouteHandler.getRouteAccessControl(routeAndPath.file(), routeAndPath.path()).isEmpty()) {
                            RouteHandler.updateRouteAccessControl(routeAndPath.file(), routeAndPath.path(),
                                    defaultViewAndLayoutAccess);
                        }
                    } catch (IllegalArgumentException e) {
                        getLogger().debug("Ignoring route which is not in the project", e);
                    } catch (IOException e) {
                        getLogger().error("Unable to update route access control for " + routeAndPath.file(), e);
                    }
                });
                layouts.forEach(layout -> {
                    try {
                        if (RouteHandler.getLayoutAccessControl(layout.file()).isEmpty()) {
                            RouteHandler.updateLayoutAccessControl(layout.file(), defaultViewAndLayoutAccess);
                        }
                    } catch (IllegalArgumentException e) {
                        getLogger().debug("Ignoring route which is not in the project", e);
                    } catch (IOException e) {
                        getLogger().error("Unable to update route access control for " + layout.file(), e);
                    }
                });

            } catch (IOException | SAXException e) {
                throw new CopilotException("Unable to make project modifications", e);
            }
        }
    }

    private static String addPackage(String javaSource, String packageName) {
        String packageDefinition = "package " + packageName + ";";
        if (javaSource.startsWith("package ")) {
            return javaSource.replaceFirst("package .*;", packageDefinition);
        } else {
            return packageDefinition + "\n" + javaSource;
        }
    }

    private static String getSecurityResource(String filename) throws IOException {
        try (InputStream stream = ProjectModifier.class
                .getResourceAsStream("/com/vaadin/copilot/securityconfig/" + filename)) {
            if (stream == null) {
                throw new IOException("Unable to find security resource: " + filename);
            }
            return StringUtil.toUTF8String(stream);
        }
    }
}
