package com.vaadin.copilot.javarewriter;

import java.io.File;
import java.io.IOException;
import java.lang.reflect.Constructor;
import java.lang.reflect.Method;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.text.Normalizer;
import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.ZoneOffset;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeParseException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.vaadin.copilot.IdentityHashSet;
import com.vaadin.copilot.JavaReflectionUtil;
import com.vaadin.copilot.ProjectFileManager;
import com.vaadin.copilot.UIServiceCreator;
import com.vaadin.copilot.Util;
import com.vaadin.copilot.exception.CopilotException;
import com.vaadin.copilot.exception.UnknownExpressionTypeException;
import com.vaadin.flow.component.Component;
import com.vaadin.flow.component.Composite;
import com.vaadin.flow.component.internal.ComponentTracker;
import com.vaadin.flow.shared.util.SharedUtil;

import com.github.javaparser.Range;
import com.github.javaparser.StaticJavaParser;
import com.github.javaparser.ast.CompilationUnit;
import com.github.javaparser.ast.DataKey;
import com.github.javaparser.ast.ImportDeclaration;
import com.github.javaparser.ast.Node;
import com.github.javaparser.ast.NodeList;
import com.github.javaparser.ast.body.BodyDeclaration;
import com.github.javaparser.ast.body.CallableDeclaration;
import com.github.javaparser.ast.body.ClassOrInterfaceDeclaration;
import com.github.javaparser.ast.body.ConstructorDeclaration;
import com.github.javaparser.ast.body.EnumDeclaration;
import com.github.javaparser.ast.body.FieldDeclaration;
import com.github.javaparser.ast.body.MethodDeclaration;
import com.github.javaparser.ast.body.RecordDeclaration;
import com.github.javaparser.ast.body.TypeDeclaration;
import com.github.javaparser.ast.body.VariableDeclarator;
import com.github.javaparser.ast.expr.AnnotationExpr;
import com.github.javaparser.ast.expr.ArrayCreationExpr;
import com.github.javaparser.ast.expr.ArrayInitializerExpr;
import com.github.javaparser.ast.expr.AssignExpr;
import com.github.javaparser.ast.expr.BinaryExpr;
import com.github.javaparser.ast.expr.BooleanLiteralExpr;
import com.github.javaparser.ast.expr.CastExpr;
import com.github.javaparser.ast.expr.CharLiteralExpr;
import com.github.javaparser.ast.expr.ClassExpr;
import com.github.javaparser.ast.expr.DoubleLiteralExpr;
import com.github.javaparser.ast.expr.Expression;
import com.github.javaparser.ast.expr.FieldAccessExpr;
import com.github.javaparser.ast.expr.IntegerLiteralExpr;
import com.github.javaparser.ast.expr.LambdaExpr;
import com.github.javaparser.ast.expr.LongLiteralExpr;
import com.github.javaparser.ast.expr.MarkerAnnotationExpr;
import com.github.javaparser.ast.expr.MemberValuePair;
import com.github.javaparser.ast.expr.MethodCallExpr;
import com.github.javaparser.ast.expr.NameExpr;
import com.github.javaparser.ast.expr.NormalAnnotationExpr;
import com.github.javaparser.ast.expr.NullLiteralExpr;
import com.github.javaparser.ast.expr.ObjectCreationExpr;
import com.github.javaparser.ast.expr.SingleMemberAnnotationExpr;
import com.github.javaparser.ast.expr.StringLiteralExpr;
import com.github.javaparser.ast.expr.VariableDeclarationExpr;
import com.github.javaparser.ast.nodeTypes.NodeWithArguments;
import com.github.javaparser.ast.nodeTypes.NodeWithCondition;
import com.github.javaparser.ast.nodeTypes.NodeWithSimpleName;
import com.github.javaparser.ast.stmt.BlockStmt;
import com.github.javaparser.ast.stmt.DoStmt;
import com.github.javaparser.ast.stmt.ExpressionStmt;
import com.github.javaparser.ast.stmt.ForEachStmt;
import com.github.javaparser.ast.stmt.ForStmt;
import com.github.javaparser.ast.stmt.IfStmt;
import com.github.javaparser.ast.stmt.ReturnStmt;
import com.github.javaparser.ast.stmt.Statement;
import com.github.javaparser.ast.stmt.WhileStmt;
import com.github.javaparser.ast.type.ClassOrInterfaceType;
import com.github.javaparser.ast.type.PrimitiveType;
import com.github.javaparser.ast.type.Type;
import com.github.javaparser.resolution.Resolvable;
import com.github.javaparser.resolution.UnsolvedSymbolException;
import com.github.javaparser.resolution.declarations.ResolvedConstructorDeclaration;
import com.github.javaparser.resolution.declarations.ResolvedDeclaration;
import com.github.javaparser.resolution.declarations.ResolvedEnumConstantDeclaration;
import com.github.javaparser.resolution.declarations.ResolvedFieldDeclaration;
import com.github.javaparser.resolution.declarations.ResolvedMethodDeclaration;
import com.github.javaparser.resolution.declarations.ResolvedTypeParameterDeclaration;
import com.github.javaparser.resolution.declarations.ResolvedValueDeclaration;
import com.github.javaparser.resolution.model.typesystem.ReferenceTypeImpl;
import com.github.javaparser.resolution.types.ResolvedArrayType;
import com.github.javaparser.resolution.types.ResolvedPrimitiveType;
import com.github.javaparser.resolution.types.ResolvedReferenceType;
import com.github.javaparser.resolution.types.ResolvedType;
import com.github.javaparser.resolution.types.ResolvedTypeVariable;
import com.github.javaparser.symbolsolver.javaparsermodel.declarations.JavaParserEnumConstantDeclaration;
import com.github.javaparser.symbolsolver.javaparsermodel.declarations.JavaParserFieldDeclaration;
import com.github.javaparser.symbolsolver.javaparsermodel.declarations.JavaParserVariableDeclaration;
import com.github.javaparser.symbolsolver.reflectionmodel.ReflectionEnumConstantDeclaration;
import com.github.javaparser.symbolsolver.reflectionmodel.ReflectionFieldDeclaration;
import com.github.javaparser.symbolsolver.reflectionmodel.ReflectionTypeParameter;
import org.jetbrains.annotations.NotNull;
import org.jspecify.annotations.NonNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Util methods for rewriting Java
 */
public class JavaRewriterUtil {

    private static final Set<String> javaKeywords = Set.of("abstract", "assert", "boolean", "break", "byte", "case",
            "catch", "char", "class", "const", "continue", "default", "double", "do", "else", "enum", "extends",
            "false", "final", "finally", "float", "for", "goto", "if", "implements", "import", "instanceof", "int",
            "interface", "long", "native", "new", "null", "package", "private", "protected", "public", "return",
            "short", "static", "strictfp", "super", "switch", "synchronized", "this", "throw", "throws", "transient",
            "true", "try", "void", "volatile", "while");
    private static final int MAX_VARIABLE_NAME_LENGTH = 20;

    private static final Map<Class<?>, Class<?>> WRAPPER_TYPE_MAP;

    private static final Pattern NORMALIZED_PATTERN = Pattern.compile("\\p{M}");
    private static final Pattern SANITIZE_PATTERN = Pattern.compile("[^a-zA-Z0-9 -]");
    private static final Pattern SPACE_PATTERN = Pattern.compile("\\s+");

    static {
        WRAPPER_TYPE_MAP = new HashMap<>(16);
        WRAPPER_TYPE_MAP.put(Integer.class, int.class);
        WRAPPER_TYPE_MAP.put(Byte.class, byte.class);
        WRAPPER_TYPE_MAP.put(Character.class, char.class);
        WRAPPER_TYPE_MAP.put(Boolean.class, boolean.class);
        WRAPPER_TYPE_MAP.put(Double.class, double.class);
        WRAPPER_TYPE_MAP.put(Float.class, float.class);
        WRAPPER_TYPE_MAP.put(Long.class, long.class);
        WRAPPER_TYPE_MAP.put(Short.class, short.class);
        WRAPPER_TYPE_MAP.put(Void.class, void.class);
    }

    private JavaRewriterUtil() {
        // Only static helpers
    }

    private static Logger getLogger() {
        return LoggerFactory.getLogger(JavaRewriterUtil.class);
    }

    static Optional<MethodCallExpr> findLastFunctionCall(List<MethodCallExpr> functionCalls, String function,
            Expression... parameterExpressions) {

        ArrayList<MethodCallExpr> reversedFunctionCalls = new ArrayList<>(functionCalls);
        Collections.reverse(reversedFunctionCalls);

        return reversedFunctionCalls.stream()
                .filter(call -> hasAncestor(call, Statement.class) && hasAncestor(call, BlockStmt.class)).findFirst();
    }

    /*
     * Adds a function call finding the correct location according to the use case.
     *
     * If no initializer, the function call is added after the last function call to
     * the parent in the relevant block. If initializer, if the last function call
     * is before the initializer then the function call is added after the
     * initializer.
     *
     */
    static MethodCallExpr addAfterLastFunctionCallInRelevantBlockStmt(List<MethodCallExpr> functionCalls,
            ObjectCreationExpr initializer, String function, Expression... parameterExpressions) {
        Optional<MethodCallExpr> lastFunctionCall = findCandidateLastMethodToAddExprAfter(functionCalls);
        if (lastFunctionCall.isEmpty()) {
            return null;
        }

        Statement initStmt = initializer != null ? findAncestor(initializer, Statement.class) : null;
        if (initStmt == null) {
            // NO initializer scenario

            MethodCallExpr lastCall = lastFunctionCall.get();
            Optional<Expression> maybeScope = getOutermostScope(lastCall);
            Expression scope = maybeScope.orElse(null);

            Statement statement = findAncestorOrThrow(lastCall, Statement.class);
            BlockStmt block = findAncestorOrThrow(lastCall, BlockStmt.class);
            MethodCallExpr newCall = createMethodCall(scope, function, parameterExpressions);
            block.addStatement(block.getStatements().indexOf(statement) + 1, new ExpressionStmt(newCall));
            return newCall;
        }

        BlockStmt block = findAncestorOrThrow(initStmt, BlockStmt.class);

        // Decide anchor statement
        Statement anchorStmt = initStmt;

        MethodCallExpr lastCall = lastFunctionCall.get();
        Statement lastStmt = findAncestor(lastCall, Statement.class);

        if (lastStmt != null && findAncestor(lastStmt, BlockStmt.class) == block) {
            int initIdx = block.getStatements().indexOf(initStmt);
            int lastIdx = block.getStatements().indexOf(lastStmt);

            // Insert after last call only if it is after initializer
            if (initIdx >= 0 && lastIdx > initIdx) {
                anchorStmt = lastStmt;
            }
        }

        Expression scope = getOutermostScope(lastFunctionCall.get()).orElse(null);
        MethodCallExpr newCall = createMethodCall(scope, function, parameterExpressions);

        int anchorIdx = block.getStatements().indexOf(anchorStmt);
        if (anchorIdx < 0) {
            return null;
        }

        block.addStatement(anchorIdx + 1, new ExpressionStmt(newCall));
        return newCall;

    }

    public static MethodCallExpr createMethodCall(Expression scope, String function,
            Expression... parameterExpressions) {
        MethodCallExpr newCall = new MethodCallExpr(scope, function);
        for (Expression parameterExpression : parameterExpressions) {
            newCall.addArgument(parameterExpression);
        }
        return newCall;
    }

    static List<MethodCallExpr> findMethodCallsInClass(Expression expressionInsideClass,
            Node.TreeTraversal traversalOrder) {
        return (List) findUsageInClass(expressionInsideClass, body -> findMethodCalls(body, traversalOrder));
    }

    static List<Expression> findUsageInClass(Expression expressionInsideClass,
            Function<BlockStmt, Stream<? extends Expression>> filter) {
        ClassOrInterfaceDeclaration classDeclaration = findAncestor(expressionInsideClass,
                ClassOrInterfaceDeclaration.class);
        if (classDeclaration == null) {
            return Collections.emptyList();
        }

        List<BlockStmt> blocks = classDeclaration.findAll(BlockStmt.class);
        // Filter out inner classes
        return blocks.stream().filter(block -> JavaRewriterUtil.findAncestorOrThrow(block,
                ClassOrInterfaceDeclaration.class) == classDeclaration).flatMap(filter).toList();
    }

    static VariableDeclarator findLocalVariableDeclarator(String variableName, BlockStmt block) {
        Predicate<VariableDeclarator> equalsPredicate = v -> v.getNameAsString().equals(variableName);
        VariableDeclarator variableDeclarator = block.findFirst(VariableDeclarator.class, equalsPredicate).orElse(null);
        if (variableDeclarator != null) {
            return variableDeclarator;
        }
        // when declaration and assignments are different block, we should look for
        // variable declaration from ancestor blocks.
        // assigned to a variable that defined out of the block
        Node nodeToSearchForVariableDeclaration = block.getParentNode().orElse(null);
        while (nodeToSearchForVariableDeclaration != null) {
            if (nodeToSearchForVariableDeclaration instanceof ClassOrInterfaceDeclaration) {
                // stops at class declaration. Because variables inside class considered as
                // Field variables.
                return null;
            } else if (nodeToSearchForVariableDeclaration instanceof BlockStmt blockStmt) {
                variableDeclarator = blockStmt.findFirst(VariableDeclarator.class, equalsPredicate).orElse(null);
                if (variableDeclarator != null) {
                    return variableDeclarator;
                }
            }
            nodeToSearchForVariableDeclaration = nodeToSearchForVariableDeclaration.getParentNode().orElse(null);
        }
        return null;
    }

    static NodeList<Expression> toExpressionList(Object parameter) {
        if (parameter instanceof List<?> list) {
            return new NodeList<>(list.stream().map(JavaRewriterUtil::toExpression).toList());
        } else {
            return new NodeList<>(toExpression(parameter));
        }
    }

    static Expression toExpression(Object parameter) {
        if (parameter == null) {
            return new NullLiteralExpr();
        } else if (parameter instanceof Expression e) {
            return e;
        } else if (parameter instanceof String s) {
            StringLiteralExpr expr = new StringLiteralExpr();
            expr.setString(s);
            return expr;
        } else if (parameter instanceof Long l) {
            return new LongLiteralExpr(String.valueOf(l));
        } else if (parameter instanceof Double d) {
            return new DoubleLiteralExpr(d);
        } else if (parameter instanceof Boolean b) {
            return new BooleanLiteralExpr(b);
        } else if (parameter instanceof Integer d) {
            return new IntegerLiteralExpr(d + "");
        } else if (parameter instanceof Enum e) {
            return new FieldAccessExpr(new NameExpr(e.getClass().getSimpleName()), e.name());
        } else if (parameter instanceof JavaRewriter.Code code) {
            return StaticJavaParser.parseExpression(code.code());
        } else if (parameter instanceof LocalTime lt) {
            return new MethodCallExpr(new NameExpr(LocalTime.class.getSimpleName()), "parse")
                    .addArgument(new StringLiteralExpr(lt.toString()));
        } else if (parameter instanceof Float f) {
            return new DoubleLiteralExpr(f + "f");
        } else if (parameter instanceof Character c) {
            return new CharLiteralExpr(c);
        } else if (parameter instanceof Byte b) {
            return new CastExpr(new PrimitiveType(PrimitiveType.Primitive.BYTE), new IntegerLiteralExpr(b + ""));
        } else if (parameter instanceof Short s) {
            return new CastExpr(new PrimitiveType(PrimitiveType.Primitive.SHORT), new IntegerLiteralExpr(s + ""));
        } else if (parameter instanceof BigDecimal bigDecimal) {
            MethodCallExpr methodCallExpr = new MethodCallExpr(new NameExpr(BigDecimal.class.getSimpleName()),
                    "valueOf");
            if (bigDecimal.scale() != 0) {
                methodCallExpr.addArgument(new DoubleLiteralExpr(bigDecimal.doubleValue()));
            } else {
                methodCallExpr.addArgument(new IntegerLiteralExpr(bigDecimal.intValue() + ""));
            }
            return methodCallExpr;
        } else if (parameter instanceof BigInteger bigInteger) {
            MethodCallExpr methodCallExpr = new MethodCallExpr(new NameExpr(BigInteger.class.getSimpleName()),
                    "valueOf");
            methodCallExpr.addArgument(new LongLiteralExpr(bigInteger.longValue() + ""));
            return methodCallExpr;
        } else {
            throw new IllegalArgumentException("Unknown type: " + parameter.getClass().getName());
        }
    }

    static boolean equalsByNameAsString(Expression e1, Expression e2) {
        if (e1 == null && e2 == null) {
            return true;
        }
        if (e1 == null) {
            return false;
        }
        if (e2 == null) {
            return false;
        }
        if (e1.isNullLiteralExpr() && e2.isNullLiteralExpr()) {
            return true;
        }
        if (e1.isStringLiteralExpr() && e2.isStringLiteralExpr()) {
            return e1.asStringLiteralExpr().toString().equals(e2.asStringLiteralExpr().toString());
        }
        if (e1.isFieldAccessExpr() && e2.isFieldAccessExpr()) {
            return e1.toString().equals(e2.toString());
        }
        if (e1 instanceof NodeWithSimpleName<?> && e2 instanceof NodeWithSimpleName<?>) {
            return ((NodeWithSimpleName<?>) e1).getNameAsString()
                    .equals(((NodeWithSimpleName<?>) e2).getNameAsString());
        }
        if (e1 instanceof ObjectCreationExpr && e2 instanceof ObjectCreationExpr) {
            return e1.toString().equals(e2.toString());
        }
        return false;
    }

    /**
     * Parses the given expression and returns the object it represents.
     *
     * <p>
     * If the argument is a null literal, it is returned as is. Method call
     * expressions are also returned as is.
     *
     * @param arg
     *            the expression to parse
     * @param expectedType
     *            the expected type of the object or null if unknown
     * @return the object represented by the expression
     */
    public static Object fromExpression(Expression arg, ResolvedType expectedType) {
        if (arg instanceof StringLiteralExpr literalExpression) {
            return literalExpression.asString();
        } else if (arg instanceof IntegerLiteralExpr integerLiteralExpr) {
            return integerLiteralExpr.asNumber();
        } else if (arg instanceof DoubleLiteralExpr doubleLiteralExpr) {
            if (expectedType != null && expectedType.isPrimitive()
                    && expectedType.asPrimitive().getBoxTypeClass() == Float.class) {
                return (float) doubleLiteralExpr.asDouble();
            }
            return doubleLiteralExpr.asDouble();
        } else if (arg instanceof BooleanLiteralExpr booleanLiteralExpr) {
            return booleanLiteralExpr.getValue();
        } else if (arg instanceof CharLiteralExpr charLiteralExpr) {
            return charLiteralExpr.asChar();
        } else if (arg instanceof LongLiteralExpr longLiteralExpr) {
            return longLiteralExpr.asNumber().longValue();
        } else if (arg instanceof NullLiteralExpr) {
            return arg;
        } else if (arg instanceof MethodCallExpr methodCallExpr) {
            return methodCallExpr;
        } else if (arg instanceof FieldAccessExpr fieldAccessExpr) {
            Optional<Object> result = fromFieldAccessExpression(fieldAccessExpr, expectedType);
            if (result.isPresent()) {
                return result.get();
            }
        } else if (arg instanceof NameExpr nameExpr) {
            Optional<Object> result = fromNameExpr(nameExpr, expectedType);
            if (result.isPresent()) {
                return result.get();
            }
        } else if (arg instanceof CastExpr castExpr) {
            if (expectedType.describe().equals(castExpr.getType().resolve().describe())) {
                return fromExpression(castExpr.getExpression(), expectedType);
            }
            if (expectedType instanceof ReferenceTypeImpl referenceType && castExpr.getType().isPrimitiveType()
                    && referenceType.isUnboxableTo(castExpr.getType().asPrimitiveType().resolve())) {
                return fromExpression(castExpr.getExpression(), castExpr.getType().asPrimitiveType().resolve());
            }
        } else if (arg instanceof ArrayCreationExpr arrayCreationExpr
                && arrayCreationExpr.getInitializer().isPresent()) {
            ArrayInitializerExpr arrayInitializerExpr = arrayCreationExpr.getInitializer().get();
            return arrayInitializerExpr.getValues().stream().map(expression -> fromExpression(expression, expectedType))
                    .toArray();
        }

        throw new UnknownExpressionTypeException(arg.getClass());
    }

    /**
     * Resolves field access expression to find java value that matches with
     * expected type.
     *
     * @param fieldAccessExpr
     *            field access expression e.g. setName(AClass.aField);
     * @param expectedType
     *            argument type that should be match with field value
     * @return returns {@link Optional#get()} if value is found and matched, *
     *         {@link Optional#empty()} otherwise.
     */
    private static Optional<Object> fromFieldAccessExpression(FieldAccessExpr fieldAccessExpr,
            ResolvedType expectedType) {
        ResolvedValueDeclaration resolvedValueDeclaration = fieldAccessExpr.resolve();
        if (resolvedValueDeclaration instanceof ReflectionFieldDeclaration reflectionFieldDeclaration) {
            return findValue(reflectionFieldDeclaration, expectedType);
        } else if (resolvedValueDeclaration instanceof JavaParserFieldDeclaration javaParserFieldDeclaration) {
            return findValue(javaParserFieldDeclaration, expectedType);
        } else if (resolvedValueDeclaration instanceof ReflectionEnumConstantDeclaration reflectionEnumConstantDeclaration) {
            return findValue(reflectionEnumConstantDeclaration, expectedType);
        } else if (resolvedValueDeclaration instanceof JavaParserEnumConstantDeclaration javaParserEnumConstantDeclaration) {
            return findValue(javaParserEnumConstantDeclaration, expectedType);
        }
        return Optional.empty();
    }

    /**
     * Resolves name expression to find java value that matches with expected type.
     *
     * @param nameExpr
     *            name expression that represents statements like
     *            {@code aMethod(NAME_EXPRESSION)}
     * @param expectedType
     *            argument type that should be match with field value
     * @return returns {@link Optional#get()} if value is found and matched, *
     *         {@link Optional#empty()} otherwise.
     */
    private static Optional<Object> fromNameExpr(NameExpr nameExpr, ResolvedType expectedType) {
        if (expectedType == null) {
            return Optional.empty();
        }
        ResolvedValueDeclaration resolvedValueDeclaration = nameExpr.resolve();
        if (resolvedValueDeclaration instanceof ReflectionFieldDeclaration reflectionFieldDeclaration) {
            return findValue(reflectionFieldDeclaration, expectedType);
        } else if (resolvedValueDeclaration instanceof JavaParserFieldDeclaration javaParserFieldDeclaration) {
            return findValue(javaParserFieldDeclaration, expectedType);
        }
        return Optional.empty();
    }

    /**
     * Gets the Java value of field if it matches with ResolvedType
     *
     * @param reflectionFieldDeclaration
     *            Field declaration from Java Parser
     * @param expectedType
     *            Expected Type
     * @return returns {@link Optional#get()} if value is found and matched,
     *         {@link Optional#empty()} otherwise.
     */
    private static Optional<Object> findValue(ReflectionFieldDeclaration reflectionFieldDeclaration,
            ResolvedType expectedType) {
        if (expectedType == null) {
            return Optional.empty();
        }
        String qualifiedName = reflectionFieldDeclaration.declaringType().getQualifiedName();
        Optional<Object> staticFieldValue = JavaReflectionUtil.getStaticFieldValue(qualifiedName,
                reflectionFieldDeclaration.getName());

        if (staticFieldValue.isPresent()) {
            Object value = staticFieldValue.get();
            boolean expectedValueType = false;
            if (value.getClass().isPrimitive() && expectedType.isPrimitive()) {
                expectedValueType = true;
            } else if (!value.getClass().isPrimitive() && expectedType.isPrimitive()
                    && expectedType.asPrimitive().getBoxTypeClass().equals(value.getClass())) {
                expectedValueType = true;
            } else if (!value.getClass().isPrimitive() && !expectedType.isPrimitive()
                    && expectedType.asReferenceType().getQualifiedName().equals(value.getClass().getName())) {
                expectedValueType = true;
            }
            if (expectedValueType) {
                return Optional.of(value);
            }
        }
        return Optional.empty();
    }

    /**
     * Gets the java value of field declaration. Value is returned if declaration
     * has initializer. Otherwise, it returns Optional.empty
     *
     * @param javaParserFieldDeclaration
     *            Field declaration
     * @param expectedType
     *            Expected type
     * @return returns {@link Optional#get()} if value is found and matched, *
     *         {@link Optional#empty()} otherwise.
     */
    private static Optional<Object> findValue(JavaParserFieldDeclaration javaParserFieldDeclaration,
            ResolvedType expectedType) {
        if (javaParserFieldDeclaration == null) {
            return Optional.empty();
        }
        if (expectedType == null) {
            return Optional.empty();
        }
        VariableDeclarator variableDeclarator = javaParserFieldDeclaration.getVariableDeclarator();
        if (variableDeclarator == null) {
            return Optional.empty();
        }
        ResolvedType variableResolvedType = variableDeclarator.getType().resolve();
        // an example case is that field is Boolean.TRUE, argument is boolean
        if (!variableResolvedType.equals(expectedType)
                && expectedType instanceof ResolvedPrimitiveType expectedPrimitiveType) {
            try {
                Class<?> variableClass = JavaReflectionUtil.getClass(variableResolvedType.describe());
                if (!expectedPrimitiveType.getBoxTypeClass().equals(variableClass)) {
                    return Optional.empty();
                }
            } catch (Exception e) {
                return Optional.empty();
            }
        }
        if (variableDeclarator.getInitializer().isEmpty()) {
            return Optional.empty();
        }
        Expression expression = variableDeclarator.getInitializer().get();
        Object o = fromExpression(expression, expectedType);
        return Optional.of(o);
    }

    private static Optional<Object> findValue(ResolvedEnumConstantDeclaration resolvedEnumConstantDeclaration,
            ResolvedType expectedType) {
        if (expectedType == null
                && resolvedEnumConstantDeclaration instanceof ReflectionEnumConstantDeclaration reflectionEnumConstantDeclaration) {
            expectedType = reflectionEnumConstantDeclaration.getType();
        }
        if (resolvedEnumConstantDeclaration == null) {
            return Optional.empty();
        }
        if (expectedType == null) {
            return Optional.empty();
        }
        if (expectedType.isArray()) {
            ResolvedType componentType = expectedType.asArrayType().getComponentType();
            return findValue(resolvedEnumConstantDeclaration, componentType);
        }
        String expectedEnumQualifiedName = null;
        if (expectedType.isReferenceType()) {
            expectedEnumQualifiedName = expectedType.asReferenceType().getQualifiedName();
        }
        if (expectedEnumQualifiedName == null) {
            return Optional.empty();
        }
        String parameterEnumQualifiedName = resolvedEnumConstantDeclaration.getType().describe();
        if (!parameterEnumQualifiedName.equals(expectedEnumQualifiedName)) {
            return Optional.empty();
        }
        Class<?> aClass = JavaReflectionUtil.getClass(parameterEnumQualifiedName);
        return Optional.of(JavaReflectionUtil.getEnumConstant(aClass, resolvedEnumConstantDeclaration.getName()));
    }

    private static Stream<MethodCallExpr> findMethodCalls(BlockStmt scope, Node.TreeTraversal traversalOrder) {
        if (scope == null) {
            return Stream.empty();
        }
        return scope.findAll(MethodCallExpr.class, traversalOrder).stream();
    }

    /**
     * Finds all name references to the given variable name in the given scope.
     *
     * @param variableName
     *            the variable name
     * @param scope
     *            the scope
     * @return a stream of name references
     */
    public static Stream<Expression> findNameReferences(String variableName, BlockStmt scope) {
        if (scope == null) {
            return Stream.empty();
        }
        return scope
                .findAll(Expression.class, n -> (n.isNameExpr() || n.isFieldAccessExpr())
                        && nameMatches(n, variableName) && !isVariableNamePassedToAncestorCallable(n, variableName))
                .stream();
    }

    /**
     * Checks whether the specified variable name is declared as a parameter in the
     * nearest ancestor {@link CallableDeclaration} of the given expression.
     *
     * <p>
     * This method walks up the AST from the given {@link Expression} node and looks
     * for the closest enclosing {@link CallableDeclaration} (e.g., a method or
     * constructor). If such a declaration is found, it checks whether the provided
     * {@code variableName} matches the name of its parameters.
     *
     * @param expression
     *            the starting {@link Expression} node in the AST
     * @param variableName
     *            the name of the variable to check
     * @return {@code true} if the variable name is passed as a parameter in an
     *         ancestor callable declaration; {@code false} otherwise
     */
    static boolean isVariableNamePassedToAncestorCallable(Expression expression, String variableName) {
        CallableDeclaration<?> callableDeclaration = findAncestor(expression, CallableDeclaration.class);
        if (callableDeclaration == null) {
            return false;
        }
        return callableDeclaration.getParameters().stream()
                .anyMatch(parameter -> parameter.getNameAsString().equals(variableName));
    }

    /**
     * Find all method calls that are statements, i.e. calls like foo.bar() or
     * baz.qux("zug")). Does not find usage of the variable in other expressions,
     * like otherFunction("something" + foo.bar())
     *
     * @param componentDefinition
     *            the component to look for
     * @return a list of method calls
     */
    public static List<MethodCallExpr> findMethodCallStatements(ComponentInfo componentDefinition) {
        List<MethodCallExpr> methodCalls = findMethodCalls(componentDefinition);
        return methodCalls.stream()
                .filter(m -> isParentNode(m, ExpressionStmt.class) || isParentNode(m, VariableDeclarator.class))
                .toList();
    }

    /**
     * Find all method calls that are statements inside the block of an insertion
     * point, i.e. calls like foo.bar() or baz.qux("zug")). Does not find usage of
     * the variable in other expressions, like otherFunction("something" +
     * foo.bar())
     *
     * @param insertionPoint
     *            the insertion point to look for
     * @param componentDefinition
     *            the component to look for
     * @return a list of method calls
     */
    public static List<MethodCallExpr> findMethodCallStatements(InsertionPoint insertionPoint,
            ComponentInfo componentDefinition) {
        List<MethodCallExpr> methodCalls = findMethodCallStatements(componentDefinition);
        return methodCalls.stream().filter(m -> insertionPoint.containsRange(m.getRange().orElse(null))).toList();
    }

    /**
     * Attempts to determine the most suitable {@link MethodCallExpr} after which a
     * new expression could be inserted, based on structural and contextual analysis
     * of the provided method calls.
     *
     * <p>
     * The method performs several filtering steps on the input list:
     * <ul>
     * <li>Removes method calls that occur inside loops.</li>
     * <li>Keeps only method calls that have a {@link Statement} ancestor.</li>
     * <li>Keeps only method calls that are contained inside a
     * {@link BlockStmt}.</li>
     * <li>Filters out method calls that appear inside {@link IfStmt} conditions or
     * bodies.</li>
     * </ul>
     * These filters ensure that the candidate is a method call found in a normal,
     * linear execution path, not inside loops or conditional branches.
     * </p>
     *
     * <p>
     * After filtering, the method groups the remaining method calls by their
     * enclosing {@link BlockStmt}. For each block, it tracks:
     * <ul>
     * <li>The number of method calls found inside that block.</li>
     * <li>The last encountered method call inside that block.</li>
     * </ul>
     * The block with the highest number of method calls is considered the "target
     * block". The method then attempts to return the last method call found in that
     * block.
     * </p>
     *
     * <p>
     * If no dominant block can be determined, or if the internal mappings end up
     * empty due to the filtering, the method falls back to returning the last
     * method call from the filtered list.
     * </p>
     *
     * @param allMethodCalls
     *            a list of {@link MethodCallExpr} instances to analyze; may be
     *            empty
     * @return an {@link Optional} containing the selected {@link MethodCallExpr},
     *         or {@link Optional#empty()} if the input list is empty
     *
     */
    static Optional<MethodCallExpr> findCandidateLastMethodToAddExprAfter(List<MethodCallExpr> allMethodCalls) {
        if (allMethodCalls.isEmpty()) {
            return Optional.empty();
        }
        // eliminating method calls in loop and conditions
        List<MethodCallExpr> list = allMethodCalls.stream().filter(call -> !checkExpressionInLoop(call))
                .filter(call -> hasAncestor(call, Statement.class)).filter(call -> hasAncestor(call, BlockStmt.class))
                .filter(call -> !hasAncestor(call, IfStmt.class)).toList();
        if (list.isEmpty()) {
            return Optional.empty();
        }
        // this map contains the last method in the block
        Map<BlockStmt, MethodCallExpr> targetMethodInBlock = new HashMap<>();
        // contains the method calls in each block
        Map<BlockStmt, Integer> methodCallsInBlockStmts = new HashMap<>();
        for (MethodCallExpr methodCallExpr : list) {
            BlockStmt ancestor = findAncestor(methodCallExpr, BlockStmt.class);
            assert ancestor != null;
            if (!methodCallsInBlockStmts.containsKey(ancestor)) {
                methodCallsInBlockStmts.put(ancestor, 0);
            }
            methodCallsInBlockStmts.put(ancestor, methodCallsInBlockStmts.get(ancestor) + 1);
            targetMethodInBlock.put(ancestor, methodCallExpr);
        }
        Map.Entry<BlockStmt, Integer> targetEntry = methodCallsInBlockStmts.entrySet().stream()
                .max(Map.Entry.comparingByValue()).orElse(null);
        if (targetEntry == null) {
            // if target is empty, return the last method as a fallback.
            return Optional.of(list.getLast());
        }
        BlockStmt targetBlock = targetEntry.getKey();
        if (!targetMethodInBlock.containsKey(targetBlock)) {
            // if target is empty, return the last method as a fallback.
            return Optional.of(list.getLast());
        }
        MethodCallExpr methodCallExpr = targetMethodInBlock.get(targetBlock);
        return Optional.of(methodCallExpr);
    }

    /**
     * Find all method calls statements for the given function, i.e. calls like
     * foo.bar() or baz.qux("zug")). Does not find usage of the variable in other
     * expressions, like otherFunction("something" + foo.bar())
     *
     * @param componentInfo
     *            the component to look for
     * @param functionName
     *            the function name to look for
     * @return a list of method calls
     */
    public static Stream<MethodCallExpr> findMethodCallStatements(ComponentInfo componentInfo, String functionName) {
        return JavaRewriterUtil.findMethodCallStatements(componentInfo).stream()
                .filter(m -> m.getNameAsString().equals(functionName));
    }

    /**
     * Finds usage of the variable outside of method call statements, i.e. all usage
     * not reported by {@link #findMethodCallStatements(ComponentInfo)}.
     *
     * @param componentDefinition
     *            the component to look for
     * @return a list of method calls
     */
    public static List<MethodCallExpr> findMethodCallNonStatements(ComponentInfo componentDefinition) {
        List<MethodCallExpr> methodCalls = findMethodCalls(componentDefinition);
        return methodCalls.stream().filter(m -> !isParentNode(m, ExpressionStmt.class)).toList();
    }

    /**
     * Finds usage of the variable outside of method call statements, i.e. all usage
     * not reported by {@link #findMethodCallStatements(ComponentInfo)}.
     *
     * @param componentDefinition
     *            the component to look for
     * @param functionName
     *            the function name to look for
     * @return a list of method calls
     */
    public static Stream<MethodCallExpr> findMethodCallNonStatements(ComponentInfo componentDefinition,
            String functionName) {
        return JavaRewriterUtil.findMethodCallNonStatements(componentDefinition).stream()
                .filter(m -> m.getNameAsString().equals(functionName));
    }

    /**
     * Finds all method calls that are related to the given component.
     *
     * @param componentInfo
     *            the component to look for
     * @return a list of method calls
     */
    public static List<MethodCallExpr> findMethodCalls(ComponentInfo componentInfo) {
        return findMethodCalls(componentInfo, Node.TreeTraversal.PREORDER);
    }

    /**
     * Finds all method calls that are related to the given component.
     *
     * @param componentInfo
     *            the component to look for
     * @param traversalOrder
     *            the order in which the nodes are traversed
     * @return a list of method calls
     */
    public static List<MethodCallExpr> findMethodCalls(ComponentInfo componentInfo, Node.TreeTraversal traversalOrder) {
        if (componentInfo.getCreateInfoOrThrow().getFieldName() != null) {
            // If we have a field, we need to go through the whole class because
            // any method anywhere can run functions on it

            // FIXME This returns all methods call anywhere in the class but
            // what the logic would want in most cases
            // is a list of methods called when calling the class constructor or
            // other init methods
            List<MethodCallExpr> classMethodCalls = new ArrayList<>();
            classMethodCalls.addAll(findMethodCallsInClass(componentInfo.getCreateInfoOrThrow().getObjectCreationExpr(),
                    traversalOrder));
            classMethodCalls.addAll(findDistinctMethodCallsUsingAttachLocationInfo(componentInfo, traversalOrder));
            return distinctByIdentity(classMethodCalls.stream()
                    .filter(m -> scopeIs(m, componentInfo.getCreateInfoOrThrow().getFieldName())).toList());
        } else if (componentInfo.routeConstructor() != null) {
            // Only consider the constructor for now. For full support, would
            // need to traverse all
            // methods called from the constructor in order. And also consider
            // other init methods like @PostConstruct
            return distinctByIdentity(
                    JavaRewriterUtil.findMethodCalls(componentInfo.routeConstructor().getBody(), traversalOrder)
                            .filter(m -> scopeIs(m, null)).toList());
        } else if (componentInfo.getCreateInfoOrThrow().getLocalVariableName() != null) {
            List<MethodCallExpr> list = new ArrayList<>(JavaRewriterUtil
                    .findMethodCalls(componentInfo.getCreateInfoOrThrow().getComponentCreateScope(), traversalOrder)
                    .filter(m -> scopeIs(m, componentInfo.getCreateInfoOrThrow().getLocalVariableName())).toList());
            Set<MethodCallExpr> methodExprsToRemove = getMethodCallExprsInLambdaToRemove(list);
            list.removeIf(methodExprsToRemove::contains);
            ArrayList<MethodCallExpr> functionCalls = new ArrayList<>();
            functionCalls.addAll(list);
            functionCalls.addAll(findDistinctMethodCallsUsingAttachLocationInfo(componentInfo, traversalOrder));
            return functionCalls.stream().distinct().toList();
        }
        return findDistinctMethodCallsUsingAttachLocationInfo(componentInfo, traversalOrder);
    }

    @NotNull
    private static Set<MethodCallExpr> getMethodCallExprsInLambdaToRemove(List<MethodCallExpr> list) {
        Set<MethodCallExpr> methodExprsToRemove = new HashSet<>();
        for (int i = 1; i < list.size(); i++) {
            MethodCallExpr prev = list.get(i - 1);
            MethodCallExpr curr = list.get(i);

            // removes the method call expressions in lambda block since prev call covers it
            if (JavaRewriterUtil.hasAncestor(curr, LambdaExpr.class)
                    && JavaRewriterUtil.findOutermostAncestor(curr, node -> node.equals(prev)) != null) {
                methodExprsToRemove.add(curr);
            }
        }
        return methodExprsToRemove;
    }

    /**
     * Finds distinct method calls related to a {@link ComponentInfo}'s attach
     * location.
     * <p>
     * This method analyzes the attach scope of a component and returns method calls
     * made on the component's local variable or field, based on whether the attach
     * location is in the same file as the create location or not. It avoids
     * duplicate method calls already accounted for from the create scope.
     * <p>
     * Several conditions are considered to return an accurate and filtered list of
     * method calls:
     * <ul>
     * <li>If no attach info is present, returns an empty list.</li>
     * <li>If the attach location is in the same file and already processed during
     * creation, returns an empty list.</li>
     * <li>If the attach location is in a different file, searches within the attach
     * scope block for method calls related to the component field or variable.</li>
     * </ul>
     *
     * @param componentInfo
     *            the {@link ComponentInfo} holding metadata about the component's
     *            creation and attachment.
     * @param traversalOrder
     *            the {@link Node.TreeTraversal} strategy to use when traversing the
     *            syntax tree for method calls.
     * @return a list of distinct {@link MethodCallExpr} instances found in the
     *         attach location scope.
     */
    private static List<MethodCallExpr> findDistinctMethodCallsUsingAttachLocationInfo(ComponentInfo componentInfo,
            Node.TreeTraversal traversalOrder) {
        if (componentInfo.componentAttachInfoOptional().isEmpty()) {
            return new ArrayList<>();
        }
        ComponentAttachInfo componentAttachInfo = componentInfo.componentAttachInfoOptional().get();
        Optional<BlockStmt> componentAttachScopeOptional = componentAttachInfo.getComponentAttachScope();
        if (componentAttachScopeOptional.isEmpty()) {
            // no attach scope found.
            return new ArrayList<>();
        }
        if (componentInfo.createAndAttachLocationsAreInSameFile()) {
            if (componentAttachInfo.getFieldName() != null) {
                // if create and attach location are in the same file, the methods are added
                // from create location.
                return new ArrayList<>();
            }

            if (componentInfo.componentCreateInfoOptional().isPresent()) {
                ComponentCreateInfo componentCreateInfo = componentInfo.componentCreateInfoOptional().get();
                if (componentCreateInfo.getComponentCreateScope() == componentAttachScopeOptional.get()) {
                    // scope for create and attach location is the same. Methods are already added
                    // from finding method calls of create location
                    return new ArrayList<>();
                }
            }
            if (componentAttachInfo.getLocalVariableName() == null) {
                return new ArrayList<>();
            }
            BlockStmt attachScope = componentAttachScopeOptional.get();
            return JavaRewriterUtil.findMethodCalls(attachScope, traversalOrder)
                    .filter(m -> scopeIs(m, componentAttachInfo.getLocalVariableName())).toList();
        }
        List<MethodCallExpr> methodCalls = new ArrayList<>();

        // the attach location is in different file, so we should look for method calls
        // in that class.
        if (componentAttachInfo.getLocalVariableName() != null) {

            methodCalls.addAll(findMethodCalls(componentAttachScopeOptional.get(), traversalOrder)
                    .filter(m -> scopeIs(m, componentAttachInfo.getLocalVariableName())).toList());
        }

        if (componentAttachInfo.getFieldName() != null) {
            List<MethodCallExpr> methodCallsFromFieldVars = componentAttachScopeOptional.get().getStatements().stream()
                    .filter(Statement::isExpressionStmt).map(ExpressionStmt.class::cast)
                    .map(expressionStmt -> findMethodCallsInClass(expressionStmt.getExpression(), traversalOrder)
                            .stream().filter(m -> scopeIs(m, componentAttachInfo.getFieldName())).toList())
                    .findFirst().orElseGet(ArrayList::new);
            methodCallsFromFieldVars.addAll(methodCalls);
        }

        return methodCalls;
    }

    private static <T> List<T> distinctByIdentity(List<T> list) {
        IdentityHashSet<T> found = new IdentityHashSet<>();
        return list.stream().filter(found::add).toList();
    }

    /**
     * Find all expressions where the given component is used as a call parameter.
     *
     * @param componentDefinition
     *            the component to look for
     * @return a list of expressions
     */
    public static List<Expression> findParameterUsage(ComponentInfo componentDefinition) {
        List<Expression> refs;
        var createInfo = componentDefinition.getCreateInfoOrThrow();
        if (createInfo.getFieldName() != null) {
            refs = findUsageInClass(componentDefinition.getCreateInfoOrThrow().getObjectCreationExpr(),
                    body -> findNameReferences(createInfo.getFieldName(), body));
        } else if (createInfo.getLocalVariableName() != null) {
            refs = findNameReferences(createInfo.getLocalVariableName(), createInfo.getComponentCreateScope()).toList();
        } else if (componentDefinition.getCreateInfoOrThrow().getObjectCreationExpr() != null) {
            // EDGE CASE FOR CUSTOM COMPONENT THAT CAN USE CONSTRUCTORS IN ATTACHMENTS OR
            // ANY METHOD
            refs = List.of(componentDefinition.getCreateInfoOrThrow().getObjectCreationExpr());
            return refs.stream().toList();
        } else {
            refs = Collections.emptyList();
        }
        return refs.stream().filter(JavaRewriterUtil::isMethodCallArgument).toList();
    }

    /**
     * Extracts an inline variable to local variable with a new variable name. The
     * new variable is created in the same block as the inline variable just before
     * the replacement.
     *
     * @param componentInfo
     *            Component info
     * @return Block, new variable name and index
     * @throws CopilotException
     *             if the inline variable cannot be extracted
     */
    public static JavaRewriter.ExtractInlineVariableResult extractInlineVariableToLocalVariable(
            ComponentInfo componentInfo) {
        Type type = componentInfo.getCreateInfoOrThrow().getObjectCreationExpr().getType();
        BlockStmt block = JavaRewriterUtil
                .findAncestorOrThrow(componentInfo.getCreateInfoOrThrow().getObjectCreationExpr(), BlockStmt.class);
        String newLocalVariable = JavaRewriterUtil.findFreeVariableName(componentInfo, block);
        if (componentInfo.componentAttachInfoOptional().isEmpty()) {
            throw new CopilotException("No attach location found");
        }

        Optional<Node> parentNode = componentInfo.getCreateInfoOrThrow().getObjectCreationExpr().getParentNode();
        if (parentNode.isEmpty()) {
            throw new IllegalStateException("Parent for object creation does not exist");
        }

        // add(new Input(...)) -> add(input)
        NameExpr replacementNode = new NameExpr(newLocalVariable);
        parentNode.get().replace(componentInfo.getCreateInfoOrThrow().getObjectCreationExpr(), replacementNode);
        int indexNewVariableUsage = JavaRewriterUtil.findBlockStatementIndex(replacementNode);
        // Input input = new Input(...);
        VariableDeclarator declarator = new VariableDeclarator(type, newLocalVariable,
                componentInfo.getCreateInfoOrThrow().getObjectCreationExpr());
        block.addStatement(indexNewVariableUsage, new VariableDeclarationExpr(declarator));

        return new JavaRewriter.ExtractInlineVariableResult(block, newLocalVariable, indexNewVariableUsage);

    }

    /**
     * Checks if given attach and create location refers to a source file instead of
     * instance creation.
     *
     * @param typeAndSourceLocation
     *            the type and source location to check
     * @param compilationUnit
     *            the compilation unit that contains the code
     * @return {@code true} if given source location and compilation unit refers to
     *         source file, {@code false} otherwise
     */
    public static boolean isLocationRefersSource(ComponentTypeAndSourceLocation typeAndSourceLocation,
            CompilationUnit compilationUnit) {
        Optional<ComponentTracker.Location> attachLocation = typeAndSourceLocation.attachLocationInProject();
        Optional<ComponentTracker.Location> createLocation = typeAndSourceLocation.createLocationInProject();

        if (attachLocation.isPresent()) {
            // There is no attach for a route in the project sources
            return false;
        }

        if (createLocation.isEmpty()) {
            // There IS a create location marked for a route in the project sources
            return false;
        }
        ComponentTracker.Location createLocationInProject = createLocation.get();
        boolean initCall = createLocationInProject.methodName().equals("<init>");
        if (!initCall) {
            return false;
        }

        // without this check below add(new VerticalLayout(...)) returns true
        // for VerticalLayout
        Node node = findNode(compilationUnit, createLocationInProject.lineNumber(), CompilationUnit.class::isInstance,
                ClassOrInterfaceDeclaration.class::isInstance, ConstructorDeclaration.class::isInstance);
        if (node == null) {
            return true;
        }
        if (node instanceof CompilationUnit) {
            // public class NoConstructor extends Div {} returns
            // CompilationUnit
            return true;
        }
        if (node instanceof ClassOrInterfaceDeclaration) {
            return true;
        }
        return node instanceof ConstructorDeclaration;
    }

    private static boolean isMethodCallArgument(Expression ref) {
        Optional<Node> parent = ref.getParentNode();
        if (parent.isEmpty()) {
            return false;
        }
        return parent.get() instanceof Expression e && e.isMethodCallExpr()
                && e.asMethodCallExpr().getArguments().contains(ref);
    }

    static boolean isParentNode(Node node, Class<?> parentType) {
        return node.getParentNode().filter(value -> parentType == value.getClass()).isPresent();
    }

    static boolean hasAncestor(Node node, Class<?> parentType) {
        return findAncestor(node, parentType) != null;
    }

    /**
     * Determines if the given method call expression is done either directly on the
     * given variable name, or if the method call is done on an intermediate value
     * that is retrieved from the given variable name.
     *
     * <p>
     * For instance, given the code
     *
     * <pre>
     * Button b = new Button();
     * b.setText("foo");
     * b.getStyle().setBackground("blue");
     * Style s = b.getStyle();
     * s.set("color", "red");
     * </pre>
     *
     * <p>
     * it will return true for "setText", "setBackground" and "set" when called for
     * the "b" variable as all the method calls originate in a method call on the
     * Button b.
     *
     * @param m
     *            the method call to check
     * @param variableName
     *            the variable name of the component
     * @return true if the method call is done on the given variable name or an
     *         intermediate value retrieved from the variable name
     */
    static boolean scopeIs(MethodCallExpr m, String variableName) {
        Optional<Expression> maybeScope = m.getScope();
        if (maybeScope.isEmpty() || maybeScope.get().isThisExpr()) {
            return variableName == null;
        }
        Expression scope = maybeScope.get();
        if (scope instanceof MethodCallExpr) {
            // Method call, e.g. button.getStyle() -> check scope of that method
            // call
            return scopeIs((MethodCallExpr) scope, variableName);
        } else if (scope instanceof NameExpr) {
            if (nameMatches(scope, variableName)) {
                return true;
            }

            // Variable assignment, e.g. s.setColor("black") -> check the
            // definition of the variable s, if it comes from the target
            // component or not
            NameExpr scopeName = scope.asNameExpr();
            try {
                ResolvedValueDeclaration resolved = scopeName.resolve();
                VariableDeclarator variableDeclarator = null;
                if (resolved.isVariable()
                        && resolved instanceof JavaParserVariableDeclaration javaParserVariableDeclaration) {
                    variableDeclarator = javaParserVariableDeclaration.getVariableDeclarator();
                } else if (resolved.isField()
                        && resolved instanceof JavaParserFieldDeclaration javaParserFieldDeclaration) {
                    variableDeclarator = javaParserFieldDeclaration.getVariableDeclarator();
                } else {
                    getLogger().debug("Unknown type {}", resolved);
                }

                Optional<Expression> initializer = Optional.ofNullable(variableDeclarator)
                        .flatMap(VariableDeclarator::getInitializer);
                if (initializer.isPresent()) {
                    // If the initializer is a method call, we need to check the scope of that
                    // method call
                    Optional<MethodCallExpr> methodCall = initializer.filter(Expression::isMethodCallExpr)
                            .map(Expression::asMethodCallExpr);
                    if (methodCall.isPresent()) {
                        return scopeIs(methodCall.get(), variableName);
                    } else if (initializer.filter(Expression::isObjectCreationExpr).isPresent()) {
                        // The initializer is "new Something()", it does not refer to any variable
                        return false;
                    }
                }
                getLogger().debug("Unhandled resolved scope {}", resolved);
            } catch (UnsolvedSymbolException e) {
                try {
                    if (m.resolve().isStatic()) {
                        // A static method call like UI.getCurrent never refers to the local variable
                        return false;
                    }
                } catch (UnsolvedSymbolException methodResolveException) {
                    getLogger().debug("Unable to resolve method " + m.getNameAsString()
                            + " and unable to resolve symbol for " + scopeName, methodResolveException);
                    return false;
                } catch (Exception methodResolveException) {
                    getLogger().debug("Unexpected exception when resolving method " + m.getNameAsString()
                            + " and unable to resolve symbol for " + scopeName, methodResolveException);
                }

                getLogger().debug("Unable to resolve symbol for " + scopeName, e);
            } catch (Exception e) {
                getLogger().debug("Unexpected exception when resolving symbol for " + scopeName, e);
            }
        } else if (scope instanceof FieldAccessExpr) {
            return nameMatches(scope, variableName);
        } else if (scope instanceof ObjectCreationExpr) {
            // Something like new Integer(10).intValue()
            return false;
        } else {
            getLogger().debug("Unknown scope type {}", scope);
        }
        return false;
    }

    private static boolean nameMatches(Expression expression, String variableName) {
        if (expression.isNameExpr()) {
            return expression.asNameExpr().getNameAsString().equals(variableName);
        } else if (expression.isFieldAccessExpr()) {
            return expression.asFieldAccessExpr().getNameAsString().equals(variableName);
        }

        getLogger().debug("Unknown type of scope expression {}: {}", expression.getClass().getName(), expression);
        return false;
    }

    static BlockStmt findBlock(Node node) {
        return findAncestor(node, BlockStmt.class);
    }

    static BlockStmt findBlockOrThrow(Node node) {
        return findAncestorOrThrow(node, BlockStmt.class);
    }

    static ClassOrInterfaceDeclaration findNameExprOrThrow(Node node) {
        return findAncestorOrThrow(node, ClassOrInterfaceDeclaration.class);
    }

    static <T> T findAncestor(Node node, Class<T> type) {
        return (T) findAncestor(node, n -> type.isAssignableFrom(n.getClass()));
    }

    @NonNull
    static <T> T findAncestorOrThrow(@NonNull Node node, Class<T> type) throws IllegalArgumentException {
        T ancestor = findAncestor(node, type);
        if (ancestor == null) {
            throw new IllegalArgumentException(
                    "Ancestor of type " + type.getName() + " not found for node of type " + node.getClass().getName());
        }
        return ancestor;
    }

    @NonNull
    static <T> T findOutermostAncestorOrThrow(@NonNull Node node, Class<T> type) throws IllegalArgumentException {
        T ancestor = (T) findOutermostAncestor(node, n -> type.isAssignableFrom(n.getClass()));
        if (ancestor == null) {
            throw new IllegalArgumentException(
                    "Ancestor of type " + type.getName() + " not found for node of type " + node.getClass().getName());
        }
        return ancestor;
    }

    static Node findAncestor(Node node, Predicate<Node> filter) {
        if (filter.test(node)) {
            return node;
        }
        return node.getParentNode().map(value -> findAncestor(value, filter)).orElse(null);
    }

    static Node findOutermostAncestor(Node node, Predicate<Node> filter) {
        return node.getParentNode().map(value -> findOutermostAncestor(value, filter)).orElseGet(() -> {
            if (filter.test(node)) {
                return node;
            }
            return null;
        });
    }

    static <T extends Node> Optional<T> findNodeOfType(Node node, int lineNumber, Class<T> type) {
        return findNodeOfType(node, lineNumber, type, (n) -> true);
    }

    static <T extends Node> Optional<T> findNodeOfType(Node node, int lineNumber, Class<T> type, Predicate<T> filter) {
        return Optional.ofNullable(
                (T) findNode(node, lineNumber, n -> type.isAssignableFrom(n.getClass()) && filter.test((T) n)));
    }

    static <T extends Node> List<T> findNodesOfType(Node node, int lineNumber, Class<T> type, Predicate<T> filter) {
        return (List<T>) findNodes(node, lineNumber, n -> type.isAssignableFrom(n.getClass()) && filter.test((T) n));
    }

    static Node findNode(Node startFrom, int lineNumber, Predicate<Node>... filters) {
        for (Node node : startFrom.getChildNodes()) {
            if (coversLine(node, lineNumber)) {
                Node found = findNode(node, lineNumber, filters);
                if (found != null) {
                    return found;
                }
            }
        }
        for (Predicate<Node> filter : filters) {
            if (filter.test(startFrom)) {
                return startFrom;
            }
        }

        return null;
    }

    static List<Node> findNodes(Node startFrom, int lineNumber, Predicate<Node> filter) {
        List<Node> allFound = new ArrayList<>();
        for (Node node : startFrom.getChildNodes()) {
            if (coversLine(node, lineNumber)) {
                List<Node> found = findNodes(node, lineNumber, filter);
                allFound.addAll(found);
            }
        }
        if (filter.test(startFrom)) {
            allFound.add(startFrom);
        }

        return allFound;
    }

    static boolean coversLine(Node node, int lineNumber) {
        return node.getRange().filter(value -> value.begin.line <= lineNumber && value.end.line >= lineNumber)
                .isPresent();
    }

    /**
     * Checks if the given component type has a method with the given name, taking
     * one parameter.
     *
     * @param type
     *            The component type
     * @param func
     *            The method name
     * @return {@code true} if the method exists, {@code false} otherwise
     */
    public static boolean hasSingleParameterMethod(Class<? extends Component> type, String func) {
        return Arrays.stream(type.getMethods())
                .anyMatch(method -> method.getName().equals(func) && method.getParameterCount() == 1);
    }

    /**
     * Finds the index of the statement that wraps the given node in the closest
     * block statement.
     *
     * @param node
     *            The node to find the wrapping statement for
     * @return The index of the statement in the block statement
     */
    public static int findBlockStatementIndex(Node node) {
        BlockStmt block = findAncestorOrThrow(node, BlockStmt.class);
        while (true) {
            Optional<Node> maybeParentNode = node.getParentNode();
            if (maybeParentNode.isEmpty()) {
                return -1;
            }
            Node parentNode = maybeParentNode.get();
            if (parentNode == block) {
                // Not using indexOf here as it uses equals, and there might be
                // two equal statements in the list
                for (int i = 0; i < block.getStatements().size(); i++) {
                    if (block.getStatement(i) == node) {
                        return i;
                    }
                }
            }
            node = parentNode;
        }
    }

    /**
     * Finds the index of the given node in the block statement.
     *
     * @param node
     *            The node to find the index for
     * @return The index of the node in the block statement or -1 if not found
     */
    public static int findDeclarationIndex(Node node) {
        ClassOrInterfaceDeclaration classOrInterfaceDeclaration = findAncestorOrThrow(node,
                ClassOrInterfaceDeclaration.class);
        while (true) {
            Optional<Node> maybeParentNode = node.getParentNode();
            if (maybeParentNode.isEmpty()) {
                return -1;
            }
            Node parentNode = maybeParentNode.get();
            if (parentNode == classOrInterfaceDeclaration) {
                // Not using indexOf here as it uses equals, and there might be
                // two equal statements in the list
                for (int i = 0; i < classOrInterfaceDeclaration.getMembers().size(); i++) {
                    if (classOrInterfaceDeclaration.getMembers().get(i) == node) {
                        return i;
                    }
                }
            }
            node = parentNode;
        }
    }

    /**
     * Finds a free variable name based on the component type.
     *
     * @param componentInfo
     *            The component info
     * @param block
     *            The block the variable will be used in
     * @return A free variable name
     */
    public static String findFreeVariableName(ComponentInfo componentInfo, BlockStmt block) {
        String base = componentInfo.type().getSimpleName().toLowerCase(Locale.ENGLISH);
        return findFreeVariableName(base, block);
    }

    /**
     * Finds a free variable name based on the given base name.
     *
     * @param base
     *            The base name
     * @param block
     *            The block the variable will be used in
     * @return A free variable name
     */
    public static String findFreeVariableName(String base, BlockStmt block) {
        return findFreeVariableName(base, findUsedVariableNames(block),
                findUsedNames(findAncestorOrThrow(block, ClassOrInterfaceDeclaration.class)));
    }

    /**
     * Finds a free field name based on the given base name.
     *
     * @param base
     *            The base name
     * @param block
     *            The ClassOrInterfaceDeclaration the variable will be used in
     * @return A free variable name
     */
    public static String findFreeVariableName(String base, ClassOrInterfaceDeclaration block) {
        return findFreeVariableName(base, findUsedNames(block));
    }

    public static String findFreeRecordName(String base, ClassOrInterfaceDeclaration block) {
        return findFreeVariableName(base, findUsedRecordNames(block));
    }

    private static String findFreeVariableName(String base, Set<String>... setsUsedNames) {

        Set<String> allUsedNames = new HashSet<>();
        for (Set<String> usedNames : setsUsedNames) {
            allUsedNames.addAll(usedNames);
        }

        base = base.length() > MAX_VARIABLE_NAME_LENGTH ? base.substring(0, MAX_VARIABLE_NAME_LENGTH) : base;

        if (base.isEmpty()) {
            base = "unknown";
        }
        String name = base;
        int i = 2;
        while (allUsedNames.contains(name)) {
            if (base.matches("[0-9]$")) {
                name = base + "_" + i;
            } else {
                name = base + i;
            }
            i++;
        }
        return name;
    }

    private static Set<String> findUsedVariableNames(BlockStmt block) {
        return block.findAll(VariableDeclarator.class).stream().map(VariableDeclarator::getNameAsString)
                .collect(java.util.stream.Collectors.toSet());
    }

    private static Set<String> findUsedNames(ClassOrInterfaceDeclaration block) {
        return block.getChildNodes().stream().flatMap(node -> {
            if (node instanceof FieldDeclaration fieldDeclaration) {
                return fieldDeclaration.getVariables().stream().map(VariableDeclarator::getNameAsString);
            } else if (node instanceof ClassOrInterfaceDeclaration subClass) {
                return Stream.of(subClass.getNameAsString());
            } else if (node instanceof RecordDeclaration recordDeclaration) {
                return Stream.of(recordDeclaration.getNameAsString());
            } else if (node instanceof EnumDeclaration enumDeclaration) {
                return Stream.of(enumDeclaration.getNameAsString());
            } else if (node instanceof MethodDeclaration methodDeclaration) {
                return Stream.of(methodDeclaration.getNameAsString());
            }
            return Stream.empty();
        }).collect(Collectors.toSet());
    }

    private static Set<String> findUsedRecordNames(ClassOrInterfaceDeclaration block) {
        return block.findAll(RecordDeclaration.class).stream().map(RecordDeclaration::getNameAsString)
                .collect(java.util.stream.Collectors.toSet());
    }

    /**
     * Removes the statement that wraps the given node.
     *
     * @param node
     *            the node to remove
     */
    public static void removeStatement(Node node) {
        JavaRewriterUtil.findAncestorOrThrow(node, Statement.class).remove();
    }

    /**
     * Removes the given node from a string concatenation expression.
     *
     * @param node
     *            the node to remove
     * @return {@code true} if the node was removed, {@code false} otherwise
     */
    public static boolean removeFromStringConcatenation(Node node) {
        Optional<Node> parent = node.getParentNode();
        if (parent.isEmpty()) {
            return false;
        }

        if (parent.get() instanceof BinaryExpr binaryExpr) {
            Expression otherExpr = binaryExpr.getLeft() == node ? binaryExpr.getRight() : binaryExpr.getLeft();
            if (otherExpr instanceof StringLiteralExpr) {
                binaryExpr.replace(otherExpr);
                return true;
            }
        }
        return false;
    }

    /**
     * Finds a node among the nodes which refer to the given component.
     *
     * @param nodes
     *            the nodes to search
     * @param componentDefinition
     *            the component definition
     * @return the first node that refers to the component, if any
     */
    public static Optional<Expression> findReference(NodeList<Expression> nodes, ComponentInfo componentDefinition) {
        return nodes.stream().filter(node -> {
            var createInfo = componentDefinition.getCreateInfoOrThrow();
            if (createInfo.getLocalVariableName() != null) {
                return isName(node, createInfo.getLocalVariableName());
            } else if (createInfo.getFieldName() != null) {
                return isName(node, createInfo.getFieldName()) || isFieldReference(node, createInfo.getFieldName());
            } else {
                // Inline
                return (node == componentDefinition.getCreateInfoOrThrow().getObjectCreationExpr());
            }
        }).findFirst();
    }

    private static boolean isFieldReference(Expression node, String s) {
        return node.isFieldAccessExpr() && node.asFieldAccessExpr().getNameAsString().equals(s);
    }

    private static boolean isName(Expression node, String s) {
        return node.isNameExpr() && node.asNameExpr().getNameAsString().equals(s);
    }

    /**
     * Finds the field declaration for the given field name in the class of the
     * given node.
     *
     * @param nodeInClass
     *            the node in the class
     * @param fieldName
     *            the field name
     * @return the field declaration
     * @throws IllegalArgumentException
     *             if the field is not found
     */
    public static FieldDeclaration findFieldDeclaration(Node nodeInClass, String fieldName) {
        ClassOrInterfaceDeclaration classDeclaration = findAncestorOrThrow(nodeInClass,
                ClassOrInterfaceDeclaration.class);

        return classDeclaration.getFieldByName(fieldName)
                .orElseThrow(() -> new IllegalArgumentException("No field found with name " + fieldName));
    }

    /**
     * Gets the setter name and value for the given component type, property and
     * value.
     *
     * @param componentType
     *            the component type
     * @param property
     *            the property
     * @param value
     *            the value
     * @return the setter name and value
     */
    public static JavaRewriter.SetterAndValue getSetterAndValue(Class<?> componentType, String property, Object value) {
        String setterName = getSetterName(property, componentType, true);

        if (FlowComponentQuirks.isInvertedBoolean(property, componentType)) {
            return new JavaRewriter.SetterAndValue(setterName, !Boolean.TRUE.equals(value));
        }
        Object propertyValue = getPropertyValue(componentType, setterName, property, value);
        return new JavaRewriter.SetterAndValue(setterName, propertyValue);
    }

    /**
     * Gets the setter name for the given property.
     *
     * @param property
     *            the property
     * @param type
     *            the component type
     * @param includeReactConversions
     *            whether to include React property name conversions in the setter
     *            name
     * @return the setter name
     */
    public static String getSetterName(String property, Class<?> type, boolean includeReactConversions) {
        if (includeReactConversions) {
            String setterName = FlowComponentQuirks.convertReactPropertyToJavaSetter(property, type);
            if (setterName != null) {
                return setterName;
            }
        }
        return Util.getSetterName(SharedUtil.dashSeparatedToCamelCase(property));
    }

    /**
     * Gets the camel case property name for the given setter.
     *
     * @param setter
     *            the setter name
     * @return the property name
     */
    public static String getPropertyName(String setter) {
        if (!setter.startsWith("set")) {
            throw new IllegalArgumentException("A setter name must start with 'set': " + setter);
        }
        return SharedUtil.firstToLower(setter.substring("set".length()));
    }

    /**
     * Gets the field or local variable name for the given component.
     *
     * @param componentInfo
     *            the component info
     * @return the field or local variable name
     */
    public static String getFieldOrVariableName(ComponentInfo componentInfo) {
        if (componentInfo.getCreateInfoOrThrow().getLocalVariableName() != null) {
            return componentInfo.getCreateInfoOrThrow().getLocalVariableName();
        } else if (componentInfo.getCreateInfoOrThrow().getFieldName() != null) {
            return componentInfo.getCreateInfoOrThrow().getFieldName();
        } else if (componentInfo.routeConstructor() != null) {
            return "this";
        }
        return null;
    }

    /**
     * Adds an import to the given compilation unit if it is not already imported.
     *
     * @param compilationUnit
     *            the compilation unit
     * @param qualifiedName
     *            the qualified name of the import
     */
    public static void addImport(CompilationUnit compilationUnit, String qualifiedName) {
        addImport(compilationUnit, qualifiedName, false, false);
    }

    /**
     * Adds an import to the given compilation unit if it is not already imported.
     *
     * @param compilationUnit
     *            the compilation unit
     * @param qualifiedName
     *            the qualified name of the import
     * @param isStatic
     *            whether the import is static
     * @param isAsterisk
     *            whether the import end with ".*"
     */
    public static void addImport(CompilationUnit compilationUnit, String qualifiedName, boolean isStatic,
            boolean isAsterisk) {
        String className;
        if (qualifiedName.contains("$")) {
            className = qualifiedName.replace("$", ".");
        } else {
            className = qualifiedName;
        }
        if (compilationUnit.getImports().stream().anyMatch(i -> i.getNameAsString().equals(className))) {
            return;
        }
        ImportDeclaration importDeclaration = new ImportDeclaration(className, isStatic, isAsterisk);
        compilationUnit.addImport(importDeclaration);
    }

    /**
     * Converts the given string into a valid Java identifier.
     *
     * @param str
     *            the string
     * @param maxLength
     *            the maximum length of the identifier, or 0 for unlimited
     * @return the Java identifier
     */
    public static String getJavaIdentifier(String str, int maxLength) {
        if (str == null || str.isEmpty()) {
            return "empty";
        }

        // Normalize the string to remove accents and other diacritical marks
        String normalized = Normalizer.normalize(str, Normalizer.Form.NFD).replaceAll(NORMALIZED_PATTERN.pattern(), "");

        // Remove all non-alphanumeric characters and replace spaces with
        // underscores
        String sanitized = normalized.replaceAll(SANITIZE_PATTERN.pattern(), "").replaceAll(SPACE_PATTERN.pattern(),
                "-");

        String camelCase;
        // Convert to camelCase (start with a lowercase letter)
        if (sanitized.matches("^(.+)[A-Z][a-z].*") && !sanitized.contains("-")) {
            // If it already is camelCased, keep it
            camelCase = SharedUtil.firstToLower(sanitized);
        } else {
            camelCase = SharedUtil.dashSeparatedToCamelCase(sanitized.toLowerCase(Locale.ENGLISH));
        }

        // Ensure the string is within the specified length
        if (maxLength > 0 && camelCase.length() > maxLength) {
            camelCase = camelCase.substring(0, maxLength);
        }

        // Ensure the variable name starts with a lowercase letter
        if (!camelCase.isEmpty() && !Character.isLowerCase(camelCase.charAt(0))) {
            camelCase = camelCase.substring(0, 1).toLowerCase(Locale.ROOT) + camelCase.substring(1);
        }

        if (isReservedJavaKeyword(camelCase)) {
            camelCase += "_";
        }

        StringBuilder sb = new StringBuilder();
        for (int i = 0; i < camelCase.length(); i++) {
            if (Character.isJavaIdentifierStart(camelCase.charAt(i))
                    || (sb.length() > 0 && Character.isJavaIdentifierPart(camelCase.charAt(i)))) {
                sb.append(camelCase.charAt(i));
            }
        }

        if (sb.isEmpty()) {
            // If e.g. there are only numbers, there are no valid characters at all as the
            // first character
            // cannot be a number in Java
            sb.append("a");
            for (int i = 0; i < camelCase.length(); i++) {
                if (Character.isJavaIdentifierPart(camelCase.charAt(i))) {
                    sb.append(camelCase.charAt(i));
                }
            }
        }
        return sb.toString();
    }

    private static boolean isReservedJavaKeyword(String identifier) {
        return javaKeywords.contains(identifier);
    }

    /**
     * Checks if the given component type has the given method.
     *
     * @param type
     *            the component type
     * @param methodName
     *            the method to check for
     * @return {@code true} if the component has the method, {@code false} otherwise
     */
    public static boolean hasMethod(Class<?> type, String methodName) {
        return Arrays.stream(type.getMethods()).anyMatch(method -> method.getName().equals(methodName));
    }

    private static Object getPropertyValue(Class<?> componentType, String setterName, String prop, Object value) {
        value = FlowComponentQuirks.componentSpecificValueMapping(componentType, prop, value);
        if (value == null) {
            return null;
        }

        if (setterName.startsWith("getElement().getThemeList()")) {
            return value;
        }
        if (setterName.equals("setStyle")) {
            return value;
        }
        if (setterName.equals("setAdditionalOptions")) {
            return value;
        }
        if (componentType.getName().equalsIgnoreCase("com.vaadin.flow.component.charts.model.ListSeries")
                && setterName.equals("setPlotOptions")) {
            return value;
        }
        if (hasSetterForType(componentType, setterName, value.getClass())) {
            return value;
        }
        if (componentType.getName().equalsIgnoreCase("com.vaadin.flow.component.charts.Chart")
                && hasSetterForType(JavaReflectionUtil.getClass("com.vaadin.flow.component.charts.model.Configuration"),
                        setterName, value.getClass())) {
            return value;
        }
        if (hasSetterForType(componentType, setterName, Component.class) && value instanceof JavaComponent) {
            return value;
        }
        // Handle List with single JavaComponent for Component setters (from JSON
        // parsing where component props are always arrays)
        if (hasSetterForType(componentType, setterName, Component.class) && value instanceof List<?> list
                && !list.isEmpty() && list.get(0) instanceof JavaComponent) {
            if (list.size() == 1) {
                // Silently accept single element list for Component setter
                return list.get(0);
            } else {
                // Throw exception if multiple components provided for single Component setter
                throw new IllegalArgumentException(String.format(
                        "Setter %s expects a single Component but received %d components", setterName, list.size()));
            }
        }
        if (hasSetterForType(componentType, setterName, Component[].class) && value instanceof JavaComponent[]) {
            return value;
        }
        // Handle List of JavaComponents (from JSON parsing) for Component[] setters
        if (hasSetterForType(componentType, setterName, Component[].class) && value instanceof List<?> list
                && !list.isEmpty() && list.get(0) instanceof JavaComponent) {
            return list.toArray(new JavaComponent[0]);
        }
        // Handle single JavaComponent for Component[] varargs setters
        if (hasSetterForType(componentType, setterName, Component[].class) && value instanceof JavaComponent) {
            return new JavaComponent[] { (JavaComponent) value };
        }
        Object enumValue = getEnumValue(componentType, setterName, value.toString());
        if (enumValue != null) {
            return enumValue;
        }
        // Setter to be Split in several calls because it does not have setItems
        // or addItems
        // but just single addItem method. i.e.: MenuBar
        if (setterName.equals("addItem") && value instanceof ArrayList<?>) {
            return value;
        }
        List<Method> setters = findSetters(componentType, setterName);
        for (Method setter : setters) {
            Class<?> setterType = getSetterType(setter);
            if (setterType == Instant.class) {
                Instant instant = parseInstant(((String) value));

                if (instant == null) {
                    throw new IllegalArgumentException("Unable to parse Instant from " + value);
                }
                NameExpr scope = new NameExpr(Instant.class.getName());
                return new MethodCallExpr(scope, "ofEpochSecond").addArgument(toExpression(instant.getEpochSecond()));
            } else if (setterType == Integer.class && value instanceof String) {
                try {
                    return Integer.parseInt((String) value);
                } catch (NumberFormatException e) {
                    // Throw below as it was not an integer
                }
            } else if (double.class.equals(setterType) && value instanceof Integer) {
                // setMin setMax
                return value;
            } else if (setterType == LocalTime.class && value instanceof String) {
                try {
                    return LocalTime.parse((String) value);
                } catch (DateTimeParseException e) {
                    // Throw below as it was not a LocalTime
                }
            }
        }
        throw new IllegalArgumentException(String.format(
                "Unable to find a suitable setter for %s.%s for value of type %s. Setter will be ignored.",
                componentType.getName(), setterName, value.getClass().getName()));
    }

    private static Instant parseInstant(String value) {
        if (value.equals("now") || value.equals("right now")) {
            return Instant.now();
        } else if (value.equals("yesterday")) {
            return Instant.now().minusSeconds(60 * 60 * 24L);
        }

        try {
            return Instant.parse(value);
        } catch (DateTimeParseException e) {
            // Try the other options
        }
        try {
            return DateTimeFormatter.ofPattern("dd.MM.yyyy HH:mm").parse(value, LocalDateTime::from)
                    .toInstant(ZoneOffset.UTC);
        } catch (DateTimeParseException e) {
            // Try the other options
        }
        try {
            return DateTimeFormatter.ofPattern("dd.MM.yyyy").parse(value, LocalDate::from).atTime(0, 0)
                    .toInstant(ZoneOffset.UTC);
        } catch (DateTimeParseException e) {
            // Try the other options
        }

        return null;
    }

    public static boolean hasSetterForType(Class<?> componentType, String setterName, Class<?> valueType) {
        List<Method> setters = findSetters(componentType, setterName);
        for (Method method : setters) {
            Class<?> setterType = getSetterType(method);
            if (setterType == Object.class && method.isBridge()) {
                continue;
            }
            if (setterType.isAssignableFrom(valueType)) {
                return true;
            }
            if (WRAPPER_TYPE_MAP.containsKey(valueType)
                    && setterType.isAssignableFrom(WRAPPER_TYPE_MAP.get(valueType))) {
                return true;
            }
        }
        return false;
    }

    public static Enum<?> getEnumValue(Class<?> componentType, String setterName, String value) {
        List<Method> setters = findSetters(componentType, setterName);
        for (Method setter : setters) {
            Class<?> setterType = getSetterType(setter);
            if (!setterType.isEnum()) {
                continue;
            }
            for (Object enumValue : setterType.getEnumConstants()) {
                if (enumValue.toString().equals(value.toUpperCase(Locale.ENGLISH))) {
                    return (Enum<?>) enumValue;
                }
            }
        }
        return null;
    }

    private static List<Method> findSetters(Class<?> componentType, String setterName) {
        return Arrays.stream(componentType.getMethods()).filter(method -> method.getName().equals(setterName))
                .filter(method -> method.getParameterCount() == 1).toList();
    }

    private static Class<?> getSetterType(Method setter) {
        return setter.getParameterTypes()[0];
    }

    /**
     * Removes the given arguments from the given method calls.
     *
     * @param methods
     *            the method calls to process
     * @param argumentsToRemove
     *            the arguments to remove
     * @param removeMethodIfNoArgs
     *            whether to remove the method call if no arguments are left
     */
    public static void removeArgumentCalls(List<MethodCallExpr> methods, List<? extends Expression> argumentsToRemove,
            boolean removeMethodIfNoArgs) {
        for (MethodCallExpr methodCall : methods) {
            removeArgumentCalls(methodCall, argumentsToRemove, removeMethodIfNoArgs);
        }
    }

    /**
     * Removes the given arguments from the given method call.
     *
     * @param methodCallExpr
     *            the method call
     * @param argumentsToRemove
     *            the arguments to remove
     * @param removeMethodIfNoArgs
     *            whether to remove the method call if no arguments are left
     * @return {@code true}
     */
    public static boolean removeArgumentCalls(MethodCallExpr methodCallExpr,
            List<? extends Expression> argumentsToRemove, boolean removeMethodIfNoArgs) {

        List<Expression> willRemoveArg = new ArrayList<>();
        for (Expression argument : methodCallExpr.getArguments()) {
            for (Expression argToRemove : argumentsToRemove) {
                if (equalsByNameAsString(argument, argToRemove)) {
                    willRemoveArg.add(argument);
                    break;
                }
            }
        }
        for (Expression argument : willRemoveArg) {
            boolean remove = argument.remove();
            if (!remove) {
                throw new IllegalArgumentException(
                        "Argument remove has failed for " + methodCallExpr.getNameAsString() + ", " + argument);
            }
        }
        if (removeMethodIfNoArgs && methodCallExpr.getArguments().isEmpty()) {
            removeStatement(methodCallExpr);
        }
        return true;
    }

    /**
     * Removes the argument at the given index from the given method call.
     *
     * @param methodCallExpr
     *            the method call
     * @param index
     *            the index of the argument to remove
     * @param removeMethodIfNoArgs
     *            whether to remove the method call if no arguments are left
     * @return {@code true}
     */
    public static boolean removeArgumentCall(MethodCallExpr methodCallExpr, int index, boolean removeMethodIfNoArgs) {
        methodCallExpr.getArguments().remove(index);
        if (removeMethodIfNoArgs && methodCallExpr.getArguments().isEmpty()) {
            removeStatement(methodCallExpr);
        }
        return true;
    }

    /**
     * Finds the insert location before of given expression.
     *
     * @param node
     *            Reference node
     * @return Returns code block and index
     */
    public static InsertionPoint findLocationBefore(Node node) {
        InsertionPoint loc = findLocationAfter(node);
        if (loc.getBlock() != null) {
            return new InsertionPoint(loc.getBlock(), loc.getIndex() - 1);
        } else {
            return new InsertionPoint(loc.getDeclaration(), loc.getIndex() - 1);
        }
    }

    /**
     * Finds the insert location after of given expression.
     *
     * @param node
     *            Reference node
     * @return the insert location
     */
    public static InsertionPoint findLocationAfter(Node node) {
        try {
            BlockStmt insertBlock = findBlockOrThrow(node);
            int insertIndex = JavaRewriterUtil.findBlockStatementIndex(node) + 1;
            return new InsertionPoint(insertBlock, insertIndex);
        } catch (IllegalArgumentException e) {
            ClassOrInterfaceDeclaration insertBlock = findNameExprOrThrow(node);
            int insertIndex = JavaRewriterUtil.findDeclarationIndex(node) + 1;
            return new InsertionPoint(insertBlock, insertIndex);
        }
    }

    /**
     * Finds the insert location at the end of the code block containing the given
     * statement.
     *
     * @param statement
     *            The reference statement
     * @return the insert location
     */
    public static InsertionPoint findLocationAtEnd(Statement statement) {
        BlockStmt insertBlock = findBlockOrThrow(statement);
        int insertIndex = insertBlock.getStatements().size();
        return new InsertionPoint(insertBlock, insertIndex);
    }

    /**
     * Adds a field declaration after the given reference field.
     *
     * @param newField
     *            the new field declaration
     * @param reference
     *            the reference field
     */
    public static void addFieldAfter(FieldDeclaration newField, FieldDeclaration reference) {
        ClassOrInterfaceDeclaration classDeclaration = JavaRewriterUtil.findAncestorOrThrow(reference,
                ClassOrInterfaceDeclaration.class);
        NodeList<BodyDeclaration<?>> members = classDeclaration.getMembers();
        int referenceIndex = members.indexOf(reference);
        members.add(referenceIndex + 1, newField);
    }

    /**
     * Clones the node in a way that does not mess up the lexical printer.
     *
     * @param node
     *            the node to clone
     * @param <T>
     *            the type of the node
     * @return the cloned node
     */
    public static <T extends Node> T clone(T node) {
        T newNode = (T) node.clone();

        clearData(newNode);
        return newNode;
    }

    private static void clearData(Node node) {
        HashSet<DataKey<?>> dataKeys = new HashSet<>(node.getDataKeys());
        dataKeys.forEach(node::removeData);
        for (Node child : node.getChildNodes()) {
            clearData(child);
        }
    }

    /**
     * Finds the attach argument reference of the given component.
     *
     * @param component
     *            the component
     * @return the attach argument reference, if found
     */
    public static Optional<Expression> getAttachArgument(ComponentInfo component) {
        if (component.componentAttachInfoOptional().isEmpty()) {
            return Optional.empty();
        }
        ComponentAttachInfo componentAttachInfo = component.componentAttachInfoOptional().get();
        AttachExpression attachCall = componentAttachInfo.getAttachCall();
        if (attachCall == null) {
            return Optional.empty();
        }

        Optional<Expression> attachArg = findReference(attachCall.getNodeWithArguments().getArguments(), component);
        if (attachArg.isEmpty() && attachCall.expression().isMethodCallExpr()) {
            // add(new VerticalLayout(new Button("something"));
            MethodCallExpr methodCallExpr = attachCall.expression().asMethodCallExpr();
            for (Expression argument : methodCallExpr.getArguments()) {
                if (argument instanceof ObjectCreationExpr objectCreationExpr) {
                    attachArg = findReference(objectCreationExpr.getArguments(), component);
                    if (attachArg.isPresent()) {
                        return attachArg;
                    }
                }
            }
        }
        return attachArg;
    }

    /**
     * Find the attach argument reference of given component or throws
     * {@code IllegalArgumentException} if not found.
     *
     * @param component
     * @return attach expression
     */
    public static Expression getAttachArgumentOrThrow(ComponentInfo component) {
        return getAttachArgument(component)
                .orElseThrow(() -> new IllegalArgumentException("No attach argument found for the component"));
    }

    /**
     * Sets the name expression scope for the given method call. For a simple call
     * like `foo.setBar()` this changes `foo` and for nested calls like
     * `foo.baz().fo().a().b()` it also changes `foo`.
     *
     * @param newCall
     *            the method call to change
     * @param nameExpr
     *            the new scope
     * @return {@code true} if the scope was changed, {@code false} otherwise
     */
    public static boolean setNameExprScope(MethodCallExpr newCall, NameExpr nameExpr) {
        Optional<Expression> scope = newCall.getScope();
        if (scope.isEmpty()) {
            return false;
        }
        if (scope.get().isMethodCallExpr()) {
            return setNameExprScope(scope.get().asMethodCallExpr(), nameExpr);
        }
        newCall.setScope(nameExpr);
        return true;
    }

    /**
     * Checks if the given type is equal to the given reflection type.
     *
     * @param javaParserType
     *            the Java parser type
     * @param javaReflectionType
     *            the Java reflection type
     * @return {@code true} if the types are equal, {@code false} otherwise
     */
    public static boolean typesEqual(ResolvedType javaParserType, Class<?> javaReflectionType) {
        if (javaParserType.isReferenceType()) {
            ResolvedReferenceType refType = javaParserType.asReferenceType();
            return refType.getQualifiedName().equals(javaReflectionType.getName().replace("$", "."));
        }
        if (javaParserType.isArray() && javaReflectionType.isArray()) {
            ResolvedArrayType arrayType = javaParserType.asArrayType();
            ResolvedType resolvedArrayComponentType = arrayType.getComponentType();
            Class<?> javaArrayComponentType = javaReflectionType.componentType();
            return typesEqual(resolvedArrayComponentType, javaArrayComponentType);
        }
        if (javaParserType.isTypeVariable()) {
            ResolvedTypeVariable typeVariable = javaParserType.asTypeVariable();
            ResolvedTypeParameterDeclaration typeParameter = typeVariable.asTypeParameter();
            if (typeParameter instanceof ReflectionTypeParameter
                    && javaReflectionType.getName().equals("java.lang.Object")) {
                return true;
            }
        }
        if (javaParserType.isPrimitive() && javaReflectionType.isPrimitive()) {
            ResolvedPrimitiveType primitive = javaParserType.asPrimitive();
            return JavaReflectionUtil.getClass(primitive.describe()).equals(javaReflectionType);
        }
        getLogger().debug("Do not know how to compare type {} with {}", javaParserType, javaReflectionType);
        return false;
    }

    /**
     * Compares two {@link ResolvedMethodDeclaration} instances to determine if they
     * are semantically equivalent in terms of their signature and declaring
     * context.
     *
     * @param methodDeclaration1
     *            the first method to compare
     * @param methodDeclaration2
     *            the second method to compare
     * @return {@code true} if both methods are considered equal based on the
     *         criteria, {@code false} otherwise
     */
    public static boolean typesEqual(ResolvedMethodDeclaration methodDeclaration1,
            ResolvedMethodDeclaration methodDeclaration2) {
        return methodDeclaration1.getQualifiedSignature().equals(methodDeclaration2.getQualifiedSignature());
    }

    /**
     * Creates a Div creation expression <code>new Div()</code> with given
     * arguments. Given arguments will be cloned and clones will be added to Div
     * constructor.
     *
     * @param compilationUnit
     *            current compilation unit
     * @param args
     *            Argument expressions
     * @return Div object creation expression
     */
    public static ObjectCreationExpr getWrapperDivObjectCreationExpr(CompilationUnit compilationUnit,
            NodeList<Expression> args) {
        ClassOrInterfaceType classOrInterfaceType = new ClassOrInterfaceType();
        classOrInterfaceType.setName("Div");
        ObjectCreationExpr divCreationExpression = new ObjectCreationExpr();
        divCreationExpression.setType(classOrInterfaceType);
        JavaRewriterUtil.addImport(compilationUnit, "com.vaadin.flow.component.html.Div");
        if (args != null) {
            for (Expression arg : args) {
                divCreationExpression.getArguments().add(JavaRewriterUtil.clone(arg));
            }
        }
        return divCreationExpression;
    }

    private static boolean testArgumentInConstructors(Class<?> clazz, int argIndex, Predicate<Class<?>> argPredicate) {
        Constructor<?>[] constructors = clazz.getConstructors();
        for (Constructor<?> constructor : constructors) {
            Class<?>[] parameterTypes = constructor.getParameterTypes();
            Class<?> parameterToCheck = null;
            if (parameterTypes.length != 0) {
                if (argIndex < parameterTypes.length) {
                    parameterToCheck = parameterTypes[argIndex];
                } else {
                    parameterToCheck = parameterTypes[parameterTypes.length - 1];
                }
            }
            if (parameterToCheck != null && argPredicate.test(parameterToCheck)) {
                return true;
            }
        }
        return false;
    }

    /***
     * Resolves object creation expression to access Class API to check if given
     * argument in the argIndex exactly equals to
     * <code>com.vaadin.flow.component.Component</code>
     *
     * @param expr
     *            Object creation expression of a statement e.g.
     *            <code>new Div(a,b,c)</code>
     * @param argIndex
     *            Argument to look for. If value exceeds the argument in the
     *            constructor, it looks for the last argument
     * @return true if argument class is com.vaadin.flow.component.Component or
     *         resolving of the expression fails, false otherwise.
     */
    public static boolean isConstructorArgumentFlowComponentType(ObjectCreationExpr expr, int argIndex) {
        try {
            ResolvedType resolve = expr.getType().resolve();
            Class<?> clazz = JavaReflectionUtil.getClass(resolve.describe());
            return testArgumentInConstructors(clazz, argIndex,
                    argClazz -> "com.vaadin.flow.component.Component".equals(argClazz.getName()));
        } catch (UnsolvedSymbolException ex) {
            getLogger().warn("Could not resolve argument of expression = {}", expr, ex);
            return false;
        }
    }

    /**
     * Resolves object creation expression to access Class API to get
     * argument.isArray() value of given argument index.
     *
     * @param expr
     *            Object creation expression of a statement e.g.
     *            <code>new Div(a,b,c)</code>
     * @param argIndex
     *            Argument to look for. If value exceeds the argument in the
     *            constructor, it looks for the last argument
     * @return true when array is an argument or resolving of expression fails,
     *         false otherwise.
     */
    public static boolean constructorArgumentIsArray(ObjectCreationExpr expr, int argIndex) {
        try {
            ResolvedType resolve = expr.getType().resolve();
            Class<?> clazz = JavaReflectionUtil.getClass(resolve.describe());
            boolean isArray = testArgumentInConstructors(clazz, argIndex, Class::isArray);
            if (isArray) {
                return true;
            }
        } catch (UnsolvedSymbolException ex) {
            getLogger().warn("Could not solve constructor argument index of {}", expr, ex);
            return true;
        }
        return false;
    }

    /**
     * Checks given two expression are in different blocks. If any expression is not
     * a descendant of a method, it returns false.
     *
     * @param expr1
     *            first expression
     * @param expr2
     *            second expression
     * @return true if expressions present in different methods , false otherwise
     */
    public static boolean expressionsInDifferentBlocks(Expression expr1, Expression expr2) {
        BlockStmt blockStmtOfExpr1 = JavaRewriterUtil.findAncestor(expr1, BlockStmt.class);
        // checking reference and moved components are in the same method call
        if (blockStmtOfExpr1 != null) {
            BlockStmt blockStmtOfExpr2 = JavaRewriterUtil.findAncestor(expr2, BlockStmt.class);
            return blockStmtOfExpr2 != null && !blockStmtOfExpr1.equals(blockStmtOfExpr2);
        }
        return false;
    }

    /**
     * Appends new expression to original one as sibling in a block statement.
     *
     * @param original
     *            Original expression
     * @param newExpression
     *            New expression
     * @return true if added, otherwise false.
     */
    public static boolean appendExpressionAsNextSiblingInBlockAncestor(Node original, Statement newExpression) {
        BlockStmt ancestor = JavaRewriterUtil.findAncestor(original, BlockStmt.class);
        if (ancestor == null) {
            return false;
        }
        if (original.getRange().isEmpty()) {
            return false;
        }
        Range originalRange = original.getRange().get();

        for (int i = 0; i < ancestor.getStatements().size(); i++) {
            Statement statement = ancestor.getStatement(i);
            if (statement.getRange().isPresent()) {
                Range stmtRange = statement.getRange().get();
                if (originalRange.overlapsWith(stmtRange)) {
                    ancestor.addStatement(i + 1, newExpression);
                    return true;
                }
            }
        }
        return false;
    }

    /**
     * Returns the scope by ignoring the {@code getContent()} if component is in a
     * composite container. Otherwise, returns the scope of method. <br>
     *
     * <ul>
     * <li>{@code getContent().add(...)} returns {@code empty}
     * <li>{@code this.getContent().add(...)} returns {@code this}
     * <li>{@code setSomething()} returns {@code empty}
     * <li>{@code abc.setSomething()} returns {@code abc}
     * <li>{@code this.setSomething()} returns {@code this}
     * </ul>
     *
     * @param componentInfo
     *            Component Info
     * @param expr
     *            Method to get scope
     * @return Scope of the method, empty otherwise.
     */
    public static Optional<Expression> getScopeIgnoreComposite(ComponentInfo componentInfo, MethodCallExpr expr) {
        if (componentInfo.containerComposite()) {
            Optional<Expression> scopeOpt = expr.getScope();
            if (scopeOpt.isPresent()) {
                Expression expression = scopeOpt.get();
                if (expression.isMethodCallExpr()
                        && ((MethodCallExpr) expression).getNameAsString().equals("getContent")) {
                    return getScopeIgnoreComposite(componentInfo, expression.asMethodCallExpr());
                }
            }
            return Optional.empty();
        }
        return expr.getScope();
    }

    /**
     * Checks if the given node is a child of a Composite
     *
     * @param node
     *            Any node in the tree
     * @return {@code true} if child of a composite , {@code false} otherwise
     */
    public static boolean isNodeInCompositeClass(Node node) {
        ClassOrInterfaceDeclaration classDefinition = JavaRewriterUtil.findAncestorOrThrow(node,
                ClassOrInterfaceDeclaration.class);
        Class<?> clazz;
        try {
            clazz = JavaReflectionUtil.getClass(classDefinition.resolve().getQualifiedName());
        } catch (IllegalArgumentException e) {
            // this happens in unit tests where test cases are written in strings instead of
            // classes
            getLogger().debug("Could not resolve class = {}", classDefinition.getNameAsString(), e);
            return false;
        }

        while (clazz != null) {
            if (clazz.equals(Composite.class)) {
                return true;
            }
            clazz = clazz.getSuperclass();
        }
        return false;
    }

    /**
     * Finds constructor for the given component from the constructor that is
     * provided by {@link Resolvable} implementation
     *
     * @param componentType
     *            Component class
     * @param resolvable
     *            Resolvable constructor declaration node
     * @return Constructor if found, empty otherwise.
     */
    public static Optional<Constructor<?>> findConstructor(Class<? extends Component> componentType,
            Resolvable<ResolvedConstructorDeclaration> resolvable) {
        try {
            ResolvedConstructorDeclaration resolved = resolvable.resolve();
            return findCandidateConstructor(componentType, resolved);
        } catch (UnsolvedSymbolException e) {
            getLogger().debug("Unable to resolve constructor for {}: {}", componentType.getName(), resolvable, e);
            return Optional.empty();
        }
    }

    private static Optional<Constructor<?>> findCandidateConstructor(Class<? extends Component> componentType,
            ResolvedConstructorDeclaration resolvedConstructorDeclaration) {
        List<Constructor<?>> candidates = Arrays.stream(componentType.getDeclaredConstructors())
                .filter(c -> c.getParameterCount() == resolvedConstructorDeclaration.getNumberOfParams()).filter(c -> {
                    for (int i = 0; i < resolvedConstructorDeclaration.getNumberOfParams(); i++) {
                        ResolvedType paramType = resolvedConstructorDeclaration.getParam(i).getType();
                        if (!JavaRewriterUtil.typesEqual(paramType, c.getParameterTypes()[i])) {
                            return false;
                        }
                    }
                    return true;
                }).toList();
        if (candidates.isEmpty()) {
            return Optional.empty();
        } else if (candidates.size() > 1) {
            throw new IllegalStateException(
                    "Multiple constructors found for " + componentType.getName() + ": " + candidates);
        } else {
            return Optional.of(candidates.get(0));
        }
    }

    /**
     * Creates the convenient component constructor for a new JavaComponent to be
     * added to the code.
     *
     * @return ObjectCreationExpr for Component
     */
    public static ObjectCreationExpr createComponentConstructor(JavaComponent javaComponent,
            ClassOrInterfaceType componentType, String dataEntityRecordName, Consumer<String> importAdder) {

        ObjectCreationExpr constructor = null;

        if ((componentType.getName().asString().equals("Grid") || componentType.getName().asString().equals("TreeGrid"))
                && !javaComponent.getItemsFromProperty().isEmpty()) {
            // Extract the type argument, which should be DataEntity
            Optional<Type> typeArgument = componentType.getTypeArguments()
                    .flatMap(typeArguments -> typeArguments.stream().findFirst());
            ClassExpr classExpr = new ClassExpr(
                    typeArgument.orElse(StaticJavaParser.parseClassOrInterfaceType(dataEntityRecordName)));
            constructor = new ObjectCreationExpr(null, componentType, new NodeList<>(classExpr));
        } else if (componentType.getName().asString().equals("SideNavItem") && javaComponent.innerText().isEmpty()) {
            // SideNavItem does not have an empty constructor..
            constructor = new ObjectCreationExpr(null, componentType, new NodeList<>(new StringLiteralExpr("")));
        } else if (componentType.getName().asString().equals("Chart") && javaComponent.props().containsKey("type")) {
            // Chart has no setter for type...
            constructor = new ObjectCreationExpr(null, componentType, new NodeList<>(
                    new NameExpr("ChartType" + "." + codeToUpperCase(javaComponent.props().get("type").toString()))));
            importAdder.accept(FlowComponentQuirks.CHART_TYPE);
            javaComponent.props().remove("type");
        } else {
            constructor = new ObjectCreationExpr(null, componentType, new NodeList<>());
        }
        return constructor;
    }

    private static String codeToUpperCase(String code) {
        return code.toUpperCase(Locale.ENGLISH);
    }

    /**
     * Returns the single String parameter name from mapped properties of the
     * component that matches an existing single String parameter constructor of the
     * component
     *
     * @param type
     *            The component type
     * @param setters
     *            The setters of the component
     * @return The name of the property that is the setter of the constructor
     */
    public static Optional<String> getSingleStringParamConstructor(ClassOrInterfaceType type, Set<String> setters) {
        Class<?> componentType = JavaReflectionUtil.getClass(type.getNameWithScope());
        Optional<Constructor<?>> constructor = Arrays.stream(componentType.getDeclaredConstructors())
                .filter(c -> c.getParameterCount() == 1 && c.getParameterTypes()[0] == String.class).findFirst();
        if (constructor.isEmpty()) {
            return Optional.empty();
        }
        String mappedProperty = getMappedProperty(constructor.get(), 0);
        if (mappedProperty != null && setters.contains(mappedProperty)) {
            return Optional.of(mappedProperty);
        }
        return Optional.empty();
    }

    /**
     * Returns a property used in a constructor at the given index
     *
     * @param c
     *            The constructor
     * @param propertyIndex
     *            The index of the property
     * @return The property name
     */
    public static String getMappedProperty(Constructor<?> c, int propertyIndex) {
        return ConstructorAnalyzer.get().getMappings(c).entrySet().stream()
                .filter(entry -> entry.getKey() == propertyIndex).map(Map.Entry::getValue).findFirst().orElse(null);
    }

    /**
     * Analyzes bytecode of the given component to search for method with methodName
     * is called in an empty constructor
     *
     * @param componentInfo
     *            Component info to get empty constructor
     * @param methodName
     *            The method name to search for.
     * @return Returns true when method call found in constructor, false when
     *         constructor is not found or method is not present
     */
    public static boolean containsMethodInEmptyConstructorCall(ComponentInfo componentInfo, String methodName) {
        Optional<Constructor<?>> constructor = Arrays.stream(componentInfo.type().getDeclaredConstructors())
                .filter(c -> c.getParameterCount() == 0).findFirst();
        if (constructor.isPresent()) {
            List<String> methodCallsInConstructor = ConstructorAnalyzer.get()
                    .getMethodCallsInConstructor(constructor.get());
            return methodCallsInConstructor.contains(methodName);
        }
        return false;
    }

    /**
     * Provides a free name to be used when adding a new component.
     *
     * @param javaComponent
     *            The JavaComponent to be added
     * @param type
     *            The type of the component
     * @param insertionPoint
     *            The insertion point
     * @return A free variable name
     */
    public static String generateVariableName(JavaComponent javaComponent, ClassOrInterfaceType type,
            InsertionPoint insertionPoint) {
        if (javaComponent.metadata() != null) {
            if (javaComponent.metadata().getFieldVariableName() != null) {
                return javaComponent.metadata().getFieldVariableName();
            }
            if (javaComponent.metadata().getLocalVariableName() != null) {
                return javaComponent.metadata().getLocalVariableName();
            }
        }

        String textProperty = (String) javaComponent.props().get("text");
        String innerText = javaComponent.innerText().orElse(null);
        String labelProperty = (String) javaComponent.props().get("label");
        String idProperty = (String) javaComponent.props().get("id");
        String variableBaseName = generateVariableBaseName(type.getNameAsString(), idProperty, textProperty, innerText,
                labelProperty);

        return insertionPoint.getFreeVariableName(variableBaseName);
    }

    private static String generateVariableBaseName(String classSimpleName, String idProperty, String textProperty,
            String innerText, String labelProperty) {
        if (idProperty != null) {
            return JavaRewriterUtil.getJavaIdentifier(idProperty, 0);
        } else if (innerText != null) {
            return JavaRewriterUtil.getJavaIdentifier(innerText, 0);
        } else if (textProperty != null) {
            return JavaRewriterUtil.getJavaIdentifier(textProperty, 0);
        } else if (labelProperty != null) {
            return JavaRewriterUtil.getJavaIdentifier(labelProperty, 0);
        }

        return classSimpleName.toLowerCase(Locale.ENGLISH);
    }

    /**
     * Regenerates the variable name for the given component.
     *
     * @param componentInfo
     *            the component to regenerate the variable name for
     * @param newText
     *            the text of the component
     * @param newLabel
     *            the label of the component
     * @return the new variable name
     */
    public static String regenerateVariableName(ComponentInfo componentInfo, String newText, String newLabel) {
        String variableBaseName = generateVariableBaseName(componentInfo.type().getSimpleName(), null, null, newText,
                newLabel);
        String newVariableName = null;
        if (componentInfo.getCreateInfoOrThrow().getLocalVariableName() != null) {
            newVariableName = findFreeVariableName(variableBaseName,
                    findBlock(componentInfo.getCreateInfoOrThrow().getLocalVariableDeclarator()));
        } else if (componentInfo.getCreateInfoOrThrow().getFieldName() != null) {
            newVariableName = findFreeVariableName(variableBaseName, findAncestorOrThrow(
                    componentInfo.getCreateInfoOrThrow().getFieldDeclaration(), ClassOrInterfaceDeclaration.class));
        }

        if (newVariableName != null) {
            renameVariable(componentInfo, newVariableName);
        }
        return newVariableName;
    }

    /**
     * Renames the variable of the given component to the given name.
     *
     * @param componentInfo
     *            the component to rename
     * @param newVariableName
     *            the new variable name
     */
    private static void renameVariable(ComponentInfo componentInfo, String newVariableName) {
        VariableDeclarator variableDeclarator = componentInfo.getVariableDeclarator();

        findMethodCalls(componentInfo).forEach(methodCall -> {
            Optional<Expression> maybeScope = methodCall.getScope();
            if (maybeScope.isEmpty()) {
                return;
            }
            Expression scope = maybeScope.get();

            if (scope instanceof Resolvable<?> resolvable
                    && scope instanceof NodeWithSimpleName<?> nodeWithSimpleName) {
                Object resolved = resolvable.resolve();
                if (resolved instanceof JavaParserVariableDeclaration javaParserVariableDeclaration) {
                    if (javaParserVariableDeclaration.getVariableDeclarator() == variableDeclarator) {
                        nodeWithSimpleName.setName(newVariableName);
                    }
                } else if (resolved instanceof JavaParserFieldDeclaration javaParserFieldDeclaration) {
                    if (javaParserFieldDeclaration.getVariableDeclarator() == variableDeclarator) {
                        nodeWithSimpleName.setName(newVariableName);
                    }
                } else {
                    getLogger().debug("Unhandled resolved type for {}: {}", nodeWithSimpleName.getClass(),
                            resolved.getClass());
                }
            } else {
                getLogger().debug("Unhandled scope: {}", scope.getClass());
            }

        });
        findParameterUsage(componentInfo).forEach(parameter -> {
            if (parameter instanceof NameExpr nameExpr) {
                nameExpr.setName(newVariableName);
            } else {
                getLogger().debug("Unhandled parameter type: {}", parameter.getClass());
            }
        });

        if (variableDeclarator != null) {
            variableDeclarator.setName(newVariableName);
        }
        var createInfo = componentInfo.getCreateInfoOrThrow();
        if (createInfo.getAssignmentExpression() != null) {
            Expression target = createInfo.getAssignmentExpression().getTarget();

            if (target instanceof NameExpr nameExpr) {
                nameExpr.setName(newVariableName);
            } else {
                getLogger().debug("Unhandled assignment expression target type: {}", target.getClass());
            }
        }
    }

    /**
     * Checks if the value passed to the given function affects the variable name
     * generated for the component.
     *
     * @param functionName
     *            the function name
     * @return {@code true} if the function affects the variable name, {@code false}
     *         otherwise
     */
    public static boolean functionAffectsVariableName(String functionName) {
        return functionName.equals("setText") || functionName.equals("setLabel");
    }

    /**
     * Checks if the given component has a variable name that is auto-generated
     * based on the type/label/text of the component.
     *
     * @param componentInfo
     *            the component to check
     * @param currentText
     *            the current text of the component
     * @param currentLabel
     *            the current label of the component
     * @return {@code true} if the variable name is auto-generated, {@code false}
     *         otherwise
     */
    public static boolean isVariableNameAutoGenerated(ComponentInfo componentInfo, String currentText,
            String currentLabel) {
        String variableBaseName = generateVariableBaseName(
                componentInfo.type().getSimpleName().toLowerCase(Locale.ENGLISH), null, null, currentText,
                currentLabel);
        String currentName = getFieldOrVariableName(componentInfo);
        if (currentName == null) {
            return false;
        }
        if (!currentName.startsWith(variableBaseName)) {
            return false;
        }
        String suffix = currentName.substring(variableBaseName.length());
        return suffix.isEmpty() || isNumber(suffix);
    }

    private static boolean isNumber(String suffix) {
        try {
            return String.valueOf(Integer.parseInt(suffix, 10)).equals(suffix);
        } catch (Exception e) {
            return false;
        }
    }

    /**
     * Find all calls done to methods in the given class through a reference to the
     * given component.
     *
     * <p>
     * For instance, if you pass in Style.class, then it will find all invocations
     * like <code>
     * button.getStyle().set("color","red")</code>,
     * <code>button.getStyle().setColor("blue")</code> and also <code>
     * Style s = button.getStyle(); s.setColor("black")</code>
     *
     * @param classWithMethods
     *            the class the check for calls
     * @param componentInfo
     *            the component
     * @return a list of method calls
     */
    public static List<MethodCallExpr> findCalls(Class<?> classWithMethods, ComponentInfo componentInfo) {
        // Post order here is needed to return chained calls in the order they
        // are excuted so e.g.
        // Styles.set("color", "red").set("color", "blue") will return styles in
        // the order red, blue and not the other way around
        List<MethodCallExpr> methodCallsInComponentScope = JavaRewriterUtil.findMethodCalls(componentInfo,
                Node.TreeTraversal.POSTORDER);

        return methodCallsInComponentScope.stream().filter(testMethodInClass(classWithMethods, componentInfo)).toList();
    }

    /**
     * Looks for given method call is a member of given class for the component
     *
     * @param classWithMethods
     *            Class that where method instance is declared
     * @param componentInfo
     *            Component to look for variable names
     * @return predicate that
     */
    public static Predicate<MethodCallExpr> testMethodInClass(Class<?> classWithMethods, ComponentInfo componentInfo) {
        return methodCallExpr -> {
            ResolvedMethodDeclaration r = methodCallExpr.resolve();
            if (!r.declaringType().getQualifiedName().equals(classWithMethods.getName())) {
                return false;
            }

            // Find the target class reference (e.g. Styles), it can be a method
            // call, a local variable or a field reference
            return JavaRewriterUtil.scopeIs(methodCallExpr,
                    componentInfo.getCreateInfoOrThrow().getLocalVariableName() != null
                            ? componentInfo.getCreateInfoOrThrow().getLocalVariableName()
                            : componentInfo.getCreateInfoOrThrow().getFieldName());
        };
    }

    /**
     * Adds a function call to the given component.
     *
     * @param componentInfo
     *            the component
     * @param function
     *            the function to call
     * @param parameterExpressions
     *            the parameters to pass to the function
     */
    public static @NonNull MethodCallExpr addFunctionCall(ComponentInfo componentInfo, String function,
            List<Expression> parameterExpressions) {
        return addFunctionCall(null, componentInfo, function, parameterExpressions, null);
    }

    public static @NonNull MethodCallExpr addFunctionCall(InsertionPoint insertionPoint, ComponentInfo componentInfo,
            String function, List<Expression> parameterExpressions, ObjectCreationExpr initializer) {
        List<MethodCallExpr> functionCalls = null;
        if (insertionPoint != null) {
            functionCalls = JavaRewriterUtil.findMethodCallStatements(insertionPoint, componentInfo);
        } else {
            functionCalls = JavaRewriterUtil.findMethodCallStatements(componentInfo);
        }

        if (!functionCalls.isEmpty()) {
            MethodCallExpr added = JavaRewriterUtil.addAfterLastFunctionCallInRelevantBlockStmt(functionCalls,
                    initializer, function, parameterExpressions.toArray(new Expression[0]));
            if (added != null) {
                return added;
            }
        }

        int addAfterStatementIndex = -2;
        var createInfo = componentInfo.getCreateInfoOrThrow();
        BlockStmt scope = createInfo.getComponentCreateScope();

        if (createInfo.getLocalVariableDeclarator() != null) {
            // Local variable but no calls found, add after the declaration
            addAfterStatementIndex = JavaRewriterUtil.findBlockStatementIndex(createInfo.getLocalVariableDeclarator());
        } else if (createInfo.getFieldDeclaration() != null && createInfo.getFieldDeclarationAndAssignment() == null) {
            // Field defined but assignment happens elsewhere -> add after that
            // assignment
            addAfterStatementIndex = JavaRewriterUtil.findBlockStatementIndex(createInfo.getAssignmentExpression());
        } else if (createInfo.getFieldDeclarationAndAssignment() != null) {
            // Field defined and assigned in the same place -> add before attach
            addAfterStatementIndex = JavaRewriterUtil
                    .findBlockStatementIndex(componentInfo.getAttachCallInSameFileOrThrow().getNode()) - 1;
            scope = componentInfo.getComponentAttachScopeOrThrowIfDifferentFile().orElseThrow(
                    () -> new IllegalArgumentException("Component " + componentInfo + " has no attach information"));
        }

        if (addAfterStatementIndex >= -1) {
            NameExpr variable = new NameExpr(JavaRewriterUtil.getFieldOrVariableName(componentInfo));
            MethodCallExpr newCall = new MethodCallExpr(variable, function);
            parameterExpressions.forEach(newCall::addArgument);
            scope.addStatement(addAfterStatementIndex + 1, new ExpressionStmt(newCall));
            return newCall;
        }
        if (componentInfo.routeConstructor() != null) {
            Expression methodCallScope = JavaRewriterUtil.isNodeInCompositeClass(componentInfo.routeConstructor())
                    ? new MethodCallExpr("getContent")
                    : null;
            MethodCallExpr newCall = new MethodCallExpr(methodCallScope, function);
            parameterExpressions.forEach(newCall::addArgument);
            BlockStmt body = componentInfo.routeConstructor().getBody();
            body.addStatement(newCall);
            return newCall;
        } else if (componentInfo.isAnonymousComponent() && !componentInfo.isReturnValue()) {
            // Inline assignment, like add(new Button("foo"))
            // Extract to a local variable, add the call and replace the inline
            // add
            JavaRewriter.ExtractInlineVariableResult extractInlineVariableResult = JavaRewriterUtil
                    .extractInlineVariableToLocalVariable(componentInfo);
            if (extractInlineVariableResult != null) {

                // input.setFoo("bar")
                MethodCallExpr newCall = new MethodCallExpr(new NameExpr(extractInlineVariableResult.newVariableName()),
                        function);
                parameterExpressions.forEach(newCall::addArgument);

                extractInlineVariableResult.blockStmt().addStatement(extractInlineVariableResult.index() + 1, newCall);
                return newCall;
            }
        }
        throw new IllegalStateException("Unable to determine where to add function call");
    }

    /**
     * Gets the outermost scope for a chain of method calls.
     *
     * <p>
     * E.g. for {@code foo.bar().baz().qux()} it will return {@code foo}
     *
     * @param methodCallExpr
     *            The method call expression
     * @return The outermost scope
     */
    public static Optional<Expression> getOutermostScope(MethodCallExpr methodCallExpr) {
        Optional<Expression> scope = methodCallExpr.getScope();
        while (scope.isPresent() && scope.get().isMethodCallExpr()) {
            methodCallExpr = scope.get().asMethodCallExpr();
            scope = methodCallExpr.getScope();
        }

        return scope;
    }

    /**
     * Removes the given method call from a getStyle call chain.
     *
     * <p>
     * If the method call is the only style call, removes the whole statement.
     *
     * @param methodCallExpr
     *            the method call to remove
     */
    public static void removeFromChainedStyleCall(MethodCallExpr methodCallExpr) {
        // There are calls before in the chain if the scope is a method call
        // that is not getStyle()
        Optional<Expression> scope = methodCallExpr.getScope();
        Optional<String> methodScopeName = scope.filter(Expression::isMethodCallExpr).map(Expression::asMethodCallExpr)
                .map(MethodCallExpr::getNameAsString);
        Optional<Node> parentNode = methodCallExpr.getParentNode();
        Optional<ExpressionStmt> parentExpressionStmt = parentNode.filter(ExpressionStmt.class::isInstance)
                .map(ExpressionStmt.class::cast);

        boolean callsBefore = methodScopeName.isPresent() && !methodScopeName.get().equals("getStyle");
        boolean callsAfter = parentExpressionStmt.isEmpty();

        if (!callsBefore && !callsAfter) {
            // If this is the only style call, remove the whole statement
            parentExpressionStmt.get().remove();
            return;
        }
        if (scope.isEmpty()) {
            throw new IllegalArgumentException("Scope cannot be empty when there are calls before this in the chain");
        }
        if (callsBefore) {
            if (callsAfter) {
                // If this is in the middle of the chain, remove this call

                // Parent is the call following this in the chain
                // Scope is the call before this in the chain
                if (parentNode.isEmpty()) {
                    throw new IllegalArgumentException(
                            "Parent node is empty even when there is a call after this in the chain");
                }
                MethodCallExpr callAfterThis = (MethodCallExpr) parentNode.get();
                MethodCallExpr callBeforeThis = (MethodCallExpr) scope.get();

                callAfterThis.setScope(callBeforeThis);
            } else {
                // If this is the last call in the chain, remove the call by
                // moving scope to parent
                parentExpressionStmt.get().setExpression(scope.get());
            }
        } else {
            // no call before except getStyle(), but there are calls after the given method
            // call expression
            Optional<MethodCallExpr> nextMethodCallOptional = methodCallExpr.getParentNode()
                    .filter(MethodCallExpr.class::isInstance).map(MethodCallExpr.class::cast)
                    .filter(parentMethodCall -> parentMethodCall.getScope()
                            .filter(methodScope -> methodScope.equals(methodCallExpr)).isPresent());
            if (nextMethodCallOptional.isEmpty()) {
                throw new IllegalArgumentException("Not sure how to remove style call " + methodCallExpr
                        + " in context of " + JavaRewriterUtil.findAncestor(methodCallExpr, ExpressionStmt.class));
            }
            MethodCallExpr nextMethodCallExpr = nextMethodCallOptional.get();
            nextMethodCallExpr.setScope(scope.get());
        }
    }

    private static String toCamelCase(String input) {
        StringBuilder result = new StringBuilder();

        boolean convertNext = false;
        for (char c : input.toCharArray()) {
            if (Character.isLetterOrDigit(c)) {
                if (convertNext) {
                    result.append(Character.toUpperCase(c));
                    convertNext = false;
                } else {
                    result.append(c);
                }
            } else {
                convertNext = true;
            }
        }

        return result.toString();
    }

    public static void moveAboveMethodsAndConstructors(BodyDeclaration<?> toMove,
            ClassOrInterfaceDeclaration classOrInterface) {
        int fieldIndex = classOrInterface.getMembers().indexOf(toMove);
        int newIndex = -1;
        for (int i = 0; i < fieldIndex; i++) {
            BodyDeclaration<?> member = classOrInterface.getMember(i);
            if (member.isConstructorDeclaration() || member.isMethodDeclaration()) {
                newIndex = i;
                break;
            }
        }
        if (newIndex != -1) {
            classOrInterface.getMembers().add(newIndex, toMove);
        }

    }

    /**
     * Creates a Type that can be used in generics to end up with `<>`.
     *
     * @return an empty type
     */
    public static Type createEmptyType() {
        // This hack is to render the expression with empty type arguments in the
        // instantiation as Java Parser interprets an empty NodeList as not
        // parametrized. The token JavaRewriter.DELETE_THIS_MARKER is used to replace
        // the empty NodeList with an empty type arguments when rendering the result.
        ClassOrInterfaceType toDeleteType = new ClassOrInterfaceType();
        toDeleteType.setName(JavaRewriter.DELETE_THIS_MARKER);
        return toDeleteType;
    }

    /**
     * Checks if the given component is created inside the constructor of the class
     * (as opposed to being created in another method).
     *
     * @param componentInfo
     *            the component
     * @return {@code true} if the component is created in the constructor,
     *         {@code false} otherwise
     */
    public static boolean isObjectCreationExprInConstructor(ComponentInfo componentInfo) {
        if (componentInfo.getCreateInfoOrThrow().getObjectCreationExpr() == null) {
            return false;
        }

        return findAncestor(componentInfo.getCreateInfoOrThrow().getObjectCreationExpr(),
                ConstructorDeclaration.class) != null;

    }

    /**
     * Parses the given class and finds bean properties in the order they are
     * defined.
     *
     * @param itemType
     *            the item class name
     * @return a list of property names in the order they are defined in the source
     * @throws IOException
     *             if the class file cannot be read
     */
    public static List<UIServiceCreator.FieldInfo> getPropertiesInSourceOrder(String itemType) throws IOException {
        File fileForClass = ProjectFileManager.get().getFileForClass(itemType);
        JavaSource javaSource = new JavaFileSourceProvider().getJavaSource(fileForClass);
        boolean isInternalClass = isInnerClass(itemType, javaSource.getCompilationUnit());
        List<FieldDeclaration> fields = null;

        if (isInternalClass) {
            fields = getInnerClassFields(itemType, javaSource.getCompilationUnit());
        } else {
            fields = getOuterClassFields(javaSource.getCompilationUnit());
        }

        List<UIServiceCreator.FieldInfo> properties = new ArrayList<>();

        for (FieldDeclaration field : fields) {
            if (field.isStatic()) {
                continue;
            }
            for (VariableDeclarator declarator : field.getVariables()) {
                String varName = declarator.getNameAsString();
                ResolvedType type = declarator.getType().resolve();
                String typeName;
                if (type.isReferenceType()) {
                    typeName = type.asReferenceType().getQualifiedName();
                } else {
                    typeName = declarator.getTypeAsString();
                }

                properties.add(new UIServiceCreator.FieldInfo(varName, typeName));

            }
        }

        return properties;
    }

    /**
     * Returns all FieldDeclarations of an inner class, given its fully qualified
     * name (FQN) and the compilation unit of the outer class.
     * <p>
     * Example FQN: "test.vaadin.copilot.flow.testviews.domain.Product.Priority"
     */
    public static List<FieldDeclaration> getInnerClassFields(String fqn, CompilationUnit cu) {
        String[] parts = fqn.split("\\.");
        String innerSimpleName = parts[parts.length - 1]; // "Priority"
        String outerSimpleName = parts[parts.length - 2]; // "Product"

        // 1. Find top-level type (Product)
        Optional<TypeDeclaration<?>> outerOpt = cu.getTypes().stream()
                .filter(t -> t.getNameAsString().equals(outerSimpleName)).findFirst();

        if (outerOpt.isEmpty()) {
            return List.of();
        }

        TypeDeclaration<?> outer = outerOpt.get();

        // 2. Find the inner class/enum/interface
        Optional<? extends TypeDeclaration<?>> innerOpt = outer.getMembers().stream()
                .filter(m -> m instanceof TypeDeclaration).map(m -> (TypeDeclaration<?>) m)
                .filter(t -> t.getNameAsString().equals(innerSimpleName)).findFirst();

        if (innerOpt.isEmpty()) {
            return List.of();
        }

        TypeDeclaration<?> inner = innerOpt.get();

        // 3. Return only FieldDeclaration inside the inner class
        return inner.getMembers().stream().filter(m -> m instanceof FieldDeclaration).map(m -> (FieldDeclaration) m)
                .collect(Collectors.toList());
    }

    public static List<FieldDeclaration> getOuterClassFields(CompilationUnit cu) {

        if (cu.getTypes().isEmpty()) {
            // No class declared in this CU
            return List.of();
        }

        TypeDeclaration<?> outer = cu.getTypes().get(0); // Safe now

        return outer.getMembers().stream().filter(m -> m instanceof FieldDeclaration).map(m -> (FieldDeclaration) m)
                .collect(Collectors.toList());
    }

    /**
     * Determines whether a fully qualified name (FQN) refers to an inner class.
     *
     * @param fqn
     *            fully qualified name, e.g. "my.pkg.Outer.Inner"
     * @param cu
     *            the parsed CompilationUnit of the source file that may contain the
     *            class
     * @return true if the FQN points to an inner class inside the top-level class
     */
    public static boolean isInnerClass(String fqn, CompilationUnit cu) {
        String[] parts = fqn.split("\\.");

        // Last part = simple name of the queried class
        String targetSimpleName = parts[parts.length - 1];

        // Top-level type name
        String topLevelSimpleName = parts[parts.length - 2];

        // Find the top-level class in the compilation unit
        TypeDeclaration<?> topLevel = cu.getTypes().stream().filter(t -> t.getNameAsString().equals(topLevelSimpleName))
                .findFirst().orElse(null);

        if (topLevel == null) {
            return false;
        }

        // Search inside its members for a matching inner class
        return topLevel.getMembers().stream().filter(m -> m instanceof TypeDeclaration).map(m -> (TypeDeclaration<?>) m)
                .anyMatch(inner -> inner.getNameAsString().equals(targetSimpleName));
    }

    static boolean isAttachCall(MethodCallExpr methodCallExpr) {
        return methodCallExpr.getNameAsString().equals("add");
    }

    /**
     * Gets the name of the symbol that could not be resolved.
     *
     * @param e
     *            the exception from java parser
     * @return the name of the symbol
     */
    public static String getSymbolNameFromJavaParserException(UnsolvedSymbolException e) {
        String name = e.getName();

        // These feel like java parser bugs
        String[] prefixes = new String[] { "Solving ",
                "We are unable to find the value declaration corresponding to " };
        for (String prefix : prefixes) {
            if (name.startsWith(prefix)) {
                return name.substring(prefix.length());
            }
        }

        return name;

    }

    public static boolean isComponentOrComponentArray(Class<?> type) {
        if (Component.class.isAssignableFrom(type)) {
            return true;
        }
        if (type.isArray()) {
            return Component.class.isAssignableFrom(type.getComponentType());
        }
        return false;
    }

    /**
     * Checks if given expression is descendant of loop statements. {@link ForStmt},
     * {@link WhileStmt}, {@link DoStmt}
     *
     * @param expression
     *            expression to check
     * @return true if loop statements found in ancestor, false otherwise.
     */
    public static boolean checkExpressionInLoop(Expression expression) {
        if (expression == null) {
            return false;
        }
        Node ancestor = findAncestor(expression, node -> node instanceof ForStmt || node instanceof WhileStmt
                || node instanceof DoStmt || node instanceof ForEachStmt);
        if (ancestor != null) {
            return true;
        }

        ancestor = findAncestor(expression, LambdaExpr.class::isInstance);
        if (ancestor == null) {
            return false;
        }
        Optional<Node> parentNode = ancestor.getParentNode();
        if (parentNode.isPresent() && parentNode.get() instanceof MethodCallExpr methodCallExpr) {
            int index = -1;
            for (int i = 0; i < methodCallExpr.getChildNodes().size(); i++) {
                if (methodCallExpr.getChildNodes().get(i).equals(ancestor)) {
                    index = i - 1;
                }
            }
            if (index == -1) {
                return false;
            }
            Node node = methodCallExpr.getChildNodes().get(index);
            if (node.toString().startsWith("map") || node.toString().startsWith("for")
                    || node.toString().equals("flatMap")) {
                return true;
            }
        }
        return false;
    }

    /**
     * Determines whether all invocations of the method containing the given
     * expression are made from within a loop construct (e.g., {@code for},
     * {@code while}, or {@code do-while}) via
     * {@link #checkExpressionInLoop(Expression)}
     *
     * @param expression
     *            the expression located within a method whose calls are to be
     *            analyzed
     * @return {@code true} if all method calls to the enclosing method are found
     *         inside loops; {@code false} otherwise or if any step in the process
     *         fails
     */
    public static boolean checkMethodHasExpressionCalledInLoop(Expression expression) {
        if (expression == null) {
            return false;
        }
        MethodDeclaration ancestor = JavaRewriterUtil.findAncestor(expression, MethodDeclaration.class);
        if (ancestor == null) {
            return false;
        }
        List<MethodCallExpr> callerInClass = findCallerInClass(ancestor);
        if (callerInClass.isEmpty()) {
            return false;
        }

        return callerInClass.stream().allMatch(JavaRewriterUtil::checkExpressionInLoop);
    }

    /**
     * Iterates arguments to check any of the argument equals to argExpression.
     *
     * @param nodeWithArgs
     *            node with arguments. e.g. {@link MethodCallExpr},
     *            {@link com.github.javaparser.ast.stmt.ExplicitConstructorInvocationStmt}
     * @param argExpression
     *            expression to check
     * @return argument index or -1 if not found.
     */
    public static int findArgumentIndex(NodeWithArguments<?> nodeWithArgs, Expression argExpression) {
        for (int i = 0; i < nodeWithArgs.getArguments().size(); i++) {
            Expression expression = nodeWithArgs.getArguments().get(i);
            if (expression.equals(argExpression)) {
                return i;
            }
        }
        return -1;
    }

    private static List<MethodCallExpr> findCallerInClass(MethodDeclaration methodDeclaration) {
        List<MethodCallExpr> methodCallExprs = new ArrayList<>();
        ResolvedMethodDeclaration resolve;
        try {
            resolve = methodDeclaration.resolve();
        } catch (Exception e) {
            getLogger().trace("Could not resolve method declaration. Method={}", methodDeclaration.getNameAsString(),
                    e);
            return new ArrayList<>();
        }
        ClassOrInterfaceDeclaration classOrInterfaceDeclaration = JavaRewriterUtil
                .findAncestorOrThrow(methodDeclaration, ClassOrInterfaceDeclaration.class);
        classOrInterfaceDeclaration.findAll(MethodCallExpr.class).forEach(call -> {
            try {
                ResolvedMethodDeclaration calledMethodDeclaration = call.resolve();
                if (typesEqual(calledMethodDeclaration, resolve)) {
                    methodCallExprs.add(call);
                }
            } catch (Exception e) {
                getLogger().trace("Failed to resolve methodCall {}", call.getNameAsString(), e);
            }
        });
        return methodCallExprs;
    }

    public static String getVariableNameForAdd(ComponentInfo referenceComponent) {
        if (referenceComponent.routeConstructor() != null) {
            return "this";
        }
        return JavaRewriterUtil.getFieldOrVariableName(referenceComponent);
    }

    public static ComponentInfo createComponentInfoForTemplateComponent(JavaSource javaSource,
            ObjectCreationExpr objectCreationExpr, AttachExpression attachExpression) {
        ComponentCreateInfo componentCreateInfo = ComponentInfoFinder.createComponentCreateInfo(javaSource,
                objectCreationExpr);
        BlockStmt attachScope = null;
        if (attachExpression != null) {
            attachScope = JavaRewriterUtil.findBlock(attachExpression.getNode());
        }
        ComponentAttachInfo componentAttachInfo = ComponentInfoFinder.createAttachInfo(javaSource, attachExpression,
                attachScope, componentCreateInfo);
        return new ComponentInfo(Component.class, null, false, false, false, false, true, Optional.empty(),
                Optional.of(componentCreateInfo), Optional.of(componentAttachInfo));
    }

    /**
     * Find return statements referring to the component.
     *
     * @param componentInfo
     *            the component to find return statements for
     * @return a list of return statements that refer to the component
     */
    public static List<ReturnStmt> findReturnStatements(ComponentInfo componentInfo) {
        ComponentCreateInfo createInfo = componentInfo.getCreateInfoOrThrow();
        return findNameReferences(createInfo.getLocalVariableName(), createInfo.getComponentCreateScope())
                .map(Node::getParentNode).flatMap(Optional::stream).filter(ReturnStmt.class::isInstance)
                .map(ReturnStmt.class::cast).toList();

    }

    public static Optional<MethodDeclaration> getMethodIfEmpty(BlockStmt createScope) {
        if (createScope != null && createScope.getStatements().isEmpty()) {
            Optional<Node> parent = createScope.getParentNode();
            if (parent.isPresent() && parent.get() instanceof MethodDeclaration methodDeclaration) {
                return Optional.of(methodDeclaration);
            }
        }
        return Optional.empty();
    }

    /**
     * Gets the {@literal value} attribute of the given annotation or null if no
     * value is found.
     *
     * @param annotation
     *            the annotation to check
     * @return the value of the {@literal value} attribute
     */
    public static String getAnnotationValue(AnnotationExpr annotation) {
        if (annotation instanceof NormalAnnotationExpr normalAnnotationExpr) {
            Optional<MemberValuePair> valueAttribute = normalAnnotationExpr.getPairs().stream()
                    .filter(pair -> pair.getNameAsString().equals("value")).findFirst();
            if (valueAttribute.isPresent()) {
                Expression value = valueAttribute.get().getValue();
                if (value.isStringLiteralExpr()) {
                    return value.asStringLiteralExpr().getValue();
                }
            }
        } else if (annotation instanceof SingleMemberAnnotationExpr singleMemberAnnotationExpr) {
            Expression value = singleMemberAnnotationExpr.getMemberValue();
            if (value.isStringLiteralExpr()) {
                return value.asStringLiteralExpr().getValue();
            }
        } else if (annotation instanceof MarkerAnnotationExpr) {
            return "";
        }
        return null;
    }

    public static FieldDeclaration findIdAnnotatedField(String id, ClassOrInterfaceDeclaration classType) {
        // Look for fields with @Id annotation matching the id property
        for (FieldDeclaration field : classType.getFields()) {
            for (AnnotationExpr annotation : field.getAnnotations()) {
                if (annotation.getNameAsString().equals("Id")) {
                    String idValue = getAnnotationValue(annotation);
                    if (id.equals(idValue)) {
                        return field;
                    }
                }
            }
        }
        return null;
    }

    /**
     * Gets the qualified class name of ResolvedDeclaration when the given
     * declaration is an instance of {@link ResolvedValueDeclaration} For generic
     * classes such as <code>Binder</code>, it does not return the generic type
     * instead it would return <code>com.vaadin.flow.data.binder.Binder</code>
     * 
     * @param resolvedDeclaration
     *            Resolved declaration from {@link Resolvable#resolve()}
     * @return the qualified class name of the given declaration.
     */
    public static Optional<String> getQualifiedClassName(ResolvedDeclaration resolvedDeclaration) {
        if (!(resolvedDeclaration instanceof ResolvedValueDeclaration resolvedValueDeclaration)) {
            return Optional.empty();
        }
        ResolvedType type = resolvedValueDeclaration.getType();
        if (type.isReferenceType()) {
            ResolvedReferenceType referenceType = type.asReferenceType();
            return Optional.of(referenceType.getQualifiedName());
        }
        return Optional.of(type.describe());
    }

    /**
     * Removes the given expression from the condition
     *
     * @param node
     *            the node to remove
     * @return {@code true} if node is in the condition and there are multiple
     *         conditions, {@code false} otherwise.
     */
    public static boolean removeFromCondition(Node node) {
        if (node.getParentNode().isEmpty()) {
            return false;
        }
        Node parentNode = node.getParentNode().get();

        if (!(parentNode instanceof BinaryExpr binaryExpr)) {
            return false;
        }
        Statement statement = JavaRewriterUtil.findAncestor(node, Statement.class);
        if (statement instanceof NodeWithCondition<?> nodeWithCondition) {
            Expression condition = nodeWithCondition.getCondition();
            if (condition.equals(binaryExpr)) {
                // when condition equals to binary expression, deleting such condition should
                // remove the whole statement
                return false;
            }
            if (condition instanceof BinaryExpr conditionExpr) {
                if (conditionExpr.getLeft().equals(binaryExpr)) {
                    nodeWithCondition.setCondition(conditionExpr.getRight());
                    return true;
                } else if (conditionExpr.getRight().equals(binaryExpr)) {
                    nodeWithCondition.setCondition(conditionExpr.getLeft());
                    return true;
                } else {
                    throw new IllegalArgumentException("Unable to handle condition expression " + conditionExpr);
                }
            }
            return binaryExpr.remove();
        }
        return false;
    }

    /**
     * Removes arguments from <code>getThemeList().add()</code> or
     * <code>getThemeList().addAll()</code>. If there is no arguments left after
     * removing, removes the method.
     *
     * @param methodCallStatements
     *            Method calls of the given node
     * @param startsWith
     *            string literal args that starts with the given param to be removed
     */
    public static void removeThemeArgStartsWith(List<MethodCallExpr> methodCallStatements, String startsWith) {
        List<MethodCallExpr> themeAddCalls = getThemeAddCalls(methodCallStatements);
        for (MethodCallExpr addCall : themeAddCalls) {
            List<Expression> argsToRemove = addCall.getArguments().stream()
                    .filter(f -> f.isStringLiteralExpr() && f.asStringLiteralExpr().asString().startsWith(startsWith))
                    .collect(Collectors.toList());
            JavaRewriterUtil.removeArgumentCalls(addCall, argsToRemove, true);
        }
    }

    public static void removeSetThemeArgs(List<MethodCallExpr> methodCallExprList, String argName) {
        List<MethodCallExpr> list = methodCallExprList.stream()
                .filter(methodCallExpr -> methodCallExpr.getScope().isPresent())
                .filter(methodCallExpr -> methodCallExpr.getScope().get().toString().contains("getThemeList"))
                .filter(methodCallExpr -> methodCallExpr.getNameAsString().equals("set")).toList();
        for (MethodCallExpr methodCallExpr : list) {
            if (!methodCallExpr.getArguments().isEmpty()) {
                Expression argument = methodCallExpr.getArgument(0);
                if (argument.isStringLiteralExpr() && argument.asStringLiteralExpr().asString().equals(argName)) {
                    JavaRewriterUtil.removeStatement(methodCallExpr);
                }
            }
        }
    }

    private static List<MethodCallExpr> getThemeAddCalls(List<MethodCallExpr> methodCallStatements) {
        return methodCallStatements.stream().filter(methodCallExpr -> methodCallExpr.getScope().isPresent())
                .filter(methodCallExpr -> methodCallExpr.getScope().get().toString().contains("getThemeList"))
                .filter(methodCallExpr -> methodCallExpr.getNameAsString().equals("add")
                        || methodCallExpr.getNameAsString().equals("addAll"))
                .collect(Collectors.toList());
    }

    /**
     * Searches the given CompilationUnit for the first line containing the
     * specified text.
     * <p>
     * This method performs a simple string-based search on the CompilationUnit's
     * source representation. It does not parse the AST or consider Java structure;
     * it purely looks for the substring within each line of the generated source.
     * </p>
     *
     * @param compilationUnit
     *            the CompilationUnit whose source text will be scanned
     * @param stringToSearch
     *            the substring to search for within the source
     * @return an Optional containing the zero-based line number if found, otherwise
     *         empty
     */
    public static Optional<Integer> findLineContaining(CompilationUnit compilationUnit, String stringToSearch) {
        String compilationUnitString = compilationUnit.toString();
        String[] lines = compilationUnitString.split(compilationUnit.getLineEndingStyle().asRawString());
        for (int i = 0; i < lines.length; i++) {
            String line = lines[i];
            if (line.contains(stringToSearch)) {
                return Optional.of(i);
            }
        }
        return Optional.empty();
    }

    /**
     * Finds all assignments to the given field declaration within its containing
     * class.
     *
     * @param fieldDeclaration
     *            the field declaration to find assignments for
     * @return a list of assignment expressions assigning to the field
     */
    public static List<AssignExpr> findFieldAssignments(FieldDeclaration fieldDeclaration) {

        // resolve the actual field once
        VariableDeclarator vd = fieldDeclaration.getVariables().get(0);
        ResolvedFieldDeclaration targetField = vd.resolve().asField();

        ClassOrInterfaceDeclaration clazz = JavaRewriterUtil.findAncestorOrThrow(fieldDeclaration,
                ClassOrInterfaceDeclaration.class);

        List<AssignExpr> assignExprs = new ArrayList<>();

        for (AssignExpr a : clazz.findAll(AssignExpr.class)) {
            if (isExpressionRelatedToField(a.getTarget(), fieldDeclaration)) {
                assignExprs.add(a);
            }
        }
        return assignExprs;
    }

    /**
     * Checks if the expression is related to a field or to a local variable
     * 
     * @param target
     *            expression to evaluate
     * @param fieldDeclaration
     *            field to evaluate
     * @return True if the expression refers to the field, False otherwise
     */
    public static boolean isExpressionRelatedToField(Expression target, FieldDeclaration fieldDeclaration) {
        // resolve the actual field once
        VariableDeclarator vd = fieldDeclaration.getVariables().get(0);
        ResolvedFieldDeclaration targetField = vd.resolve().asField();

        // try to resolve the target of the assignment as a field
        try {
            AtomicReference<ResolvedDeclaration> resolved = new AtomicReference<>();

            if (target.isNameExpr()) {
                resolved.set(target.asNameExpr().resolve());
            } else if (target.isFieldAccessExpr()) {
                resolved.set(target.asFieldAccessExpr().resolve());
            } else if (target.isMethodCallExpr()) {
                target.asMethodCallExpr().getScope().ifPresent(scope -> {
                    if (scope.isNameExpr()) {
                        resolved.set(scope.asNameExpr().resolve()); // field/param/local?
                    } else if (scope.isFieldAccessExpr()) {
                        resolved.set(scope.asFieldAccessExpr().resolve());
                    }
                });
            } else {
                return false;
            }

            if (resolved.get().isField()) {
                ResolvedFieldDeclaration f = resolved.get().asField();

                // compare by declaring type + field name (stable)
                if (f.getName().equals(targetField.getName()) && f.declaringType().getQualifiedName()
                        .equals(targetField.declaringType().getQualifiedName())) {
                    return true;
                }
            }
        } catch (Exception ignored) {
            // resolution failed; skip
        }
        return false;
    }
}
