package com.vaadin.copilot;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Stream;

import com.vaadin.base.devserver.DevToolsInterface;
import com.vaadin.copilot.javarewriter.ComponentInfo;
import com.vaadin.copilot.javarewriter.ComponentInfoFinder;
import com.vaadin.copilot.javarewriter.ComponentTypeAndSourceLocation;
import com.vaadin.copilot.javarewriter.JavaComponent;
import com.vaadin.copilot.javarewriter.JavaFileSourceProvider;
import com.vaadin.copilot.javarewriter.JavaRewriter;
import com.vaadin.copilot.javarewriter.JavaSource;
import com.vaadin.flow.component.dependency.JsModule;
import com.vaadin.flow.internal.JacksonUtils;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import tools.jackson.databind.JsonNode;
import tools.jackson.databind.node.ArrayNode;
import tools.jackson.databind.node.ObjectNode;

public class LitTemplateHandler extends CopilotCommand {
    public static final String LIT_TEMPLATE_FILENAME = "litTemplateFilename";

    private final ComponentSourceFinder sourceFinder;
    private final ProjectFileManager projectFileManager;

    public LitTemplateHandler() {
        sourceFinder = new ComponentSourceFinder(getVaadinSession());
        projectFileManager = ProjectFileManager.get();
    }

    @Override
    public boolean handleMessage(String command, JsonNode data, DevToolsInterface devToolsInterface) {
        ObjectNode responseData = JacksonUtils.createObjectNode();
        responseData.put("reqId", data.get("reqId").asString());
        String errorMsg = "Unknown error handling Lit template command";
        try {
            if (command.equals("get-lit-template-name")) {
                errorMsg = "Unable to get Lit template name";
                handleGetLitTemplateName(data, responseData);
                devToolsInterface.send(command + "-resp", responseData);
                return true;
            } else if (command.equals("import-lit-template")) {
                errorMsg = "Unable to convert Lit template to Java";
                handleImportLitTemplate(data);
                devToolsInterface.send(command + "-resp", responseData);
                return true;
            }
        } catch (Exception e) {
            ErrorHandler.sendErrorResponse(devToolsInterface, command, responseData,
                    String.format("%s. Exception message: %s", errorMsg, e.getMessage()), e);
        }
        return false;
    }

    private void handleGetLitTemplateName(JsonNode data, ObjectNode responseData) throws IOException {
        ComponentTypeAndSourceLocation litTemplate = sourceFinder.findTypeAndSourceLocation(data.get("litTemplate"));
        var jsModule = JavaReflectionUtil.findClassWithJsModule(litTemplate.component().getClass())
                .getDeclaredAnnotation(JsModule.class);
        var litTemplateFile = new File(projectFileManager.getFrontendFolder(),
                jsModule.value().replace("/", File.separator));
        responseData.put(LIT_TEMPLATE_FILENAME, litTemplateFile.getCanonicalPath());
    }

    private void handleImportLitTemplate(JsonNode data) throws IOException {
        JsonNode warnings = data.get("warnings");
        if (warnings.get("hasTemplateExpressions").asBoolean(false)) {
            getLogger().warn("Lit template conversion: Template expressions will be lost during conversion.");
        }
        ArrayNode eventBindings = warnings.withArray("eventBindings");
        if (!eventBindings.isEmpty()) {
            List<String> eventBindingNames = new ArrayList<>();
            eventBindings.forEach(name -> eventBindingNames.add(name.asString()));
            getLogger().warn("Lit template conversion: Event bindings will be lost during conversion: {}",
                    eventBindingNames);
        }
        ArrayNode stateProperties = warnings.withArray("statePropertyNames");
        if (!stateProperties.isEmpty()) {
            List<String> statePropertyNames = new ArrayList<>();
            stateProperties.forEach(name -> statePropertyNames.add(name.asString()));
            getLogger().warn("Lit template conversion: State properties will be lost during conversion: {}",
                    statePropertyNames);
        }

        // Get styles from data
        ArrayNode stylesNode = data.withArray("styles");
        List<String> styles = new ArrayList<>();
        stylesNode.forEach(style -> styles.add(style.asString()));

        String litTemplateFileName = data.get(LIT_TEMPLATE_FILENAME).asString();
        JavaFileSourceProvider javaFileSourceProvider = new JavaFileSourceProvider(true);
        ComponentTypeAndSourceLocation litTemplate = sourceFinder.findTypeAndSourceLocation(data.get("litTemplate"));
        ComponentInfoFinder componentInfoFinder = new ComponentInfoFinder(javaFileSourceProvider, litTemplate,
                sourceFinder::getSiblingsTypeAndSourceLocations);
        ComponentInfo litTemplateInfo = componentInfoFinder.find();
        var customComponentInfo = litTemplateInfo.customComponentInfo()
                .orElseThrow(() -> new IllegalArgumentException("The given lit template was not found in the project"));

        Class<?> classWithJsModule = JavaReflectionUtil.findClassWithJsModule(customComponentInfo.componentClass());

        // Transform styles for importing from css file
        String cssClassName = LitTemplateCssUtil.getConvertedCssClassName(classWithJsModule);
        List<String> transformedStyles = styles.stream()
                .map(style -> LitTemplateCssUtil.transformHostToCssClass(style, cssClassName)).toList();

        JavaSource litTemplateSource;
        try {
            File litTemplateClassFile = projectFileManager.getFileForClass(classWithJsModule);
            litTemplateSource = javaFileSourceProvider.getJavaSource(litTemplateClassFile);
        } catch (Exception e) {
            throw new IOException("Unable to read lit template class source. "
                    + "The class might be in an external dependency, making conversion not possible.", e);
        }
        JavaRewriter javaRewriter = new JavaRewriter();
        boolean extendsLitTemplate = javaRewriter.importLitTemplate(litTemplateSource,
                JavaComponent.componentsFromJson(data.withArray("componentDefinitions")),
                transformedStyles.isEmpty() ? null : cssClassName);
        String result = litTemplateSource.getResult();
        // Apply extends change via string replacement to avoid AST modification and
        // preserve formatting.
        if (extendsLitTemplate) {
            result = result.replace("extends LitTemplate", "extends Div");
        }
        ProjectFileManager.get().writeFile(litTemplateSource.getFile(), "Convert LitTemplate to Java", result);

        // Write CSS to a dedicated file
        if (!transformedStyles.isEmpty()) {
            String cssFileName = cssClassName + "-converted.css";
            File cssFile = new File(projectFileManager.getFrontendFolder(), cssFileName);
            String cssContent = String.join("\n\n", transformedStyles);
            ProjectFileManager.get().writeFile(cssFile, "Convert LitTemplate styles", cssContent);
        }

        var litTemplateFile = new File(litTemplateFileName);
        var litTemplateBakFile = new File(litTemplateFileName + ".bak");
        ProjectFileManager.get().moveFile(litTemplateFile, litTemplateBakFile);

        commentOutTemplateReferences(litTemplateFile, LitTemplateCssUtil.getTagName(classWithJsModule));
    }

    /**
     * Finds lit templates that reference the converted lit template and comments
     * out the references and import.
     *
     * @param convertedTemplateFile
     *            template file that was converted
     * @param tagName
     *            tag name of the converted template
     */
    void commentOutTemplateReferences(File convertedTemplateFile, String tagName) {
        if (StringUtils.isEmpty(tagName)) {
            getLogger().warn("Unable to find tag name for converted lit template. References are not checked.");
            return;
        }

        File frontendFolder = projectFileManager.getFrontendFolder();
        String templateFileName = convertedTemplateFile.getName();
        List<String> affectedFiles = new ArrayList<>();

        try (Stream<Path> paths = Files.walk(frontendFolder.toPath())) {
            paths.filter(Files::isRegularFile).filter(path -> {
                String name = path.getFileName().toString().toLowerCase();
                return name.endsWith(".js") || name.endsWith(".ts");
            }).filter(path -> !path.equals(convertedTemplateFile.toPath())).forEach(path -> {
                try {
                    if (commentOutReferencesInFile(path.toFile(), templateFileName, tagName)) {
                        affectedFiles.add(projectFileManager.getProjectRelativeName(path.toFile()));
                    }
                } catch (IOException e) {
                    getLogger().warn("Failed to process file {} for lit template references: {}", path, e.getMessage());
                }
            });
        } catch (IOException e) {
            getLogger().warn("Failed to search frontend folder for lit template references: {}", e.getMessage());
        }

        if (!affectedFiles.isEmpty()) {
            getLogger().warn("Converted lit template references were commented out from the following files: {}",
                    affectedFiles);
        }
    }

    /**
     * Comments out import statements and tag usages in a single file.
     *
     * @param file
     *            the file to process
     * @param templateFileName
     *            the filename of the converted template
     * @param tagName
     *            the tag name of the converted template
     * @return true if the file was modified, false otherwise
     */
    boolean commentOutReferencesInFile(File file, String templateFileName, String tagName) throws IOException {
        String originalContent = projectFileManager.readFile(file);
        String content = originalContent;

        // Comment out import statements for the template file
        String importPattern = "(import\\s+(?:[^;]*?from\\s+)?['\"][^'\"]*/"
                + Pattern.quote(templateFileName.replace(".js", "")) + "(?:\\.js|\\.ts)?['\"]\\s*;?)";
        content = applyPattern(content, importPattern, false, false);

        // Comment out self-closing tags
        String selfClosingPattern = "(<" + Pattern.quote(tagName) + "(?:\\s[^>]*)?/>)";
        content = applyPattern(content, selfClosingPattern, false, true);

        // Comment out full block tags
        String fullElementPattern = "(<" + Pattern.quote(tagName) + "(?:\\s[^>]*)?>.*?</" + Pattern.quote(tagName)
                + ">)";
        content = applyPattern(content, fullElementPattern, true, true);

        if (!originalContent.equals(content)) {
            projectFileManager.writeFile(file, "Comment out references to converted LitTemplate", content);
            return true;
        }
        return false;
    }

    private String applyPattern(String content, String regex, boolean dotAll, boolean html) {
        Pattern pattern = dotAll ? Pattern.compile(regex, Pattern.DOTALL) : Pattern.compile(regex);
        Matcher matcher = pattern.matcher(content);

        String commentStart = html ? "<!-- " : "/* ";
        String commentEnd = html ? " -->" : " */";

        StringBuilder sb = new StringBuilder();
        while (matcher.find()) {
            String match = matcher.group(1);
            matcher.appendReplacement(sb,
                    commentStart + "Lit template converted to Java: " + Matcher.quoteReplacement(match) + commentEnd);
        }
        matcher.appendTail(sb);
        return sb.toString();
    }

    Logger getLogger() {
        return LoggerFactory.getLogger(getClass());
    }
}
